C
C
C     This code is distributed under the terms and conditions of the
C     CCP4 licence agreement as `Part 2' (Annex 2) software.
C     A copy of the CCP4 licence can be obtained by writing to the
C     CCP4 Secretary, Daresbury Laboratory, Warrington WA4 4AD, UK.
C
C
C------------------------------------------------------------------
C
C----Operation on vectors
C
      FUNCTION DOT_R(N,NMAX,VEC1,VEC2)
C
C-----dot product of vectors
      REAL    VEC1(*),VEC2(*)
      INTEGER N,NMAX,I
C
      IF(N.LE.0.OR.N.GT.NMAX) CALL ERRWRT(1,'Size mismatch in DOT_R')
      DOT_R = 0.0
      DO    I=1,N
        DOT_R = DOT_R + VEC1(I)*VEC2(I)
      ENDDO
      RETURN
      END

      FUNCTION DOT1_R(N,NMAX,VEC1,VEC2)
C
C-----dot product of vectors. Second version. Intermediate
C-----summation in double precision
      REAL             VEC1(*),VEC2(*)
      DOUBLE PRECISION TEMP
      INTEGER          N,NMAX,I
C
      IF(N.LE.0.OR.N.GT.NMAX) CALL ERRWRT(1,'Size mismatch in DOT_R')
      TEMP = 0.0D0
      DO    I=1,N
        TEMP = TEMP + VEC1(I)*VEC2(I)
      ENDDO
      DOT1_R = REAL(TEMP)
      RETURN
      END
C
      SUBROUTINE CROSS_R(A,B,C)
      IMPLICIT NONE
C     =======================
C
C     compute vector product A = B x C
C     .. Array Arguments ..
      REAL   A(3),B(3),C(3)
C
C_END_CROSS
C     ..
      A(1) = B(2)*C(3) - C(2)*B(3)
      A(2) = B(3)*C(1) - C(3)*B(1)
      A(3) = B(1)*C(2) - C(1)*B(2)

      RETURN
      END
C
      SUBROUTINE CROSS_WITH_DERIVS(A,DADB,DADC,B,C)
      IMPLICIT NONE
C     =======================
C
C     compute vector product A = B x C and derivative of elements
C     of A wrt elements of B and C
C
C     .. Array Arguments ..
      REAL             A(3),B(3),C(3),DADB(3,3),DADC(3,3)
C
C_END_CROSS
C     ..
      A(1) = B(2)*C(3) - C(2)*B(3)
      A(2) = B(3)*C(1) - C(3)*B(1)
      A(3) = B(1)*C(2) - C(1)*B(2)

      DADB(1,1) =  0.0
      DADB(1,2) =  C(3)
      DADB(1,3) = -C(2)
      DADB(2,1) = -C(3)
      DADB(2,2) =  0.0
      DADB(2,3) =  C(1)
      DADB(3,1) =  C(2)
      DADB(3,2) = -C(1)
      DADB(3,3) =  0.0

      DADC(1,1) =  0.0
      DADC(1,2) = -B(3)
      DADC(1,3) =  B(2)
      DADC(2,1) =  B(3)
      DADC(2,2) =  0.0
      DADC(2,3) = -B(1)
      DADC(3,1) = -B(2)
      DADC(3,2) =  B(1)
      DADC(3,3) =  0.0

      END
C
      REAL FUNCTION FIND_CHIR_SIGN(XYZ1,XYZ2,XYZ3,XYZ4)
      IMPLICIT NONE
C
C---Find sign of chiral center
      REAL XYZ1(3),XYZ2(3),XYZ3(3),XYZ4(3)
      REAL  A(3,3)
      REAL DET3
      EXTERNAL DET3
C
      A(1,1) = XYZ2(1) - XYZ1(1)
      A(2,1) = XYZ2(2) - XYZ1(2)
      A(3,1) = XYZ2(3) - XYZ1(3)

      A(1,2) = XYZ3(1) - XYZ1(1)
      A(2,2) = XYZ3(2) - XYZ1(2)
      A(3,2) = XYZ3(3) - XYZ1(3)

      A(1,3) = XYZ4(1) - XYZ1(1)
      A(2,3) = XYZ4(2) - XYZ1(2)
      A(3,3) = XYZ4(3) - XYZ1(3)

      FIND_CHIR_SIGN = DET3(A)

      RETURN
      END
C
      REAL FUNCTION DET3(A)
      IMPLICIT NONE
C
C---Determinant of 3x3 matrix
      REAL A(9)
      DET3 = A(1)*(A(5)*A(9)-A(8)*A(6))
     .     - A(4)*(A(2)*A(9)-A(8)*A(3))
     .     + A(7)*(A(2)*A(6)-A(5)*A(3))
      RETURN
      END
C
      SUBROUTINE DET3_WITH_DERIVS(MAT_IN,DETER,DDETDE)
      IMPLICIT NONE
C
C-----Calculates determinant of 3x3 amtrix and its derivatives wrt 
C-----elements of the matrix. Matrix assumed to general real matrix
      REAL MAT_IN(3,3),DDETDE(3,3),DETER
C

      DDETDE(1,1) =  MAT_IN(2,2)*MAT_IN(3,3) - MAT_IN(2,3)*MAT_IN(3,2)
      DDETDE(1,2) = -MAT_IN(2,1)*MAT_IN(3,3) + MAT_IN(2,3)*MAT_IN(3,1)
      DDETDE(1,3) =  MAT_IN(2,1)*MAT_IN(3,2) - MAT_IN(2,2)*MAT_IN(3,1)
      DDETDE(2,1) = -MAT_IN(1,2)*MAT_IN(3,3) + MAT_IN(1,3)*MAT_IN(3,2)
      DDETDE(2,2) =  MAT_IN(1,1)*MAT_IN(3,3) - MAT_IN(1,3)*MAT_IN(3,1)
      DDETDE(2,3) = -MAT_IN(1,1)*MAT_IN(3,2) + MAT_IN(2,2)*MAT_IN(3,1)
      DDETDE(3,1) =  MAT_IN(1,2)*MAT_IN(2,3) - MAT_IN(2,2)*MAT_IN(1,3)
      DDETDE(3,2) = -MAT_IN(1,1)*MAT_IN(2,3) + MAT_IN(2,1)*MAT_IN(1,3)
      DDETDE(3,3) =  MAT_IN(1,1)*MAT_IN(2,2) - MAT_IN(1,2)*MAT_IN(2,1)
c
C---Find determinant
      DETER = MAT_IN(1,1)*DDETDE(1,1) + MAT_IN(1,2)*DDETDE(1,2)+ 
     +        MAT_IN(1,3)*DDETDE(1,3)
      RETURN
      END
C
      SUBROUTINE PLANE_AND_DERIVS_R(NPLANE,X_INP,VM,D,DVMDX,DDDX,
     &           IERROR)
      IMPLICIT NONE
C
C----Finds equation of lsq plane going through given coordinates and
C---derivatives of coefficients wrt positional parameters
      INTEGER NPLANE,IERROR
      REAL D
      REAL X_INP(3,*),VM(3),DVMDX(3,*),DDDX(3,*)
C
C----Local parameters
      INTEGER I,J,K,I1,INFO1,RANK_L,LWORK,NPLANE3
      REAL YYS2(3,3)
      REAL XS(3),X_CEN(3,50),RHS(3,150),
     &     WORKSPACE(1000),EIGENV(3)
      REAL TOLER_L,FNPLANE,SFMIN,SLAMCH,DEL_X,DEL_Y,DEL_Z
      LOGICAL ERROR
      DATA TOLER_L/1.0E-6/
      EXTERNAL SLAMCH,MAT2VECT,MAT2VEC
      
      LWORK = 1000
C     
C--Find machine parameters
      SFMIN = SLAMCH('S')
C
C
C---Return while it is not too late
      IERROR = 0
      IF(NPLANE.LE.0) THEN
        IERROR = 1
        RETURN
      ENDIF
C
C---Calculate the centre
      FNPLANE = REAL(NPLANE)
      NPLANE3 = NPLANE*3
      DO   I=1,3
        XS(I) = 0.0
        DO  J=1,NPLANE
          XS(I) = XS(I) + X_INP(I,J)
        ENDDO
        XS(I) = XS(I)/FNPLANE
      ENDDO
C
c--Relative (to the center) coordinates
      DO I=1,3
        DO   J=1,NPLANE
          X_CEN(I,J) = X_INP(I,J)-XS(I)
        ENDDO
      ENDDO
C
C--Setup matrix 3x3. It includes normalisation also.
      DO   I=1,3
         DO  J=1,3
           YYS2(I,J) = 0.0
           DO  K=1,NPLANE
             YYS2(I,J) = YYS2(I,J) + X_CEN(I,K)*X_CEN(J,K)
           ENDDO
         ENDDO
       ENDDO
cd       DO   I=1,3
cd         DO   J=1,3
cd           YYS1(I,J) = YYS2(I,J)
cd         ENDDO
cd       ENDDO
C
       call seig3dim(yys2,eigenv,info1)
c      CALL SSYEV_MO('V','U',3,YYS2,3,EIGENV,WORKSPACE,LWORK,INFO1)
C
C---Eigenvector corresponding to minimum eigenvector which is first one.
C---Be careful here, may be other way around
       VM(1) = YYS2(1,1)
       VM(2) = YYS2(2,1)
       VM(3) = YYS2(3,1)
C
C---
       D = VM(1)*XS(1) + VM(2)*XS(2)+VM(3)*XS(3)
C
cd       YYS1(1,4) = VM(1)
cd       YYS1(2,4) = VM(2)
cd       YYS1(3,4) = VM(3)
C
C---Now calculate all right hand sides. Check right hand side carefully
       I1 = 0
       DO    I=1,NPLANE
         I1 = I1 + 1
         DEL_X =  2.0*VM(1)*(X_CEN(1,I)*VM(1)+X_CEN(2,I)*VM(2)+
     &                              X_CEN(3,I)*VM(3))
         DVMDX(1,I1) = -(2.0*VM(1)*X_CEN(1,I)+VM(2)*X_CEN(2,I)+
     &                            VM(3)*X_CEN(3,I)) + DEL_X*VM(1)

         DVMDX(2,I1) = -VM(1)*X_CEN(2,I) + DEL_X*VM(2)
         DVMDX(3,I1) = -VM(1)*X_CEN(3,I) + DEL_X*VM(3)
         DEL_Y = 2.0*VM(2)*(X_CEN(1,I)*VM(1)+X_CEN(2,I)*VM(2)+
     &                              X_CEN(3,I)*VM(3))
cd         RHS(4,I1) = 0
         I1 = I1 + 1
         DVMDX(1,I1) = -VM(2)*X_CEN(1,I) + DEL_Y*VM(1)
         DVMDX(2,I1) = -(VM(1)*X_CEN(1,I)+2.0*VM(2)*X_CEN(2,I)+
     &                                VM(3)*X_CEN(3,I)) + DEL_Y*VM(2)
         DVMDX(3,I1) = -VM(2)*X_CEN(3,I) + DEL_Y*VM(3)
cd         RHS(4,I1) = 0.0
         DEL_Z = 2.0*VM(1)*(X_CEN(1,I)*VM(1)+X_CEN(2,I)*VM(2)+
     &                              X_CEN(3,I)*VM(3))
         I1 = I1 + 1
         DVMDX(1,I1) = -VM(3)*X_CEN(1,I) + DEL_Z*VM(1)
         DVMDX(2,I1) = -VM(3)*X_CEN(2,I) + DEL_Z*VM(2)
         DVMDX(3,I1) = -(VM(1)*X_CEN(1,I)+    VM(2)*X_CEN(2,I)+
     &                           2.0*VM(3)*X_CEN(3,I)) + DEL_Z*VM(3)
cd         RHS(4,I1) = 0.0
      ENDDO
      EIGENV(1) = 0.0
      EIGENV(2) = EIGENV(2)-EIGENV(1)
      EIGENV(3) = EIGENV(3)-EIGENV(1)
      DO    I=1,NPLANE3
        CALL MAT2VECT(3,3,YYS2,DVMDX(1,I),RHS(1,I),ERROR)
        DO   J=1,3
          IF(EIGENV(J).GT.AMAX1(EIGENV(3)*TOLER_L,SFMIN)) THEN
            RHS(J,I)=RHS(J,I)/EIGENV(J)
          ELSE
            RHS(J,I) = 0.0
          ENDIF
        ENDDO
        CALL MAT2VEC(3,3,YYS2,RHS(1,I),DVMDX(1,I),ERROR)
      ENDDO  
C
C---Now call SVD to solve equations to find derivatives.
cd      LWORK = NWORKSPACE - 3
cd      CALL SGELSS(3,3,NPLANE3,YYS1,3,RHS,3,,WORKSPACE(1),TOLER_L,
cd     &            RANK_L,WORKSPACE(4),LWORK,INFO)
cd      WRITE(*,*)RANK_L
cd      DO   I=1,NPLANE3
cd        DVMDX(1,I) = RHS(1,I)
cd        DVMDX(2,I) = RHS(2,I)
cd        DVMDX(3,I) = RHS(3,I)
cd      ENDDO
C
cd      WRITE(*,*)'Solved '
cd      STOP

C---Calculate derivatives of d wrt positions
      DO   I=1,NPLANE
         I1 = 3*I-2
         DDDX(1,I) = DVMDX(1,I1  )*XS(1) + DVMDX(2,I1  )*XS(2) + 
     &               DVMDX(3,I1  )*XS(3) + VM(1)/FNPLANE
         DDDX(2,I) = DVMDX(1,I1+1)*XS(1) + DVMDX(2,I1+1)*XS(2) + 
     &               DVMDX(3,I1+1)*XS(3) + VM(2)/FNPLANE
         DDDX(3,I) = DVMDX(1,I1+2)*XS(1) + DVMDX(2,I1+2)*XS(2) + 
     &               DVMDX(3,I1+2)*XS(3) + VM(3)/FNPLANE
      ENDDO
       
C
C---Return
      RETURN

      END

C     =================================
      SUBROUTINE PLANE_R(N,X,VM,D)
C     =================================
C
C    Fit a least-squares plane to a set of points
C    Uses eigenvalue analysis. Find eigenvalues of normal
C    matrix and eigenvector corresponding to the minimum eigenvalue 
C    is coefficients of plane 
C
      implicit none
C
C
C
C     .. Scalar Arguments ..
      REAL     D
      INTEGER  N
C     ..
C     .. Array Arguments ..
      REAL   VM(3),X(3,50)
C     ..
C     .. Local Scalars ..
      INTEGER I,J,K,J1,I_MIN,LWORK,INFO
      REAL    EIGEN_MIN ,VNORM 
C     ..
C     .. Local Arrays ..
      REAL    XS(3),E_V(3),E_WORK(1000),XXS1(3,3)
C     ..
C     .. Intrinsic Functions ..
      INTRINSIC ABS,SQRT
C     ..
C
C---- Set up the a matrix
      LWORK = 1000
      DO    I = 1,3
        XS(I) = 0.0
        DO    K = 1,N
          XS(I) = XS(I) + X(I,K)
        ENDDO
      ENDDO 
      J1 = 0
      DO    I = 1,3
        DO    J = 1,3
          XXS1(I,J) = 0.0
          DO    K = 1,N
            XXS1(I,J) = X(I,K)*X(J,K) + XXS1(I,J)
          ENDDO
          XXS1(I,J) = XXS1(I,J) - XS(I)*XS(J)/FLOAT(N)
        ENDDO
      ENDDO
C
C---Use lapack.
      call seig3dim(xxs1,e_v,info)
cd      CALL SSYEV_MO('V','U',3,XXS1,3,E_V,E_WORK,LWORK,INFO)
      if(info.gt.0) then
        write(*,*)'Problem in plane',info
        stop
      endif
      VM(1) = XXS1(1,1)
      VM(2) = XXS1(2,1)
      VM(3) = XXS1(3,1)
      VNORM = SQRT(VM(1)**2+VM(2)**2+VM(3)**2)
      DO   I=1,3
        VM(I) = VM(I)/VNORM
      ENDDO
      D     = (VM(1)*XS(1)+VM(2)*XS(2)+VM(3)*XS(3))/FLOAT(N)
C---Find eigenvalues of XXS
C
      RETURN
      END      

      SUBROUTINE INIT_VEC(SIZE,VEC,VALUE)
C
      REAL    VEC(*),VALUE
      INTEGER SIZE

cd      IF(SIZE.GT.0) THEN
        DO    I=1,SIZE
          VEC(I) = VALUE
        ENDDO
cd      ENDIF
      RETURN
      END
C
      SUBROUTINE INIT_VECN(SIZE,VECN,VALUEN)
C
      INTEGER SIZE,VECN(*),VALUEN
      
      DO    I=1,SIZE
        VECN(I) = VALUEN
      ENDDO
      RETURN
      END
C
C
      SUBROUTINE VECCOPY_R(N,NMAX,VEC_OUT,ALPHA,VEC_IN)
C
C---Copies vector VEC_IN to VEC_OUT. It also applies VEC_IN by
C---alpha if alpha .NE. 1.0
C---N     is current size of vector
C---NMAX  is maximum size of vector
      REAL VEC_IN(*),VEC_OUT(*),ALPHA
      INTEGER N,NMAX
C
      IERROR = 0
      IF(N.GT.NMAX) THEN
        IERROR = 1
        CALL ERRWRT(1,'Size mismatch in VECCOPY_R')
      ENDIF
      IF(ALPHA.NE.1.0) THEN
        DO   I=1,N
          VEC_OUT(I) = ALPHA*VEC_IN(I)
        ENDDO
      ELSE
        DO    I=1,N
          VEC_OUT(I) = VEC_IN(I)
        ENDDO
      ENDIF
      RETURN
      END
C
      SUBROUTINE AVECPVEC_R(N,NMAX,ALPHA,VEC1,VEC2,VEC_OUT,IERROR)
C
C   VEC_OUT = ALPHA*VEC1 + VEC2
      REAL VEC1(*),VEC2(*),VEC_OUT(*)
      REAL ALPHA
      INTEGER N,NMAX,I,IERROR

      IF(N.LE.0.OR.N.GT.NMAX)THEN
        CALL ERRWRT(1,'Size mismatch In AVECPVEC')
        IERROR = 1
      ENDIF
      IF(ALPHA.EQ.0.0) THEN
        DO    I=1,N
          VEC_OUT(I) = VEC2(I)
        ENDDO
      ELSEIF(ALPHA.NE.1.0) THEN
        DO   I=1,N
          VEC_OUT(I) = VEC2(I) + ALPHA*VEC1(I)
        ENDDO
      ELSE
        DO   I=1,N
          VEC_OUT(I) = VEC2(I) + VEC1(I)
        ENDDO
      ENDIF
      IERROR = 0
      RETURN
      END
C
C----Operation on matrices and vectors
C
      SUBROUTINE MAT2IDENT(NP,N,AMAT,ERROR)
C
C----Initialises matrix to identity matrix
C
C------Input and output parameters
C
C----NP     is maximum size of matrix
C----N      is current size of matrix
C----AMAT   is matrix to be initialised
C----ERROR  is .TRUE. if N.GT.NP or N.LT.0
C----------------------------------------------
      REAL    AMAT(NP,NP)
      INTEGER NP,N,I,J
      REAL    SZERO,SUNIT
      LOGICAL ERROR
      DATA    SZERO/0.0E0/,SUNIT/1.0E0/
C
      ERROR = .FALSE.
      IF(N.LE.0 .OR. N.GT.NP) THEN
        ERROR = .TRUE.
        RETURN
      ENDIF

      DO    I=1,N
        DO    J=1,N
          IF(I.EQ.J) THEN
             AMAT(I,J) = SUNIT
          ELSE
             AMAT(I,J) = SZERO
          ENDIF
        ENDDO
      ENDDO
      RETURN
      END
C
      SUBROUTINE MAT2VEC(NP,N,AMAT,AVEC_IN,AVEC_OUT,ERROR)
C
C---Multiplies matrix AMAT which has size NxN to AVEC_IN
C---Resultant vector is in AVEC_OUT
C
C---NP - maximal size of matrix and vectors
C---N  - current size of matrix and vectors
C---AMAT     - Matrix will applied to AVEC_IN
C---AVEC_IN  - Input vector which will be multiplied to AMAT
C---AVEC_OUT - resultant output vector
C---ERROR    - simple error checks
C--------------------------------------------------------------------
      REAL AMAT(NP,NP),AVEC_IN(NP),AVEC_OUT(NP)
      LOGICAL ERROR
C
      ERROR = .FALSE.
      IF(N.LE.0 .OR. N.GT.NP) THEN
        ERROR = .TRUE.
        RETURN
      ENDIF
      DO    I=1,N
        AVEC_OUT(I) = 0.0
      ENDDO
      DO    J=1,N
        DO    I=1,N
          AVEC_OUT(I) = AVEC_OUT(I) + AMAT(I,J)*AVEC_IN(J)
        ENDDO
      ENDDO
      RETURN
      END
C
      SUBROUTINE MAT2VECT(NP,N,AMAT,AVEC_IN,AVEC_OUT,ERROR)
C
C---Multiplies transposed matrix AMAT which has size NxN to AVEC_IN
C---Resultant vector is in AVEC_OUT
C
C------Input and output parameters
C
C---NP       - maximal size of matrix and vectors
C---N        - current size of matrix and vectors
C---AMAT     - Matrix will applied to AVEC_IN
C---AVEC_IN  - Input vector which will be multiplied to AMAT
C---AVEC_OUT - resultant output vector
C---ERROR    - simple error checks
C--------------------------------------------------------------------
      REAL    AMAT(NP,NP),AVEC_IN(NP),AVEC_OUT(NP)
      INTEGER NP,N,I,J
      LOGICAL ERROR
C
      ERROR = .FALSE.
      IF(N.LE.0 .OR. N.GT.NP) THEN
        ERROR = .TRUE.
        RETURN
      ENDIF
      DO    I=1,N
        AVEC_OUT(I) = 0.0
        DO    J=1,N
          AVEC_OUT(I) = AVEC_OUT(I) + AMAT(J,I)*AVEC_IN(J)
        ENDDO
      ENDDO
      RETURN
      END
C
      SUBROUTINE IMAT2VECT(NP,N,AMAT,AVEC_IN,AVEC_OUT,ERROR)
C
C---Multiplies transposed matrix AMAT which has size NxN to AVEC_IN
C---Resultant vector is in AVEC_OUT
C
C------Input and output parameters
C
C---NP       - maximal size of matrix and vectors
C---N        - current size of matrix and vectors
C---AMAT     - Matrix will applied to AVEC_IN
C---AVEC_IN  - Input vector which will be multiplied to AMAT
C---AVEC_OUT - resultant output vector
C---ERROR    - simple error checks
C--------------------------------------------------------------------
      INTEGER    AMAT(NP,NP),AVEC_IN(NP),AVEC_OUT(NP)
      INTEGER NP,N,I,J
      LOGICAL ERROR
C
      ERROR = .FALSE.
      IF(N.LE.0 .OR. N.GT.NP) THEN
        ERROR = .TRUE.
        RETURN
      ENDIF
      DO    I=1,N
        AVEC_OUT(I) = 0
        DO    J=1,N
          AVEC_OUT(I) = AVEC_OUT(I) + AMAT(J,I)*AVEC_IN(J)
        ENDDO
      ENDDO
      RETURN
      END
c
      SUBROUTINE MATTRANS(NP,N,AMAT,AMAT_OUT,ERROR)
C
C---Transpose matrix AMAT and stroes in AMAT_OUT
C
C-----Input and output parameters
C
C---NP       - maximal size
C---N        - current size
C---AMAT     - Input matrix to be transposed
C---AMAT_OUT - resultant output matrix
C---ERROR    - simple error checks
C--------------------------------------------------------------------
      REAL AMAT(NP,NP),AMAT_OUT(NP,NP)
      INTEGER NP,N,I1,I2
      LOGICAL ERROR
C
      ERROR = .FALSE.
      IF(N.LE.0 .OR. N.GT.NP) THEN
        ERROR = .TRUE.
        RETURN
      ENDIF
      DO    I1=1,N-1
        DO    I2=I1+1,N
          AMAT_OUT(I1,I2) = AMAT(I2,I1)
          AMAT_OUT(I2,I1) = AMAT(I1,I2)
        ENDDO
      ENDDO
      DO    I1=1,N
        AMAT_OUT(I1,I1) = AMAT(I1,I1)
      ENDDO
      RETURN
      END
C
      SUBROUTINE MAT2MAT(NP,N,AMAT1,AMAT2,AMAT_OUT,ERROR)
C
C---Multiples AMAT1 to AMAT2 and writes results to AMAT_OUT
C
C---NP       - maximal size
C---N        -  current size
C---AMAT1    - Input left matrix
C---AMAT2    - Input right matrix
C---AMAT_OUT - resultant output matrix
C---ERROR    - simple error checks
C--------------------------------------------------------------------
      REAL AMAT1(NP,NP),AMAT2(NP,NP),AMAT_OUT(NP,NP)
      INTEGER N,NP,I,J,K
      LOGICAL ERROR
C
      ERROR = .FALSE.
      IF(N.LE.0. .OR. N.GT.NP) THEN
        ERROR = .TRUE.
        RETURN
      ENDIF
      DO    J=1,N
        DO    I=1,N
          AMAT_OUT(I,J) = 0.0
        ENDDO
      ENDDO
c
      DO   J=1,N
        DO   K=1,N
          DO    I=1,N
            AMAT_OUT(I,K) = AMAT_OUT(I,K) + AMAT1(I,J)*AMAT2(J,K)
          ENDDO
        ENDDO
      ENDDO
      RETURN
      END
C
      SUBROUTINE MATT2MAT(NP,N,AMAT1,AMAT2,AMAT_OUT,ERROR)
C
C---Multiples transpose of AMAT1 to AMAT2 and writes results to AMAT_OUT
C
C----Input and output parameters
C
C---NP       - maximal size
C---N        -  current size
C---AMAT1    - Input left matrix
C---AMAT2    - Input right matrix
C---AMAT_OUT - resultant output matrix
C---ERROR    - simple error checks
C--------------------------------------------------------------------
      REAL AMAT1(NP,NP),AMAT2(NP,NP),AMAT_OUT(NP,NP)
      INTEGER NP,N,I,J,K
      LOGICAL ERROR
C
      ERROR = .FALSE.
      IF(N.LE.0 .OR. N.GT.NP) THEN
        ERROR = .TRUE.
        RETURN
      ENDIF
      DO    J=1,N
        DO    I=1,N
          AMAT_OUT(I,J) = 0.0
        ENDDO
      ENDDO
      DO   J=1,N
        DO   I=1,N
          DO    K=1,N
            AMAT_OUT(I,K) = AMAT_OUT(I,K) + AMAT1(J,I)*AMAT2(J,K)
          ENDDO
        ENDDO
      ENDDO
      RETURN
      END
C
      SUBROUTINE MATT2MATT(NP,N,AMAT1,AMAT2,AMAT_OUT,ERROR)
C
C---Multiples transpose of AMAT1 to transpose of AMAT2 and writes results
C---to AMAT_OUT
C
C-----Input and output parameters
C
C---NP       - maximal size
C---N        -  current size
C---AMAT1    - Input left matrix
C---AMAT2    - Input right matrix
C---AMAT_OUT - resultant output matrix
C---ERROR    - simple error checks
C--------------------------------------------------------------------
      REAL AMAT1(NP,NP),AMAT2(NP,NP),AMAT_OUT(NP,NP)
      INTEGER NP,N,I,J,K
      LOGICAL ERROR
C
      ERROR = .FALSE.
      IF(N.LE.0 .OR. N.GT.NP) THEN
        ERROR = .TRUE.
        RETURN
      ENDIF
      DO    J=1,N
        DO    I=1,N
          AMAT_OUT(I,J) = 0.0
        ENDDO
      ENDDO
      DO   J=1,N
        DO   K=1,N
          DO    I=1,N
            AMAT_OUT(I,K) = AMAT_OUT(I,K) + AMAT1(J,I)*AMAT2(K,J)
          ENDDO
        ENDDO
      ENDDO
      RETURN
      END
C
      SUBROUTINE MAT2MATT(NP,N,AMAT1,AMAT2,AMAT_OUT,ERROR)
C
C---Multiples AMAT1 to transpose of AMAT2 and writes results
C---to AMAT_OUT
C
C-----Input and output parameters
C
C---NP       - maximal size
C---N        -  current size
C---AMAT1    - Input left matrix
C---AMAT2    - Input right matrix
C---AMAT_OUT - resultant output matrix
C---ERROR    - simple error checks
C--------------------------------------------------------------------
      REAL AMAT1(NP,NP),AMAT2(NP,NP),AMAT_OUT(NP,NP)
      LOGICAL ERROR
C
      ERROR = .FALSE.
      IF(N.LE.0 .OR. N.GT.NP) THEN
        ERROR = .TRUE.
        RETURN
      ENDIF
      DO   I1=1,N
        DO   I2=1,N
          AMAT_OUT(I1,I2) = 0.0
          DO    I3=1,N
            AMAT_OUT(I1,I2) =
     +          AMAT_OUT(I1,I2) + AMAT1(I1,I3)*AMAT2(I2,I3)
          ENDDO
        ENDDO
      ENDDO
      RETURN
      END
C
      SUBROUTINE PUT_K_MATRIX(NP,N,NMAX,R_IN,R_OUT,K,ERROR)
C
C---Puts matrix R_IN into K-s position of R_OUT
C
C---Input and output parameters:
C
C NP    - maximum size of matrix R_IN(NP,NP)
C N     - current size of R_IN and R_OUT(NP,NP,NMAX)
C NMAX  - maximum number of matrices in R_OUT
C K     - R_IN will be put into k-s position of R_OUT
C R_IN  - input matrix with maximum size NP R_IN(NP,NP)
C R_OUT - output matrix R_OUT(NP,NP,NMAX)
C ERROR - simple error checks
C----------------------------------------------------------
      REAL    R_IN(NP,NP),R_OUT(NP,NP,NMAX)
      INTEGER NP,N,NMAX,K,I1,I2
      LOGICAL ERROR
C
      ERROR = .FALSE.
      IF(K.LE.0 .OR. K.GT.NMAX .OR. N.LE.0 .OR. N.GT.NP) THEN
        ERROR = .TRUE.
        RETURN
      ENDIF
      DO    I1 = 1,N
        DO    I2 = 1,N
          R_OUT(I1,I2,K) = R_IN(I1,I2)
        ENDDO
      ENDDO
      RETURN
      END
C
      SUBROUTINE GET_K_MATRIX(NP,N,NMAX,R_IN,R_OUT,K,ERROR)
C
C---Gets K-s matrix from R_IN and puts into R_OUT
C
C---Input and output parameters:
C
C NP    - maximum size of matrix R_IN(NP,NP,NMAX)
C N     - current size of R_IN and R_OUT(NP,NP)
C NMAX  - maximum number of matrices in R_OUT
C K     - R_IN will be put into k-s position of R_OUT
C R_IN  - input matrix with maximum size NP R_IN(NP,NP,NMAX)
C R_OUT - output matrix R_OUT(NP,NP)
C ERROR - simple error checks
C----------------------------------------------------------
      REAL    R_IN(NP,NP,NMAX),R_OUT(NP,NP)
      INTEGER NP,N,NMAX,K,I1,I2
      LOGICAL ERROR
C
      ERROR = .FALSE.
      IF(K.LE.0 .OR. K.GT.NMAX .OR. N.LE.0 .OR. N.GT.NP) THEN
        ERROR = .TRUE.
        RETURN
      ENDIF
      DO    I1 = 1,N
        DO    I2 = 1,N
          R_OUT(I1,I2) = R_IN(I1,I2,K)
        ENDDO
      ENDDO
      RETURN
      END
C
C---Solution of linear equations and matrix inversion problems
C
C
C_BEGIN_GAUSSJ
C
      SUBROUTINE GAUSSJ(A,N,NP,B,M,MP,ERROR)
C     ===========================
C
C
C---- Purpose
C     =======
C
C           Linear equation solution by Gauss-Jordan method
C
C---- Usage
C     ======
C
C           CALL GAUSSJ(A,N,NP,B,M,MP)
C
C---- Description of parameters
C     =========================
C
C    A - input matrix of order N*N, stored in an array of NP*NP
C    On output: replaced by  resultant inverse.
C
C    N - order of matrix A
C
C    NP - dimension of array in which A is stored.
C
C    B - input matrix of order N*M containing the M right hand side vectors,
C        stored in an arry of dimensions NP * MP.
C    On output: replaced by  corresponding set of solution vectors.
C
C---- Remarks
C     =======
C
C---- Method
C     ======
C
C     The standard gauss-jordan method is used. the determinant
C     is also calculated. a determinant of zero indicates that
C     the matrix is singular.
C
C
C_END_GAUSSJ
C
C---- Search for largest element
C
C     .. Scalar Arguments ..
         Parameter (NMAX=500)
C     ..
C     .. Array Arguments ..
      REAL A(NP,NP),B(NP,MP)
      INTEGER IPIV(NMAX),INDXR(NMAX),INDXC(NMAX)
      LOGICAL ERROR
C     The integer arrays IPIV INDXR and INDXC are used forthe pivot bookkeeping.
C     ..
C     .. Local Scalars ..
C     ..
C     .. Intrinsic Functions ..
      INTRINSIC ABS
C     ..
C
      IROW = 1
      ICOL = 1
      ERROR=.FALSE.
C
      DO 11 J = 1,N
       IPIV(J)=0
  11  CONTINUE
C
      DO 22 I = 1,N
        BIG = 0.0
        DO 13 J = 1,N
          IF(IPIV(J).NE.1) THEN
          DO 12 K = 1,N
            IF(IPIV(K).EQ.0) THEN
              IF (ABS(A(J,K)) .GE. BIG) THEN
                BIG = ABS(A(J,K))
                IROW = J
                ICOL = K
              END IF
            ELSE IF(IPIV(K).GT.1) THEN
               ERROR=.TRUE.
               RETURN
            END IF
  12      CONTINUE
          END IF
  13    CONTINUE
      IPIV(ICOL)=IPIV(ICOL) + 1
C
      IF (IROW.NE.ICOL) THEN
        DO 14 L = 1,N
          DUM = A(IROW,L)
          A(IROW,L) = A(ICOL,L)
          A(ICOL,L) = DUM
  14    CONTINUE
C
        DO 15 L = 1,M
          DUM = B(IROW,L)
          B(IROW,L) = B(ICOL,L)
          B(ICOL,L) = DUM
  15    CONTINUE
      END IF 
C
C  Divide pivot row by pivot element
      INDXR(I) = IROW
      INDXC(I) = ICOL
      IF ( A(ICOL,ICOL).EQ.0) THEN
        ERROR=.TRUE.
        RETURN
      END IF 
      PIVINV = 1.0/A(ICOL,ICOL)
      A(ICOL,ICOL) = 1.0
C
        DO 16 L = 1,N
          A(ICOL,L) = A(ICOL,L)*PIVINV
  16    CONTINUE
C
        DO 17 L = 1,M
          B(ICOL,L) = B(ICOL,L)*PIVINV
  17    CONTINUE
C
        DO 21 LL = 1,N
          IF(LL.NE.ICOL) THEN
            DUM = A(LL,ICOL)
              A(LL,ICOL) = 0.0
C
              DO 18 L = 1,N
                A(LL,L) = A(LL,L) -  A(ICOL,L)*DUM
  18          CONTINUE
C
              DO 19 L = 1,M
                B(LL,L) = B(LL,L) -  B(ICOL,L)*DUM
  19          CONTINUE
          END IF
  21    CONTINUE
  22  CONTINUE
C
      DO 24 L = N,1,-1
        IF(INDXR(L).NE.INDXC(L) ) THEN
          DO 23 K = 1,N
            DUM = A(K,INDXR(L))
              A(K,INDXR(L)) = A(K,INDXC(L))
              A(K,INDXR(L)) = DUM
  23      CONTINUE
        END IF
  24  CONTINUE
      RETURN
      END
C
      SUBROUTINE MATINV3_R(MAT_IN,MAT_OUT)
      IMPLICIT NONE
C
C---simple marix inversion for 3x3 matrix
      REAL MAT_IN(3,3),MAT_OUT(3,3)
c
C---Local array
      REAL A_MINOR(3,3),DETER,SIGN
      INTEGER I,J
C
C---First calculate minors of matrix

      A_MINOR(1,1) = MAT_IN(2,2)*MAT_IN(3,3) - MAT_IN(2,3)*MAT_IN(3,2)
      A_MINOR(1,2) = MAT_IN(2,1)*MAT_IN(3,3) - MAT_IN(2,3)*MAT_IN(3,1)
      A_MINOR(1,3) = MAT_IN(2,1)*MAT_IN(3,2) - MAT_IN(2,2)*MAT_IN(3,1)
      A_MINOR(2,1) = MAT_IN(1,2)*MAT_IN(3,3) - MAT_IN(1,3)*MAT_IN(3,2)
      A_MINOR(2,2) = MAT_IN(1,1)*MAT_IN(3,3) - MAT_IN(1,3)*MAT_IN(3,1)
      A_MINOR(2,3) = MAT_IN(1,1)*MAT_IN(3,2) - MAT_IN(1,2)*MAT_IN(3,1)
      A_MINOR(3,1) = MAT_IN(1,2)*MAT_IN(2,3) - MAT_IN(2,2)*MAT_IN(1,3)
      A_MINOR(3,2) = MAT_IN(1,1)*MAT_IN(2,3) - MAT_IN(2,1)*MAT_IN(1,3)
      A_MINOR(3,3) = MAT_IN(1,1)*MAT_IN(2,2) - MAT_IN(1,2)*MAT_IN(2,1)
c
C---Find determinant
      DETER = MAT_IN(1,1)*A_MINOR(1,1) - MAT_IN(1,2)*A_MINOR(1,2)+ 
     +        MAT_IN(1,3)*A_MINOR(1,3)
      SIGN = 1.0
      DO    I=1,3
        DO   J=1,3
          MAT_OUT(I,J) = SIGN*A_MINOR(J,I)/DETER
          SIGN         = -SIGN
        ENDDO
      ENDDO
      RETURN
      END

C
      SUBROUTINE CGSOLVE(NV,NMAX,NCYCL,TOLER,A,V,DV,R,SK,P,F,WORKSPACE,
     +                  LWORK,SCALE1,MATMUL1,SCALE_SHIFTS,SCALE1_INV,
     +                  IERROR)
C
C     Solves normal equation (a*dv=v) by conjugate gradient
C     This routine is efficient if matrix is sparse. Routines
C     SCALE1, MATMUL, SCALE_SHIFTS should be supplied
C
C     see for example: Golub & Loan, Matrix Computations 
C                      Press et al. Numericl recipes
C
C----If matrix AM and vector V are going to be used after this routine
C----they should be saved by the calling subroutine
C   
      REAL TOLER1
      PARAMETER (TOLER1 = 1.0E-12)
      REAL EPS_LOC
      PARAMETER (EPS_LOC = 1.0E-8)
C
      REAL      A(*),V(*),DV(*),SK(*),P(*),F(*),R(*),WORKSPACE(*)
      REAL      TOLER
      INTEGER   NV,NMAX,IERROR
      EXTERNAL  SCALE1,MATMUL1,SCALE_SHIFTS,SCALE1_INV
C
      IF(TOLER.LE.0.0) TOLER = TOLER1
C
C----Add alpha |x|^2 here alpha = lambda_max*tolerance. It is to stabilise
C----linear equation solution. 
      IERROR = 0
      IF(NV.GT.NMAX.OR.NV.LE.0) THEN
        IERROR = 1
        CALL ERRWRT(0,'CGSOLVE Size mismatch')
        RETURN
      ENDIF
C    
C----Call predoconditioner. Should be supplied
      CALL SCALE1(NV,NMAX,F,A,SK,V,WORKSPACE,LWORK,IERROR)
      IF(IERROR.GT.0) THEN
        CALL ERRWRT(0,'In CGSOLVE, after SCALE1')
        RETURN
      ENDIF
C
C---Initialise
      DO    I = 1,NV
        DV(I)  = 0.0
        R(I)   = -V(I)
        P(I)   = R(I)
      ENDDO
      R1SUM = DOT_R(NV,NMAX,R,R)
      IF(R1SUM.LE.TOLER1)RETURN
C
C----First cycle of iteration 
      CALL MATMUL1(NV,NMAX,P,A,F,WORKSPACE,LWORK,IERROR)
      IF(IERROR.GT.0) THEN
        CALL ERRWRT(0,'In CGSOLVE, After MATMUL1 1')
        RETURN
      ENDIF
      ALPHA = DOT_R(NV,NMAX,P,F)
      ALPHA = R1SUM/ALPHA
      CALL AVECPVEC_R(NV,NMAX,ALPHA,P,DV,DV,IERROR)
      CALL AVECPVEC_R(NV,NMAX,-ALPHA,F,R,R,IERROR)
      IF(IERROR.GT.0) THEN
        CALL ERRWRT(0,'In CGSOLVE After AVECPVEC_R 1')
        RETURN
      ENDIF
      R2SUM    = DOT_R(NV,NMAX,R,R)
      VNORM    = DOT_R(NV,NMAX,V,V)
      TEST_LIM = TOLER*SQRT(VNORM)
      IF(R1SUM.LE.0.0) THEN
        IERROR = 1
        CALL ERRWRT(1,'CGSOLVE R1SUM le 0.0')
      ENDIF
C
C----Loop over iterations.
      IF(SQRT(R2SUM).LE.TEST_LIM) GOTO 120
      FVAL0 = 0.0
      DO    ITER = 1,NCYCL
        BETA = R2SUM/R1SUM
        CALL AVECPVEC_R(NV,NMAX,BETA,P,R,P,IERROR)
        IF(IERROR.GT.0) THEN
          CALL ERRWRT(0,'In CGSOLVE, After AVECPVEC_R')
          RETURN
        ENDIF
C
C----Multiply matrix by vector. It is external subroutine and should be 
C----supplied
        CALL MATMUL1(NV,NMAX,P,A,F,WORKSPACE,LWORK,IERROR)
        IF(IERROR.GT.0) THEN
          CALL ERRWRT(0,'In CGSOLVE after MATMUL1')
          RETURN
        ENDIF
        ALPHA = DOT_R(NV,NMAX,P,F)
        ALPHA = R2SUM/ALPHA
        CALL AVECPVEC_R(NV,NMAX,ALPHA,P,DV,DV,IERROR)
        CALL AVECPVEC_R(NV,NMAX,-ALPHA,F,R,R,IERROR)
        IF(IERROR.GT.0) THEN
          CALL ERRWRT(0,'In CGSOLVE, After AVECPVEC_R ')
          RETURN
        ENDIF
        R1SUM = R2SUM
        R2SUM = DOT_R(NV,NMAX,R,R)
cd        WRITE(*,'(A,G12.3)')'R2sum ',R2SUM
      CALL MATMUL1(NV,NMAX,DV,A,F,WORKSPACE,LWORK,IERROR)

      FVAL1 = 0.5*DOT_R(NV,NMAX,F,DV) + DOT_R(NV,NMAX,DV,V)
      WRITE(*,*)FVAL1,FVAL0,SQRT(R2SUM),TEST_LIM
      IF(2.0*ABS(FVAL0-FVAL1).LE.
     &            TOLER1*(ABS(FVAL0)+ABS(FVAL1)+EPS_LOC)) GOTO 120
      FVAL0 = FVAL1
        IF(SQRT(R2SUM).LE.TEST_LIM) GOTO 120
      ENDDO
C
  120 CONTINUE
C
      CALL SCALE_SHIFTS(NV,NMAX,SK,DV,IERROR)
      IF(IERROR.GT.0) THEN
        CALL ERRWRT(0,'In CGSOLVE after SCALE_SHIFTS')
        RETURN
      ENDIF
      CALL SCALE1_INV(NV,NMAX,F,A,SK,V,WORKSPACE,LWORK,IERROR)
      RETURN
      END
C
      SUBROUTINE CGSOLVE1(NV,NMAX,NCYCL,TOLER,A,V,DV,R,SK,P,F,WORKSPACE,
     +                  LWORK,SCALE1,MATMUL1,SCALE_SHIFTS,SCALE1_INV,
     +                  IERROR)
C
C     Solves normal equation (A*DV=V) by conjugate gradient
C     This routine is efficient if matrix is sparse. Routines
C     SCALE1, MATMUL, SCALE_SHIFTS should be supplied
C
C     see for example: Golub & Loan, Matrix Computations 
C                      Press et al. Numericl recipes
C
C----If matrix A and vector V are going to be used after this routine
C----they should be saved by the calling subroutine
C   
      REAL TOLER1
      PARAMETER (TOLER1 = 1.0E-6)
      REAL EPS_LOC
      PARAMETER (EPS_LOC = 1.0E-8)
      REAL      A(*),V(*),DV(*),SK(*),P(*),F(*),R(*),WORKSPACE(*)
      REAL      TOLER,DIEN
      REAL      FVAL0,FVAL1
      INTEGER   NV,NMAX,IERROR,I
      EXTERNAL  SCALE1,MATMUL1,SCALE_SHIFTS,SCALE1_INV
C
      IF(TOLER.LE.0.0) TOLER = TOLER1
C
C----Add alpha |x|^2 here alpha = lambda_max*tolerance. It is to stabilise
C----linear equation solution. 
      IERROR = 0
      IF(NV.GT.NMAX.OR.NV.LE.0) THEN
        IERROR = 1
        CALL ERRWRT(0,'CGSOLVE Size mismatch')
        RETURN
      ENDIF
C    
C----Call predoconditioner. Should be supplied
      CALL SCALE1(NV,NMAX,F,A,SK,V,WORKSPACE,LWORK,IERROR)
      IF(IERROR.GT.0) THEN
        CALL ERRWRT(0,'In CGSOLVE, after SCALE1')
        RETURN
      ENDIF
C
C---Initialise
      DO    I = 1,NV
        DV(I)  = 0.0
        R(I)   = -V(I)
        P(I)   = R(I)
      ENDDO
      FVAL0 = 0.0
c
      R1SUM = DOT_R(NV,NMAX,R,R)
      IF(R1SUM.LE.TOLER1)RETURN
C
C----First cycle of iteration 
      CALL MATMUL1(NV,NMAX,P,A,F,WORKSPACE,LWORK,IERROR)
      IF(IERROR.GT.0) THEN
        CALL ERRWRT(0,'In CGSOLVE, After MATMUL1 1')
        RETURN
      ENDIF
      ALPHA = DOT_R(NV,NMAX,P,F)
      ALPHA = R1SUM/ALPHA
      CALL AVECPVEC_R(NV,NMAX,ALPHA,P,DV,DV,IERROR)
      CALL AVECPVEC_R(NV,NMAX,-ALPHA,F,R,R,IERROR)
      IF(IERROR.GT.0) THEN
        CALL ERRWRT(0,'In CGSOLVE After AVECPVEC_R 1')
        RETURN
      ENDIF

      R2SUM    = DOT_R(NV,NMAX,R,R)
      VNORM    = DOT_R(NV,NMAX,V,V)
      TEST_LIM = TOLER*SQRT(VNORM)
      IF(R1SUM.LE.0.0) THEN
        IERROR = 1
        CALL ERRWRT(1,'CGSOLVE R1SUM le 0.0')
      ENDIF
C
      IF(SQRT(R2SUM).LE.TEST_LIM) GOTO 120

c      DIEN = 1.5
C----Loop over iterations.
      DO    ITER = 1,NCYCL
        BETA = R2SUM/R1SUM
        CALL AVECPVEC_R(NV,NMAX,BETA,P,R,P,IERROR)
        IF(IERROR.GT.0) THEN
          CALL ERRWRT(0,'In CGSOLVE, After AVECPVEC_R')
          RETURN
        ENDIF
C
C----Multiply matrix by vector. It is external subroutine and should be 
C----supplied
        CALL MATMUL1(NV,NMAX,P,A,F,WORKSPACE,LWORK,IERROR)
        IF(IERROR.GT.0) THEN
          CALL ERRWRT(0,'In CGSOLVE after MATMUL1')
          RETURN
        ENDIF
cd        DO I = 1,NV
cd           F(I) = F(I) + DIEN*P(I)
cd        ENDDO
c        DIEN = DIEN/2.0
        ALPHA = DOT_R(NV,NMAX,P,F)
        ALPHA = R2SUM/ALPHA
        CALL AVECPVEC_R(NV,NMAX,ALPHA,P,DV,DV,IERROR)
        CALL AVECPVEC_R(NV,NMAX,-ALPHA,F,R,R,IERROR)
        IF(IERROR.GT.0) THEN
          CALL ERRWRT(0,'In CGSOLVE, After AVECPVEC_R ')
          RETURN
        ENDIF
        R1SUM = R2SUM
        R2SUM = DOT_R(NV,NMAX,R,R)
c        WRITE(*,*)'R2sum ',SQRT(R2SUM),TEST_LIM
      CALL MATMUL1(NV,NMAX,DV,A,F,WORKSPACE,LWORK,IERROR)

      FVAL1 = 0.5*DOT_R(NV,NMAX,F,DV) + DOT_R(NV,NMAX,DV,V)
      WRITE(*,*)FVAL1,FVAL0,SQRT(R2SUM),TEST_LIM
      IF(2.0*ABS(FVAL0-FVAL1).LE.
     &            TOLER1*(ABS(FVAL0)+ABS(FVAL1)+EPS_LOC)) GOTO 120
      FVAL0 = FVAL1
        IF(SQRT(R2SUM).LE.TEST_LIM) GOTO 120
      ENDDO
C
  120 CONTINUE
      WRITE(*,*)'R2sum ',SQRT(R2SUM),TEST_LIM
C
      CALL SCALE_SHIFTS(NV,NMAX,SK,DV,IERROR)
      IF(IERROR.GT.0) THEN
        CALL ERRWRT(0,'In CGSOLVE after SCALE_SHIFTS')
        RETURN
      ENDIF
      CALL SCALE1_INV(NV,NMAX,F,A,SK,V,WORKSPACE,LWORK,IERROR)
      RETURN
      END
C
      SUBROUTINE CGSOL_R(NV,NMAX,NCYCL,TOLER,A,V,DV,R,SK,P,F,WORKSPACE,
     +                  LWORK,SCALE1,MATMUL1,SCALE_SHIFTS,SCALE1_INV,
     +                  IERROR)
c
      IMPLICIT NONE
c
      INTEGER   NV,NMAX,IERROR,I,NCYCL,ITER
c
      REAL      TOLER1,A(*),V(*),DV(*),SK(*),P(*),F(*),R(*),
     +          WORKSPACE(*),LWORK(*),TOLER,RHO(0:NCYCL),TEST_LIM,ALPHA,
     +          BETA,VNORM,DOT_R
c
      PARAMETER (TOLER1 = 1.0E-8)
c
      EXTERNAL  SCALE1,MATMUL1,SCALE_SHIFTS,SCALE1_INV,AVECPVEC_R
c
      IERROR = 0
      IF(TOLER.LE.0.0) TOLER = TOLER1 
    
c-----preconditioning
      CALL SCALE1(NV,NMAX,F,A,SK,V,WORKSPACE,LWORK,IERROR)

c-----calculation of the 2-norm for the gradient
      VNORM = SQRT(DOT_R(NV,NMAX,V,V))
      TEST_LIM = TOLER*VNORM
c-----set initial guess for solution and calculate the gradient of the object
c-----function at the current solution

      DO I = 1,NV
         DV(I)  = 0.0
         R(I)   = V(I)
      ENDDO
c
      RHO(0) = DOT_R(NV,NMAX,R,R)
      IF(RHO(0).LT.TOLER1) GOTO 120
C
      DO ITER = 1,NCYCL
         IF (ITER.EQ.1) THEN
            DO I = 1,NV
               P(I) = R(I)
            ENDDO
         ELSE
            BETA = RHO(ITER-1)/RHO(ITER-2)
            CALL AVECPVEC_R(NV,NMAX,BETA,P,R,P,IERROR)
         ENDIF 
         
         CALL MATMUL1(NV,NMAX,P,A,F,WORKSPACE,LWORK,IERROR)   
         ALPHA = RHO(ITER-1)/DOT_R(NV,NMAX,P,F)
         CALL AVECPVEC_R(NV,NMAX,ALPHA,P,DV,DV,IERROR)
         CALL AVECPVEC_R(NV,NMAX,-ALPHA,F,R,R,IERROR)
         RHO(ITER) = DOT_R(NV,NMAX,R,R)
         WRITE(*,*) SQRT(RHO(ITER)),TEST_LIM   
         IF (SQRT(RHO(ITER)).LT.TEST_LIM) THEN
            print*, ' Convergence reached'
            print*, SQRT(DOT_R(NV,NMAX,DV,DV))
            GOTO 120
         ENDIF
      ENDDO 
c
  120 CONTINUE
      
c
      CALL SCALE_SHIFTS(NV,NMAX,SK,DV,IERROR)
c
      CALL SCALE1_INV(NV,NMAX,F,A,SK,V,WORKSPACE,LWORK,IERROR)
c
      RETURN
      END
C
C---find initil solution using block diagonal matrix
C

      SUBROUTINE CGSOL_RM(NV,NMAX,NCYCL,TOLER,A,V,DV,R,SK,P,F,WORKSPACE,
     +                  LWORK,SCALE1,MATMUL1,SCALE_SHIFTS,SCALE1_INV,
     +                  IERROR,GAMMA_FLAG,GAMMA,STEP_FLAG,STEP)
c
      IMPLICIT NONE
      INTEGER MCYCLE
      PARAMETER (MCYCLE = 100 0)
      INCLUDE 'atom_com.fh'
      INCLUDE 'pls_incl.fh'
      INTEGER   NV,NMAX,IERROR,I,NCYCL,ITER,GAMMA_CYC,MAX_GAMMA_CYC
c
      REAL      TOLER1,A(*),V(*),DV(*),SK(*),P(*),F(*),R(*),
     +          WORKSPACE(*),LWORK(*),TOLER,RHO(0:MCYCLE),TEST_LIM,
     &          ALPHA,
     +          BETA,VNORM,DOT_R,GAMMA,DV_SAVE(QQV),GAMMA_SAVE,STEP
c
      REAL FVAL0,FVAL1
      PARAMETER (TOLER1 = 0.5E-7)
      REAL EPS_LOC
      PARAMETER (EPS_LOC = 0.5E-7)
c
      LOGICAL   GAMMA_FLAG,CONVER_FLAG,STEP_FLAG
c
      EXTERNAL  SCALE1,MATMUL1,SCALE_SHIFTS,SCALE1_INV,AVECPVEC_R

C---Consider finding initial solution (e.g. Using diagonal terms only). But 
C---this should be outside of this subroutine.
C
c-----initializations
cd      NCYCL = 200
      IERROR = 0
      IF(TOLER.LE.0.0) TOLER = TOLER1
      CONVER_FLAG = .FALSE.
cd      GAMMA_FLAG  = .FALSE.
cd      STEP_FLAG   = .FALSE.
      IF (.NOT.GAMMA_FLAG) THEN
         GAMMA = 0.0
      ENDIF 
      DO I = 1,NV
         DV(I)  = 0.0
         DV_SAVE(I) = 0.0
      ENDDO 
    
c-----preconditioning
      CALL SCALE1(NV,NMAX,F,A,SK,V,WORKSPACE,LWORK,IERROR)
cd      DO   I=1,100
cd        WRITE(*,*)SK(I)
cd      ENDDO
cd      DO   I=1,100
cd         WRITE(*,*)A(I)
cd         WRITE(*,*)V(I)
cd      ENDDO
c-----calculation of the 2-norm for the gradient
      VNORM = SQRT(DOT_R(NV,NMAX,V,V))
      TEST_LIM = TOLER*VNORM
c-----set initial guess for solution and calculate the gradient of the object
c-----function at the current solution
      MAX_GAMMA_CYC = 500 
      IF (.NOT.STEP_FLAG) THEN
         STEP = 0.050
      ENDIF 
      DO GAMMA_CYC = 1,MAX_GAMMA_CYC
         IF (GAMMA_CYC.NE.1) GAMMA = GAMMA + STEP
         write(*,*) 'Trying gamma equal ', GAMMA
         
 99      CONTINUE
 89      CONTINUE
C
C---Initial lues. DV is the solution vector.
C
         CALL MATMUL1(NV,NMAX,DV,A,R,WORKSPACE,LWORK,IERROR)
         DO I = 1,NV
            R(I) = V(I) - R(I)
         ENDDO
         CALL AVECPVEC_R(NV,NMAX,-GAMMA,DV,R,R,IERROR)
c
         RHO(0) = DOT_R(NV,NMAX,R,R)

         IF (RHO(0).LT.TOLER1) GOTO 120
C
         FVAL0 = 0.0
         DO ITER = 1,NCYCL
            IF (ITER.EQ.1) THEN
               DO I = 1,NV
                  P(I) = R(I)
               ENDDO
               BETA = 0.0
            ELSE
               BETA = RHO(ITER-1)/RHO(ITER-2)
               CALL AVECPVEC_R(NV,NMAX,BETA,P,R,P,IERROR)
            ENDIF 
c
            CALL MATMUL1(NV,NMAX,P,A,F,WORKSPACE,LWORK,IERROR)
            CALL AVECPVEC_R(NV,NMAX,GAMMA,P,F,F,IERROR)
c
            ALPHA = RHO(ITER-1)/DOT_R(NV,NMAX,P,F)
            CALL AVECPVEC_R(NV,NMAX,ALPHA,P,DV,DV,IERROR)
            CALL AVECPVEC_R(NV,NMAX,-ALPHA,F,R,R,IERROR)
C
C--Value of the functional
cd            CALL MATMUL1(NV,NMAX,DV,A,F,WORKSPACE,LWORK,IERROR)
cd            FVAL1 = 0.5*(DOT_R(NV,NMAX,DV,DV)*GAMMA + 
cd     &             DOT_R(NV,NMAX,DV,F))- DOT_R(NV,NMAX,V,DV)
            RHO(ITER) = DOT_R(NV,NMAX,R,R)
cd            IF (SQRT(RHO(ITER)).GT.(5.0*SQRT(RHO(ITER-1)))) THEN
cd            IF(FVAL1.GT.FVAL0) GOTO 89
cd            WRITE(*,*)ALPHA,BETA,SQRT(RHO(ITER)),TEST_LIM
cd            WRITE(*,*)ITER,ALPHA,BETA,RHO(ITER),TEST_LIM

            IF(SQRT(RHO(ITER)).GT. 2.0*SQRT(RHO(ITER-1))) THEN
               WRITE(*,*) 'Not converging with gamma equal ', GAMMA
               STEP = STEP*1.1
cd               stop
               GOTO 110
            ENDIF
cd            STOP

             IF (SQRT(RHO(ITER)).LT.TEST_LIM) THEN
cd     &            2.0*ABS(FVAL0-FVAL1).LE.
cd     &            TOLER1*(ABS(FVAL0)+ABS(FVAL1)+EPS_LOC)) THEN
               IF (GAMMA_FLAG.EQV..FALSE.) THEN
                  write(*,*) 'Convergence reached with no gamma cycles '
                  GOTO 120
               ELSEIF (CONVER_FLAG) THEN
                  write(*,*) SQRT(DOT_R(NV,NMAX,DV,DV))
                  write(*,*) 'Convergence reached with gamma equal ',
     +                        GAMMA
                  STEP = STEP*1.01
                  GOTO 120
               ELSE
                  CONVER_FLAG = .TRUE.
                  STEP_FLAG = .TRUE.
                  GAMMA_SAVE = GAMMA
                  DO I = 1,NV
                     DV_SAVE(I) = DV(I)
                  ENDDO
                  GAMMA = AMAX1(0.0,GAMMA - STEP/5.0)
                  STEP = STEP/1.1
                  IF (GAMMA.LT.0.0) stop
                  write(*,*) 'Gamma decreased to ', GAMMA
cd                  GOTO 99
                  GOTO 120
               ENDIF
            ENDIF
cd           FVAL0 = FVAL1
         ENDDO
 110     CONTINUE
         GAMMA_FLAG = .TRUE.
         IF (.NOT.CONVER_FLAG) THEN
            DO I = 1,NV
               DV(I)  = 0.0
            ENDDO
         ELSE
            DO I = 1,NV
               DV(I) = DV_SAVE(I)
            ENDDO
            GAMMA = GAMMA_SAVE
            write(*,*) 'Back to gamma equal ', GAMMA
            GOTO 89
         ENDIF
      ENDDO 
c
  120 CONTINUE
c
      DO  I=1,NV
         DV(I) = DV(I)* (1.0+GAMMA)
      ENDDO
c
      CALL SCALE_SHIFTS(NV,NMAX,SK,DV,IERROR)
cd      DO   I=1,100
cd        WRITE(*,*)SK(I),V(I),DV(I)
cd      ENDDO
c
      CALL SCALE1_INV(NV,NMAX,F,A,SK,V,WORKSPACE,LWORK,IERROR)
c
cd      stop
      RETURN
      END
C
      SUBROUTINE DEIGEN_FILTER_INVERSE_R(RANK,D2F,NPARAM,NMAXPAR,
     &             D_INV,WORKSPACE,NWORKSPACE)
C
C---Solve linear equations using eigenvalue filtering (psuedo inversion)
C
      IMPLICIT NONE
      INTEGER NPARAM,NMAXPAR,NWORKSPACE
      REAL*8 D2F(NMAXPAR,NMAXPAR),D_INV(NMAXPAR,NMAXPAR)
      REAL*8 WORKSPACE(*)
      REAL*8 RANK
C
      INTEGER LWORK,INFO,I,J,K
      LOGICAL ERROR
      REAL*8  SFMIN,DLAMCH
      EXTERNAL SLAMCH,SSYEV_MO,MAT2VECT,MAT2VEC
      
C     
C--Find machine parameters
      SFMIN = DLAMCH('S')
C
C---find eigenvalues and eigenvectors of right hand side matrix
      LWORK = NWORKSPACE - NMAXPAR
      CALL DSYEV_MO('V','U',NPARAM,D2F,NMAXPAR,WORKSPACE(1),
     &    WORKSPACE(NMAXPAR+1),LWORK,INFO)
C
C---Now solve equations. 0 eigenvalues are not used in the inversion
C
      DO   I=1,NPARAM
         IF(WORKSPACE(I).GT.
     &     ABS(WORKSPACE(NPARAM))*RANK) THEN
           WORKSPACE(I)        = 1.0D0/WORKSPACE(I)
         ELSE
           WORKSPACE(I)        = 1.0D0/ABS(WORKSPACE(NPARAM))
         ENDIF
      ENDDO
C
C--Now psuedoinverse
C
      DO   I=1,NPARAM
        DO   K=1,NPARAM
           D_INV(I,K) = 0.0
           DO  J=1,NPARAM
              D_INV(I,K) = D_INV(I,K) + D2F(I,J)*D2F(K,J)*WORKSPACE(J)
           ENDDO
        ENDDO
      ENDDO

cd      CALL MAT2VEC(NMAXPAR,NPARAM,D2F,WORKSPACE(NPARAM+1),SHIFTS,ERROR)
      RETURN
      END
C
      SUBROUTINE EIGEN_FILTER_R(RANK,D2F,NPARAM,NMAXPAR,DF,SHIFTS,
     &             WORKSPACE,NWORKSPACE)
C
C---Solve linear equations using eigenvalue filtering (psuedo inversion)
C
      IMPLICIT NONE
      INTEGER NPARAM,NMAXPAR,NWORKSPACE
      REAL D2F(NMAXPAR,NMAXPAR),DF(NMAXPAR),SHIFTS(NMAXPAR)
      REAL WORKSPACE(*)
      REAL RANK
C
      INTEGER LWORK,INFO,I
      LOGICAL ERROR
      REAL  SFMIN,SLAMCH
      EXTERNAL SLAMCH,SSYEV_MO,MAT2VECT,MAT2VEC
      
C     
C--Find machine parameters
      SFMIN = SLAMCH('S')
C
C---find eigenvalues and eigenvectors of right hand side matrix
      LWORK = NWORKSPACE - NMAXPAR
      CALL SSYEV_MO('V','U',NPARAM,D2F,NMAXPAR,WORKSPACE(1),
     &    WORKSPACE(NMAXPAR+1),LWORK,INFO)
       if(info.gt.0) then
          write(*,*)'Problem in eigen_filter_r',info
          stop
       endif
C
C---Now solve equations. 0 eigenvalues are not used in the inversion
C
      CALL MAT2VECT(NMAXPAR,NPARAM,D2F,DF,WORKSPACE(NPARAM+1),ERROR)
      DO   I=1,NPARAM
         IF(WORKSPACE(I).GT.
     &     AMAX1(ABS(WORKSPACE(NPARAM))*RANK,SFMIN)) THEN
           WORKSPACE(NPARAM+I) = WORKSPACE(NPARAM+I)/WORKSPACE(I)
         ELSE
           WORKSPACE(NPARAM+I) = 0.0
         ENDIF
      ENDDO
      CALL MAT2VEC(NMAXPAR,NPARAM,D2F,WORKSPACE(NPARAM+1),SHIFTS,ERROR)
      RETURN
      END
C
      SUBROUTINE DEIGEN_FILTER_R(RANK,D2F,NPARAM,NMAXPAR,DF,SHIFTS,
     &             WORKSPACE,NWORKSPACE)
C
C---Solve linear equations using eigenvalue filtering (psuedo inversion)
C---Double precision version
C
      IMPLICIT NONE
      INTEGER NPARAM,NMAXPAR,NWORKSPACE
      REAL*8 D2F(NMAXPAR,NMAXPAR),DF(NMAXPAR),SHIFTS(NMAXPAR)
      REAL*8 WORKSPACE(*)
      REAL*8 RANK
C
      INTEGER LWORK,INFO,I,J
      LOGICAL ERROR
      REAL*8  SFMIN,DLAMCH
      EXTERNAL DLAMCH,DSYEV_MO
cd,MAT2VECT,MAT2VEC
      
C     
C--Find machine parameters
      SFMIN = DLAMCH('S')
C
C---find eigenvalues and eigenvectors of right hand side matrix
      LWORK = NWORKSPACE - NMAXPAR
      CALL DSYEV_MO('V','U',NPARAM,D2F,NMAXPAR,WORKSPACE(1),
     &    WORKSPACE(NMAXPAR+1),LWORK,INFO)
C
C---Now solve equations. 0 eigenvalues are not used in the inversion
C
cd      CALL MAT2VECT(NMAXPAR,NPARAM,D2F,DF,WORKSPACE(NPARAM+1),ERROR)
      DO    I=1,NPARAM
         WORKSPACE(NPARAM+I) = 0.0D0
         DO   J=1,NPARAM
           WORKSPACE(NPARAM+I) = WORKSPACE(NPARAM+I) + D2F(J,I)*DF(J)
         ENDDO
      ENDDO
C
      DO   I=1,NPARAM
         IF(WORKSPACE(I).GT.
     &     DMAX1(DABS(WORKSPACE(NPARAM))*RANK,SFMIN)) THEN
           WORKSPACE(NPARAM+I) = WORKSPACE(NPARAM+I)/WORKSPACE(I)
         ELSE
           WORKSPACE(NPARAM+I) = 0.0D0
         ENDIF
      ENDDO
cd      CALL MAT2VEC(NMAXPAR,NPARAM,D2F,WORKSPACE(NPARAM+1),SHIFTS,ERROR)
      DO   I=1,NPARAM
        SHIFTS(I) = 0.0D0
        DO  J=1,NPARAM
          SHIFTS(I) = SHIFTS(I) + D2F(I,J)*WORKSPACE(NPARAM+J)
        ENDDO
      ENDDO
      RETURN
      END
C
      SUBROUTINE MATR2EUL(R,AL1,AL2,AL3,ERROR)
C     ========================
C
C---- This s/r works out Euler rotation angles from a given rotation matrix.
C
      REAL AL1,AL2,AL3
      REAL R(3,3)
      LOGICAL ERROR
C
      REAL EPS_LOC
      REAL RINV(3,3),RTRANS(3,3)
      DATA EPS_LOC/1.0E-6/
C
C---First check if it is rotation matrix
      ERROR = .FALSE.
      CALL MATINV3_R(R,RINV)
      CALL MATTRANS(3,3,R,RTRANS,ERROR)

      ADIFF = 0.0
      DO   I=1,3
         DO   J=1,3
           ADIFF = ADIFF + (RINV(J,I)-RTRANS(J,I))**2
         ENDDO
      ENDDO
      IF(SQRT(ABS(ADIFF)).GT.EPS_LOC) THEN
         ERROR = .TRUE.
         CALL ERRWRT(0,'Matrix is not rotation matrix')
         RETURN
      ENDIF
      IF(ABS(DET3(R)-1.0).GT.EPS_LOC) THEN
         ERROR = .TRUE.
         CALL ERRWRT(0,'Matrix is not rotationa matrix')
      ENDIF
C
      IF(SQRT(ABS(R(3,1)**2+R(3,2)**2)).LT.EPS_LOC) THEN
C
C---AL2 = 0.0 or pi. Either AL1+AL3 or AL1 + AL3 can be defined
        IF(R(3,3).GT.1.0-EPS_LOC) THEN
          AL2 = 0.0
        ELSE
          AL2 = 4.0*ATAN2(1.0,1.0)
        ENDIF
        AL3 = 0.0
        AL1 = ATAN2(R(2,1),R(1,1))
      ELSE
        AL3 = ATAN2(R(3,2),-R(3,1))
        IF(ABS(R(3,1)).GT.EPS_LOC) THEN
           AL2 = ATAN2(-R(3,1)/COS(AL3),R(3,3))
        ELSE
           AL2 = ATAN2(R(3,2)/SIN(AL3),R(3,3))
        ENDIF
        AL1 = ATAN2(R(2,3),R(1,3))
      ENDIF

      RETURN
      END
C
      SUBROUTINE MATR2EUL_UNSAFE(R,AL1,AL2,AL3)
C     ========================
C
C---- This s/r works out Euler rotation angles from a given rotation matrix.
C
      REAL AL1,AL2,AL3
      REAL R(3,3)
C
      REAL EPS_LOC
      DATA EPS_LOC/1.0E-8/
C
      IF(SQRT(ABS(R(3,1)**2+R(3,2)**2)).LT.EPS_LOC) THEN
C
C---AL2 = 0.0 or pi. Either AL1+AL3 or AL1 - AL3 can be defined
        IF(R(3,3).GT.0.5) THEN
          AL2 = 0.0
        ELSE
          AL2 = 4.0*ATAN2(1.0,1.0)
        ENDIF
        AL3 = 0.0
        AL1 = ATAN2(R(2,1),R(1,1))
      ELSE         
        AL3 = ATAN2(R(3,2),-R(3,1))
        IF(ABS(R(3,1)).GT.EPS_LOC) THEN
           AL2 = ATAN2(-R(3,1)/COS(AL3),R(3,3))
        ELSE
           AL2 = ATAN2(R(3,2)/SIN(AL3),R(3,3))
        ENDIF
        AL1 = ATAN2(R(2,3),R(1,3))
      ENDIF

      RETURN
      END
c
c---Interface to Andrey's subroutine
      subroutine seig3dim(amat,eigenv,ierror)
      implicit none
c
c--input and output
      real amat(3,3)
c
c--output
      integer i,j
      integer ierror
      real eigenv(3)
c
c---locals
      real eigenvec(3,3)
      logical error
c
c--body
      ierror = 0
      call Z4_eigv3sym(amat,eigenv,eigenvec,error)
      if(error) then
         ierror = 1
         return
      endif
      do i=1,3
         do j=1,3
            amat(i,j) = eigenvec(j,i)
         enddo
      enddo
      return
      end
