
module automorphism

   real, private, parameter :: zero = 1.0D-5

   !--- for testing error messages:
   !--- zero = 1.0D-8

contains

   !==================================================================
   !
   !
   !              all symmetry operations are real 4x4 matrixes
   !
   !              input:
   !
   !   no         the number of symmetry operations
   !   r44oo_o    symmetry operations
   !   r44oo_t    twin operations ( r44oo_t(1:3,4) = 0 )
   !   imsg       0: no print,
   !              1: error messages to stdout
   !              2: 1 + error messages to errout + some matrices
   !
   !              output:
   !
   !   r44oo_t    if there exists a space group automorphism operation
   !              with the same rotation as in input twin operation
   !              then its translation is returned in r44oo_t(1:3,4)
   !              
   !   ierr       0: required automorphism operation exists and
   !                 its translation is returned in r44oo_t(1:3,4)
   !              1: automorphism is impossible
   !              2: something wrong possibly because roundoff errors
   !                 (try to increase the value of variable "zero")
   !
   !
   !

   subroutine t4twinop( r44oo_o, r44oo_t, imsg, ierr )

      implicit  none

      real,      intent(in)    :: r44oo_o(:,:,:)
      real,      intent(inout) :: r44oo_t(4,4)
      integer,   intent(in)    :: imsg
      integer,   intent(out)   :: ierr

      integer    no, io, jo
      real       r44oo_to(4,4)
      real       r44oo_ot(4,4)
      real       r44oo_x(4,4)
      real       r3o(3)

      integer,   parameter :: mk = 8
      integer    i, j, k
      real       r33op(3,mk)

      real       r44op(4,4)
      real       r44po(4,4)
      real       r44ee(4,4)

      integer,   parameter :: mp = 24
      integer    np, ip, jp
      real       r44pp_p(4,4,mp)
      real       r44pp_pt(4,4)
      real       r44pp_tp(4,4)

      real       r44pp_x(4,4)
      real       r44pp_y(4,4)
      real       r44pp_t(4,4)
      real       r4p(4)
      real       r1

      integer,   parameter :: mc = mp* 3
      integer    nc
      real       s_rc(4,mc)

      real       del
      real       non_zero
      logical    found

      character  tsubr*16

      !---------------------------------------------------------------
      no = size(r44oo_o(1,1,:))
      tsubr = 't4twinop'

      call t4t_wrops( 'r44oo_o:', no, r44oo_o, imsg )

      !--- identity matrix

      r44ee = 0
      do i = 1,4
         r44ee(i,i) = 1
      enddo

      !--- defining transformation to a primitive cell

      r33op = 0
      r33op(1:3,1:3) = r44ee(1:3,1:3)
      k = 3
      do io = 1,no
         del = maxval( abs( r44oo_o(1:3,1:3,io) - r44ee(1:3,1:3) ) )
         if( del .lt. zero )then
            k = k + 1
            if( k .gt. mk )then

               call t4t_errmsg( imsg, tsubr, 'a' )
               ierr = 2
               return

            endif
            r33op(1:3,k) = r44oo_o(1:3,4,io)
         endif
      enddo

      r44op = r44ee
      non_zero = 1.0D-02
      call t4t_reduce( 3, r44op(1:3,1:3), k, r33op, non_zero, imsg, ierr )
      if( ierr .gt. 0 )then

         call t4t_errmsg( imsg, tsubr, 'b' )
         return

      endif

      if( abs( r33op(3,3) ) .le. zero )then

         call t4t_errmsg( imsg, tsubr, 'c' )
         ierr = 2
         return

      endif

      r44po = r44op
      call t4t_invert_low( 4, r44po )

      call t4t_wrops( 'r44po:', 1, r44po, imsg )
      call t4t_wrops( 'r44op:', 1, r44op, imsg )

      !--- transforming the input group to the primitive cell

      np = 0
      do io = 1,no
         r44pp_x = matmul( matmul( r44po, r44oo_o(1:4,1:4,io) ), r44op )
         ip = 0
         found = .false.
         do while( .not. found .and. ip .lt. np )
            ip = ip + 1
            r44pp_y = r44pp_p(1:4,1:4,ip) - r44pp_x
            r4p = r44pp_y(1:4,4)
            do i = 1,4
               r44pp_y(1:4,4) = r4p(i) - nint( r4p(i) )
            enddo
            found = maxval( abs( r44pp_y(1:4,1:3) ) ) .lt. zero
         enddo
         if( .not. found )then
            np = np + 1
            if( np .gt. mp )then

               call t4t_errmsg( imsg, tsubr, 'e' )
               ierr = 2
               return

            endif
            r44pp_p(1:4,1:4,np) = r44pp_x
         endif
      enddo

      call t4t_wrops( 'r44pp_p:', np, r44pp_p, imsg )

      !--- transforming the input twin operation to the primitive cell

      r44pp_t = matmul( matmul( r44po, r44oo_t ), r44op )

      call t4t_wrops( 'r44pp_t:', 1, r44pp_t, imsg )

      !--- collecting matrix characterising automorphism by twin operation
      !--- exit with ierr=1 if twin operation is not automorphism operation

      nc = 0
      do ip = 1,np
         r44pp_pt = matmul( r44pp_p(1:4,1:4,ip), r44pp_t )

         found = .false.
         do jp = 1,np
            r44pp_tp = matmul( r44pp_t, r44pp_p(1:4,1:4,jp) )

            r44pp_x = r44pp_pt - r44pp_tp
            if( maxval( abs( r44pp_x(1:3,1:3) ) ) .lt. zero )then
               if( found )then

                  call t4t_errmsg( imsg, tsubr, 'f' )
                  ierr = 1
                  return

               endif
               found = .true.
               s_rc(4,nc+1:nc+3) = r44pp_x(1:3,4)
            endif
         enddo
         if( .not. found )then

            call t4t_errmsg( imsg, tsubr, 'g' )
            ierr = 1
            return

         endif

         r44pp_x = r44ee - r44pp_p(1:4,1:4,ip)
         do i = 1,3
            do j = 1,3
               s_rc(i,nc+j) = r44pp_x(j,i)
            enddo
         enddo
         nc = nc + 3
      enddo

      !--- transforming the matrix and finding translation for r44pp_t

      non_zero = 0.9D+00
      call t4t_reduce( 4, r44pp_x, nc, s_rc, non_zero, imsg, ierr )
      if( ierr .gt. 0 )then

         call t4t_errmsg( imsg, tsubr, 'h' )
         return

      endif

      do i = 1,4
         r1 = r44pp_x(4,i)
         r44pp_x(4,i) = r1 - nint( r1 )
      enddo

      if( abs( r44pp_x(4,4) ) .gt. zero )then

         call t4t_errmsg( imsg, tsubr, 'i' )
         ierr = 2
         return

      endif

      r44pp_x(4,4) = 1
      call t4t_invert_low( 4, r44pp_x )
      r44pp_t(1:3,4) = - r44pp_x(4,1:3)

      !--- transforming the automorphism (twin) operation to the original cell

      r44oo_t = matmul( r44op, matmul( r44pp_t, r44po ) )

      call t4t_wrops( 'r44oo_t:', 1, r44oo_t, imsg )

      !--- self-test: transformation of sg by twin operation in original setting

      do io = 1,no
         r44oo_ot = matmul( r44oo_o(1:4,1:4,io), r44oo_t )
         found = .false.
         do jo = 1,no
            r44oo_to = matmul( r44oo_t, r44oo_o(1:4,1:4,jo) )
            r44oo_x = r44oo_to - r44oo_ot
            r3o = r44oo_x(1:3,4)
            do i = 1,3
               r44oo_x(i,4) = r3o(i) - nint( r3o(i) )
            enddo
            if( maxval( abs( r44oo_x ) ) .lt. zero )then
               if( found )then

                  call t4t_errmsg( imsg, tsubr, 'j' )
                  ierr = 2
                  return

               endif
               found = .true.
            endif
         enddo
         if( .not. found )then

            call t4t_errmsg( imsg, tsubr, 'k' )
            ierr = 2
            return

         endif
      enddo

      ierr = 0
      return

   end subroutine t4twinop

   !==================================================================

   subroutine t4t_reduce( nr, t_rr, nc, t_rc, non_zero, imsg, ierr )

      implicit   none

      integer    nr, ir, jr
      real       t_rr(nr,nr)

      integer    nc, ic, jc
      real       t_rc(nr,nc)

      real       t
      integer    i, cou
      real       non_zero

      integer    imsg, ierr
      character  tsubr*16

      !---------------------------------------------------------------
      tsubr = 't4t_reduce'

      jc = 1
      do ir = 1,nr
         do ic = nc,jc+1,-1
            cou = 0
            do while( abs( t_rc(ir,ic) ) .gt. non_zero )
               cou = cou + 1
               if( cou .gt. 10 )then

                  call t4t_errmsg( imsg, tsubr, 'a' )
                  ierr = 2
                  return

               endif
               i = nint( t_rc(ir,ic-1)/ t_rc(ir,ic) )
               do jr = ir,nr
                  t = t_rc(jr,ic)
                  t_rc(jr,ic) = i* t - t_rc(jr,ic-1)
                  t_rc(jr,ic-1) = t
               enddo
            enddo
            if( abs( t_rc(ir,ic) ) .gt. zero )then

               call t4t_errmsg( imsg, tsubr, 'b' )
               ierr = 2
               return

            endif
         enddo
         if( abs( t_rc(ir,jc) ) .gt. zero )then
            t_rr(1:nr,ir) = t_rc(1:nr,jc)
            jc = jc + 1
         else
            t_rr(1:nr,ir) = 0
            t_rr(ir,ir) = 1
         endif
      enddo

      ierr = 0
      return

   end subroutine t4t_reduce

   !==================================================================

   subroutine t4t_invert_low( nr, t_rr )

      implicit   none

      real       t_rr(nr,nr)
      real       t
      integer    nr, ir, jr, kr

      !---------------------------------------------------------------
      do jr = 1,nr
         t_rr(jr,jr) = 1/ t_rr(jr,jr)
         do ir = 1,jr-1
            t = 0
            do kr = ir,jr-1
               t = t - t_rr(jr,kr)* t_rr(kr,ir)
            enddo
            t_rr(jr,ir) = t* t_rr(jr,jr)
         enddo
      enddo

   end subroutine t4t_invert_low

   !==================================================================

   subroutine t4t_wrops( t, no, x_o, imsg )

      implicit   none

      character  t*(*)
      integer    imsg

      integer    no, io
      real       x_o(4,4,no)

      integer    i, j

      !---------------------------------------------------------------
      if( imsg .lt. 3 ) return

      write(*,'(a)')
      write(*,'(a)') trim(t)
      write(*,'(a)')
      do io = 1,no
         do i = 1,4
            write(*,'(3f8.1,f12.4)') ( x_o(i,j,io), j = 1,4 )
         enddo
         write(*,'(a)')
      enddo
      write(*,'(a)') '-------------------------------------------'

   end subroutine t4t_wrops

   !==================================================================

   subroutine t4t_errmsg( imsg, tsubr, t )

      implicit   none

      integer    imsg
      character  tsubr*(*)
      character  t*(*)

      !---------------------------------------------------------------
      if( imsg .ge. 1 ) write(*,'(a)') 'subroutine '//tsubr//': exit '//t
      if( imsg .ge. 2 ) write(0,'(a)') 'subroutine '//tsubr//': exit '//t

   end subroutine t4t_errmsg

   !==================================================================

end module automorphism

