module gibbs_gm
  use linalgebra_f90
  implicit none

  integer, private :: nburn=1000

contains
  subroutine gibbs_sampler(ndiag,vn,atom_ref_flag,atom_fix_flag,am,nrest_per_atom,rest_per_atom,rest_per_atom_pos,  &
       rest_per_atom_dist,nburn_in)
    !
    !  Sampling from gauss-markov field using gibbs sampler. 
    !  Precision matrix defining the field is assumed to be sparse
    !  This subroutine first should be run with nburn_in value set to some positive value
    !  or negative value. If the value is negative then it is set to 100
    !  nburn_in is the number of dry runs.

    !
    !  It is assumed that conditioners have already been applied and block diagoanl terms are 
    !  identity
    !

    integer ndiag
    integer, intent(in) :: atom_ref_flag(:)
    integer, intent(in) :: atom_fix_flag(:)
    integer, intent(in) :: nrest_per_atom(:)
    integer, intent(in) :: rest_per_atom_pos(:)
    integer, intent(in) :: rest_per_atom(:)
    integer, intent(in) :: rest_per_atom_dist(:)
    real, intent(inout) :: vn(:)
    real, intent(in) :: am(:)
    integer, optional :: nburn_in
    
    !   
    !  locals

    integer i,j,l,l1
    integer nv,ia,iaa,ia1,ia2,ic,ja,jaa,jad,ja1,ja2
    integer nburn
    integer n_atom
    integer npos,npos1,nd_ia
    real rand_gauss(3)
    real, allocatable :: vo(:)
    !
    !  More locals
    real gamma
    real amm(3,3)
    !
    !  Diagonal terms are identity
    !
   
    n_atom = size(atom_ref_flag)

    if(present(nburn_in))then
       nburn=nburn_in
       if(nburn_in.lt.0) nburn = 1000
    else
       nburn = 0
    endif

    nv = size(vn)
    allocate(vo(nv))
    if(nburn.gt.0) then
       vn(1:nv) = 0.0
    endif
    gamma = 0.0
    !
    !   Should we use coupled runs to determine nburn
    !
    do ic=1,nburn+1
       vo = vn
       vn(1:nv) = 0.0
       do iaa=1,n_atom
          if(atom_ref_flag(iaa).gt.0) then
             !
             !  if atom_fix_flag is not 1 then this atom is 
             !      either free: atom_fix_flag = -1
             !      or  fixed:   atom_fix_flag = 1
             !
             if(atom_fix_flag(iaa).eq.0) then
                ia = atom_ref_flag(iaa)/10
                ia1 = 3*ia-2
                ia2 = 3*ia
                call random_vector_gauss(3,vn(ia1:ia2))
                vn(ia1:ia2) = vn(ia1:ia2)/(1.0+gamma)
                nd_ia = nrest_per_atom(iaa)
                npos = rest_per_atom_pos(iaa)
                npos1 = npos+nd_ia-1
                if(nd_ia.gt.0) then
                   do jad=npos,npos1
                      jaa = rest_per_atom(jad)
                      ja = atom_ref_flag(jaa)/10
                      l1 = rest_per_atom_dist(jad)
                      l = ndiag + 9*(abs(l1)-1)+1
                      do i=1,3
                         do j=1,3
                            amm(i,j) = am(l)
                            l = l + 1
                         enddo
                      enddo
                      if(l1.lt.0) amm = transpose(amm)
                      ! amm = 0.0
                      if(ja.lt.ia) then
                         ja1 = 3*ja-2
                         ja2 = 3*ja
                         vn(ia1:ia2) = vn(ia1:ia2) - matmul(amm(1:3,1:3),vn(ja1:ja2))
                      elseif(ja.gt.ia) then
                         ja1 = 3*ja-2
                         ja2 = 3*ja
                         vn(ia1:ia2) = vn(ia1:ia2) - matmul(amm(1:3,1:3),vo(ja1:ja2))
                      endif
                   enddo
                endif                
             endif
          endif
       enddo
       !       if(ic.eq.nburn) then
       !          write(*,*)vn
       !          stop
       !       endif
    enddo
    !
    !   Consider free atoms also. They need random numbers only once
    do iaa=1,n_atom
       if(atom_ref_flag(iaa).eq.1) then
          if(atom_fix_flag(iaa).eq.-1) then
             ia = atom_ref_flag(iaa)/10
             ia1 = 3*ia1-2
             ia2 = ia1 + 2
             call random_vector_gauss(3,vn(ia1:ia2))
          endif
       endif
    enddo

    deallocate(vo)

  end subroutine gibbs_sampler

  subroutine simple_atom_conditioner(ndiag,am,xcond,atom_ref_flag,atom_fix_flag,nrest_per_atom,rest_per_atom, &
    rest_per_atom_pos,rest_per_atom_dist)

    !
    !   Define block conditioners for linear equation solvers and
    !   for gibbs samplers. Conditioners are a set of 3x3 matrices
    !   that are inversion of square root of on diagonal atom matrices
    !

    integer ndiag
    integer, intent(in) :: atom_ref_flag(:)
    integer, intent(in) :: atom_fix_flag(:)
    integer, intent(in) :: nrest_per_atom(:)
    integer, intent(in) :: rest_per_atom_pos(:)
    integer, intent(in) :: rest_per_atom(:)
    integer, intent(in) :: rest_per_atom_dist(:)
    !
    real, intent(inout) :: am(:)
    real, intent(out) :: xcond(:,:,:)
    !
    !   locals
    integer i,j,ia,iaa,ja,jaa,jad,id,lm,icc,icc1,io,it,io1,it1
    integer l,l1,nd_ia
    integer n_atom,nm,ndist
    integer npos,npos1
    !
    integer, allocatable :: ref_to_cond(:)
    real(kind=8) :: xloc(3,3),xtemp(3,3)
    real(kind=8) :: xout(3,3)
    real(kind=8) :: toler=1.0d-8
    real  pw
    real temp
    real amm(3,3)
    !
    !   temps
    integer nmat_size
    !
    !   body
    pw = -0.5
    n_atom = size(atom_ref_flag)
    nm = size(am)
    !
    lm = 1
    icc = 0
    allocate(ref_to_cond(n_atom))
    ref_to_cond(1:n_atom) = 0
    !
    !   Conditioners are square root of on diagonal 3x3 matrices
    !  While conditioning save references to conditionars and make diagonal terms of 
    !  the matrix as a unit matrix
    !
    nmat_size = size(am)
    do ia=1,n_atom
       if(atom_ref_flag(ia).gt.0)then
          icc = icc + 1
          ref_to_cond(ia) = icc
          xloc(1,1) = am(lm)
          xloc(2,2) = am(lm+1)
          xloc(3,3) = am(lm+2)
          xloc(1,2) = am(lm+3)
          xloc(1,3) = am(lm+4)
          xloc(2,3) = am(lm+5)

          xloc(2,1) = xloc(1,2)
          xloc(3,1) = xloc(1,3)
          xloc(3,2) = xloc(2,3)
          xtemp = xloc
          if(atom_fix_flag(ia).eq.1) then
             temp = (xloc(1,1)+xloc(2,2)+xloc(3,3))/3.0
             if(temp.gt.0.0) then
                temp = 1.0/sqrt(temp)
             else
                temp = 0.0
             endif
             xcond(1:3,1:3,icc) = 0.0
             xcond(1,1,icc) = temp
             xcond(2,2,icc) = temp
             xcond(3,3,icc) = temp
          else
             call deigen_filter_invert_f90_r(xloc,xout,toler,pw)
             xcond(1:3,1:3,icc) = xout(1:3,1:3)
             am(lm:lm+2) = 1.0
             am(lm+3:lm+5) = 0.0
          endif
          lm = lm + 6
       endif
    enddo
    !
    !   Apply conditioners
    !
    do iaa=1,n_atom
       if(atom_ref_flag(iaa).gt.0) then
          ia = atom_ref_flag(iaa)/10
          nd_ia = nrest_per_atom(iaa)
          npos = rest_per_atom_pos(iaa)
          npos1 = npos+nd_ia-1
          icc = ref_to_cond(iaa)
          do jad=npos,npos1
             jaa = rest_per_atom(jad)
             ja = atom_ref_flag(jaa)/10
             icc1 = ref_to_cond(jaa)
             l1 = rest_per_atom_dist(jad)
             if(l1.gt.0) then
                l = ndiag + 9*(l1-1)+1
                do i=1,3
                   do j=1,3
                      amm(i,j) = am(l)
                      l = l + 1
                   enddo
                enddo
                if(atom_fix_flag(iaa).eq.1.or.atom_fix_flag(jaa).eq.1) then
                   amm = 0.0
                else
                   amm = matmul(matmul(xcond(1:3,1:3,icc),amm),xcond(1:3,1:3,icc1))
                endif
                l = ndiag + 9*(l1-1)+1
                do i=1,3
                   do j=1,3
                      am(l) = amm(i,j)
                      l = l + 1
                   enddo
                enddo
             endif
          enddo
       endif
    enddo
    !    stop
    deallocate(ref_to_cond)
    return
  end subroutine simple_atom_conditioner

  subroutine decondition_shifts(shifts,xcond,atom_ref_flag)
    !
    !   Decondition shifts. 
    !
    real, intent(in) :: xcond(:,:,:)
    integer, intent(in) :: atom_ref_flag(:)
    real, intent(inout) :: shifts(:)

    real x3(3)
    integer i,j,ip,i1,i2
    integer n_atom

    n_atom = size(atom_ref_flag)
    
    ip = 0
    do i=1,n_atom
       if(atom_ref_flag(i).gt.0) then
          ip = ip + 1
          i1 = 3*ip-2
          i2 = 3*ip
          !          write(*,*)'Before ',shifts(i1:i2)
          !do j=1,3
          !   write(*,*)xcond(j,1:3,ip)
          !enddo
          x3 = matmul(xcond(1:3,1:3,ip),shifts(i1:i2))
          !write(*,*)'After ',x3
          shifts(i1:i2) = x3
       endif
    enddo

    return
  end subroutine decondition_shifts


  subroutine random_vector_gauss(nrand,x_this)
    !
    !---Generate a vector of gaussian random numbers. 
    integer nrand
    real x_this(nrand)
    
    integer ir
    !    real gauss_random
    
    
    do ir=1,nrand
       x_this(ir) =  gauss_random()
    enddo
    
    return
  end subroutine random_vector_gauss
  !
  real function gauss_random()
    !
    !---  Gaussian random number generator. It uses random_number and generates
    !--   two random numbers. For gaussian random numbers Box and Miller 
    !---  transformation is used
    real x
    !
    real w,x1,x2
    real xrand(2)
    
    integer ihave
    real rand
    save ihave,rand
    data ihave/0/
    
    if(ihave.eq.0) then
       w = 2.0
       do while (w.gt.1.0) 
          call random_number(xrand)
          x1 = 2.0*xrand(1)-1.0
          x2 = 2.0*xrand(2)-1.0
          w = x1*x1+x2*x2
          !          w = (2.0*xrand(1)-0.5)**2+(2.0*xrand(2)-0.5)**2
       enddo
       w = sqrt((-2.0*log(w))/w)
       gauss_random = x1*w
       rand = x2*w
       ihave = 1
    else
       gauss_random = rand
       ihave = 0
    endif
    
    return
  end function gauss_random
  
  
end module gibbs_gm
