module NMRTargets
  implicit none
  
  real*8 :: B0_nmr = 0.0
  real*8 :: gammaA_nmr = 0.0
  real*8 :: gammaB_nmr = 0.0
  real*8 :: PlankHbar = 1.054571726E-34
  real*8 :: BoltzmannK = 0.0
  real*8 :: T_nmr = 298.0
  real*8, private :: pi=atan(1.0d0)*4.0d0
  real*8 :: nmr_scale_tensors = 1.0
  logical pcs_exist,rdc_exist
  character(len=5) :: nmr_tensor_estim_flag = 'SEPAR'

  !
  character(len=512) :: file_pcs_internal=' '
  character(len=512) :: file_rdc_internal=' '
  
  !
  !---  Things for parser
  integer, private :: ntok,itk,itk_in,itk_out,itok
  integer, private,parameter :: maxtok = 500
  integer, private ::  ibeg(maxtok),iend(maxtok),idec(maxtok),itype(maxtok)
  character, private :: line_parse*600,key*4
  character, private :: cvalue(maxtok)*4
  real, private :: fvalue(maxtok)
  logical, private :: lend,lprint
  integer, private :: iend_r,lend_f
  integer, private :: ios

    
  
contains
  !
  !  What are units of pcs? Weights designed for these pcs-s should reflect their unit. 
  !  Minimised function should be pure number. Otherwise change of coordinate system 
  !  would change contributors to the function and it would be difficult to bring them 
  !  togeter. I think there should be sigma
  !
  subroutine pcsrdc_nmr(nmpos,am,vect,ndist,n_target,n_object,nsym_dist,nw_uval,rest_pos_per_atom,rest_per_atom, &
       i_resid,atom_ref_flag,xyz_crd,occup,fval) 
    ! MAURO: I removed file_pcs and file_rdc and added estim_flag
    ! Garib: Now estim_flag is a parameter of this module
    !
    !   Add contribution of both Pseudocontact Shift and Residue Dipolar Coupling from NMR. 
    !   This routine is based on the paper:
    !   Bertini, Luchinat and Parigi (2002) "Magnetic susceptibility in paramagnetic NMR"
    !   Progress in Magnetic Resonance Spectroscopy, vol40, pp 249-273
    !   
    ! 

    !
    !   refmac related variables. 
    integer, intent(in) :: n_target(:),n_object(:),nsym_dist(:,:),nw_uval(:)
    integer, intent(in) :: rest_pos_per_atom(:),rest_per_atom(:)
    integer, intent(in) :: i_resid(:)
    integer, intent(in) :: atom_ref_flag(:)
    real, intent(in) :: xyz_crd(:,:),occup(:)
    integer, intent(in) :: nmpos,ndist
    
    real, intent(inout) :: am(:),vect(:)
    real, intent(out) :: fval
        
    !
    !   Things for PCS
    integer nmetal_pcs
    real xyz_metal(3)
    integer n_pcs
    real, allocatable :: xyz_pcs(:,:)
    real, allocatable :: obs_pcs(:),tols_pcs(:),w_pcs(:)
    real, allocatable :: tensor_pcs(:,:)
    integer, allocatable :: nref_metal_pcs(:),nref_pcs(:)
    integer, allocatable :: natm_metal_pcs(:),firstatm_metal_pcs(:)
    integer, allocatable :: pcs_metal_taken(:)
    !
    !   Things for RDC
    integer nmetal_rdc
    integer n_rdc
    real, allocatable :: xyz_rdc1(:,:),xyz_rdc2(:,:)
    real, allocatable :: obs_rdc(:),tols_rdc(:),w_rdc(:),k_rdc(:)
    real, allocatable :: Larmorfreq(:) ! Larmorfreq is expressed in MHz, e.g. 900, 700
    real, allocatable :: tensor_rdc(:,:)
    integer, allocatable :: nref_metal_rdc(:),nref_rdc1(:),nref_rdc2(:)
    integer, allocatable :: natm_metal_rdc(:),firstatm_metal_rdc(:)
    integer, allocatable :: rdc_metal_taken(:)
    real gammaA,gammaB,B0
    
    real, allocatable :: tens(:)

    !
    !   Other locals
    integer i,ir,im,j,ipcs,irdc,imetal,jmetal,npcs_local,nrdc_local,ipcs_global,irdc_global
    integer ibegin,iend,jbegin,jend
    real delx(3),drdx(3),dpcs(3,2),drdc(3,2),ten(5),xx(5),param(10)
    real r,r2,k,w
    real pcs_calc,rdc_calc
    real delta
    !
    integer max_mat,max_rest,n_atom
    integer ia1,ia2,ia12,ia22,ip,ipos_mat,idist,imode,iw_uval
    integer isym(4)
    real tol1
    
    !
    !  things for file
    integer ifile,ierr
    
    !
    !   body
    fval = 0.0
    if(.not.pcs_exist.and..not.rdc_exist) return
    max_mat = size(am)
    max_rest = size(n_target)
    
    !
    !   read pcs file and allocate pcs variables
    if (pcs_exist) then
       call open_unform_file(ifile,file_pcs_internal,ierr)
       if(ierr.gt.0) then
          write(*,*)'Error==> Internal PCS file could not be opened'
          stop
       endif
       read(ifile)nmetal_pcs,n_pcs
       
       if (n_pcs.gt.0) then
          allocate(NREF_PCS(n_pcs))                           
          allocate(NREF_METAL_PCS(nmetal_pcs))               
          allocate(NATM_METAL_PCS(nmetal_pcs))               
          allocate(FIRSTATM_METAL_PCS(nmetal_pcs))  
          allocate(pcs_metal_taken(nmetal_pcs))
          allocate(tensor_pcs(5,nmetal_pcs))
          allocate(xyz_pcs(3,n_pcs))
          allocate(obs_pcs(n_pcs))
          allocate(tols_pcs(n_pcs))
          allocate(w_pcs(n_pcs))         
          !   NREF_PCS              - list of atom reference for pcs
          !   NREF_METAL_PCS        - ref for current metal
          !   NATM_METAL_PCS        - n° of pcs for current metal
          !   FIRSTATM_METAL_PCS    - first atom in nref_pcs
          !   pcs_metal_taken       - a flag for tensor estimation 0=no, 1=yes
          
          ipcs_global = 0
          do imetal=1,nmetal_pcs
             read(ifile)nref_metal_pcs(imetal),npcs_local
             natm_metal_pcs(imetal) = npcs_local
             
             if (npcs_local.gt.0) then
                firstatm_metal_pcs(imetal) = ipcs_global + 1
                do ipcs=1,npcs_local
                   ipcs_global = ipcs_global + 1
                   read(ifile)nref_pcs(ipcs_global),obs_pcs(ipcs_global), &
                        tols_pcs(ipcs_global),w_pcs(ipcs_global)
                   xyz_pcs(1:3,ipcs_global) = xyz_crd(1:3,nref_pcs(ipcs_global))
                enddo
             endif
          enddo
       endif
       close(ifile)
    endif
    
    !
    !   read rdc file and allocate rdc variables
    if (rdc_exist) then
       call open_unform_file(ifile,file_rdc_internal,ierr)
       if(ierr.gt.0) then
          write(*,*)'Error==> Internal RDC file could not be opened'
          stop
       endif
       read(ifile)nmetal_rdc,n_rdc
       
       if (n_rdc.gt.0) then
          allocate(NREF_RDC1(n_rdc))      
          allocate(NREF_RDC2(n_rdc))                          
          allocate(NREF_METAL_RDC(nmetal_rdc))               
          allocate(NATM_METAL_RDC(nmetal_rdc))               
          allocate(FIRSTATM_METAL_RDC(nmetal_rdc))  
          allocate(rdc_metal_taken(nmetal_rdc))
          allocate(tensor_rdc(5,nmetal_rdc))
          allocate(Larmorfreq(nmetal_rdc))
          allocate(xyz_rdc1(3,n_rdc))
          allocate(xyz_rdc2(3,n_rdc))
          allocate(obs_rdc(n_rdc))
          allocate(tols_rdc(n_rdc))
          allocate(w_rdc(n_rdc))         
          allocate(k_rdc(n_rdc))
          !   NREF_RDC#           - list of atom reference for rdc, first and second atom
          !   NREF_METAL_RDC      - ref for current metal
          !   NATM_METAL_RDC      - n° of rdc for current metal
          !   FIRSTATM_METAL_RDC  - first atom in nref_rdc
          !   rdc_metal_taken     - a flag for tensor estimation 0=no, 1=yes
          
          irdc_global = 0
          do imetal=1,nmetal_rdc
             read(ifile)nref_metal_rdc(imetal),nrdc_local,Larmorfreq(imetal)
             natm_metal_rdc(imetal) = nrdc_local
             
             if (nrdc_local.gt.0) then
                firstatm_metal_rdc(imetal) = irdc_global + 1
                do irdc=1,nrdc_local
                   irdc_global = irdc_global + 1
                   read(ifile)nref_rdc1(irdc_global),nref_rdc2(irdc_global),obs_rdc(irdc_global), &
                        tols_rdc(irdc_global),w_rdc(irdc_global)
                   xyz_rdc1(1:3,irdc_global) = xyz_crd(1:3,nref_rdc1(irdc_global))
                   xyz_rdc2(1:3,irdc_global) = xyz_crd(1:3,nref_rdc2(irdc_global))
                   
                   gammaA = gammaA_nmr   !
                   gammaB = gammaB_nmr   ! need to be replaced with something like gammas(atomtype)
                   B0 = Larmorfreq(imetal)/42.578
                   
                   !-1/(4.0*pi*r**5) * 3*B0_nmr**2*gammaA_nmr*gammab_nmr*PlankHbar/(15*BoltzmannK*T_nmr*2.0*pi)
                   k_rdc(irdc_global) = -1/(4.0*pi) * 3*B0**2*gammaA*gammaB*PlankHbar/(15*BoltzmannK*T_nmr*2.0*pi)
                enddo
             endif
          enddo
       endif
       close(ifile)
    endif
	
    !
    !   Estimate tensors	
    
    rdc_metal_taken(:) = 0
    if (nmetal_pcs.gt.0) then
       do imetal=1,nmetal_pcs
          if (natm_metal_pcs(imetal).gt.0) then
             xyz_metal = xyz_crd(1:3,nref_metal_pcs(imetal))
             ibegin = firstatm_metal_pcs(imetal)
             iend = ibegin + natm_metal_pcs(imetal) - 1
             if (nmetal_rdc.gt.0) then
                do jmetal=1,nmetal_rdc
                   !
                   !   if metal are same
                   if (nref_metal_pcs(imetal).eq.nref_metal_rdc(jmetal)) then
                      pcs_metal_taken(imetal) = 1
                      rdc_metal_taken(jmetal) = 1
                      jbegin = firstatm_metal_rdc(jmetal)
                      jend = jbegin + natm_metal_rdc(jmetal) - 1
                      call estimate_pcsrdc_tensor(xyz_pcs(1:3,ibegin:iend),xyz_metal,obs_pcs(ibegin:iend),w_pcs(ibegin:iend) &
                           ,xyz_rdc1(1:3,jbegin:jend),xyz_rdc2(1:3,jbegin:jend),obs_rdc(jbegin:jend),w_rdc(jbegin:jend), &
                           k_rdc(jbegin:jend),tensor_pcs(:,imetal),tensor_rdc(:,jmetal),param(:),ierr)	
                      if(ierr.gt.0) then
                         write(*,*)'Error==> Problem with PCS-RDC tensor estimation'
                         stop
                      endif
                   endif
                enddo
             endif
             !
             !   if we have only pcs
             if (pcs_metal_taken(imetal).eq.0) then
                pcs_metal_taken(imetal) = 1
                call estimate_pcsrdc_tensor(xyz_pcs(1:3,ibegin:iend),xyz_metal,obs_pcs(ibegin:iend),w_pcs(ibegin:iend) &
                     ,xyz_rdc1(1:3,1:0),xyz_rdc2(1:3,1:0),obs_rdc(1:0),w_rdc(1:0), &
                     k_rdc(1:0),tensor_pcs(:,imetal),ten(:),param(:),ierr)	
                if(ierr.gt.0) then
                   write(*,*)'Error==> Problem with PCS-RDC tensor estimation'
                   stop
                endif
             endif
          endif
       enddo
    endif
    !  
    !   if we have only rdc
    if (nmetal_rdc.gt.0) then
       do jmetal=1,nmetal_rdc
          if (natm_metal_pcs(jmetal).gt.0) then
             rdc_metal_taken(jmetal) = 1
             jbegin = firstatm_metal_rdc(jmetal)
             jend = jbegin + natm_metal_rdc(jmetal) - 1
             call estimate_pcsrdc_tensor(xyz_pcs(1:3,1:0),xyz_metal,obs_pcs(1:0),w_pcs(1:0) &
                  ,xyz_rdc1(1:3,jbegin:jend),xyz_rdc2(1:3,jbegin:jend),obs_rdc(jbegin:jend),w_rdc(jbegin:jend), &
                  k_rdc(jbegin:jend),ten(:),tensor_rdc(:,jmetal),param(:),ierr)		      
             if(ierr.gt.0) then
                write(*,*)'Error==> Problem with PCS-RDC tensor estimation'
                stop
             endif
          endif
       enddo
    endif
    !
    !   at this point I have all tensors    
    fval = 0.0
    iw_uval = 10
    isym(1) = 1
    isym(2:4) = 0
    !
    !   PCS calculation	
    if (nmetal_pcs.gt.0) then
       do im=1,nmetal_pcs
          if (natm_metal_pcs(im).gt.0) then
             xyz_metal = xyz_crd(1:3,nref_metal_pcs(im))
             ten(1:5) = tensor_pcs(:,im)	 
             k = 1/(4.0*pi*r**5)
             ibegin = firstatm_metal_pcs(im)
             iend = ibegin + natm_metal_pcs(im) - 1		 
             !   
             !  loop over pcs    
             do ir=ibegin,iend
                !
                !    Add contribution of each pcs 
                w = w_pcs(ir)
                tol1 = sqrt(tols_pcs(ir))
                delx(1:3) = xyz_pcs(1:3,ir) - xyz_metal(1:3)
                !
                !    Local refmac things. If ia12 and ia22 > 0 then this atom is refinable. Otherise not. 
                ia1 = nref_pcs(ir)/10
                ia2 = nref_metal_pcs(im)/10
                ia12 = nref_pcs(ir)-10*ia1
                ia22 = nref_metal_pcs(im)-10*ia2
                
                r2 = sum(delx(1:3)**2)
                r  = sqrt(r2)
                !
                !   Calculate value of pseudocontact shift
                !
                call pcsrdc_5tokens(delx,xx)
                pcs_calc = ten(1)*xx(1)
                pcs_calc = pcs_calc + ten(2)*xx(2)
                pcs_calc = pcs_calc + ten(3)*xx(3)
                pcs_calc = pcs_calc + ten(4)*xx(4)
                pcs_calc = pcs_calc + ten(5)*xx(5)   
                
                pcs_calc = k * pcs_calc
                delta = pcs_calc-obs_pcs(ir) ! MAURO: it is often useful for us keeping trace of final calculated pcs, for example in a file
                !
                !   if difference between calculated and observed pcs is within tolerance 
                !   level then do not do calculations
                !
                if(abs(delta).gt.tol1) then
                   drdx(1:3) = delx(1:3)/r
                   dpcs(1:3,1) = -5.0*pcs_calc/r*drdx(1:3)
                   dpcs(1,1) = dpcs(1,1) + k*(-ten(1)*delx(1) + ten(2)*delx(1) + 2.0*ten(3)*delx(2) + 2.0*ten(4)*delx(3))
                   dpcs(2,1) = dpcs(2,1) + k*(-ten(1)*delx(2) - ten(2)*delx(2) + ten(3)*delx(1) + ten(5)*delx(3))
                   dpcs(3,1) = dpcs(3,1) + k*(ten(1)*2.0*delx(3)+ 2.0*ten(4)*delx(1)+2.0*ten(5)*delx(2))
                   dpcs(1:3,2) = -dpcs(1:3,2)
                   !
                   !   Add contributions of the derivatives of the tensor wrt positions. 
                   !   Following lines are for demonstration purpose only. These contributions will need to be added
                   !   to the gradient vector and sparse matrix used in refmac.
                   !  
                   !   Find position of the contribution of the involved atoms in the gradient vector and
                   !   add contributions
                   ip = 3*(ia1-1)
                   vect(ip:ip+3) = vect(ip:ip+3) + w*delta*dpcs(1:3,1)
                   ip = 3*(ia2-1)
                   vect(ip:ip+3) = vect(ip:ip+3) + w*delta*dpcs(1:3,2)
                   
                   !
                   !    Find postion of the contribution of the involved atoms in the diagonal of sec.deriv. matrix and add contributions
                   ipos_mat = 6*(ia1-1)
                   call incr_matr_diagonal(max_mat,ipos_mat,am,w,dpcs(1:3,1))
                   ipos_mat = 6*(ia2-1)
                   call incr_matr_diagonal(max_mat,ipos_mat,am,w,dpcs(1:3,2))
                   !
                   !    Find position of the contributions of the involved atoms to the non-diagonal terms of the sec.deriv matrix
                   !    and add contributions
                   call find_restraint(ndist,ndist,idist,ia1,ia2,isym,n_atom,rest_pos_per_atom, &
                        rest_per_atom,n_target,n_object,nsym_dist,nw_uval,iw_uval,imode)
                   ipos_mat = nmpos + 9*(idist-1)
                   
                   if(imode.eq.0) then
                      call incr_matr_ndiag(max_mat,ipos_mat,am,w,dpcs(1:3,1),dpcs(1:3,2))
                   else
                      call incr_matr_ndiag(max_mat,ipos_mat,am,w,dpcs(1:3,2),dpcs(1:3,1))
                   endif
                   
                endif
                fval = fval + w*max(delta**2-tols_pcs(ir),0.0d0)
             enddo
          endif
       end do
    endif
	
	
    !
    !   RDC calculation	
    if (nmetal_rdc.gt.0) then
       do im=1,nmetal_rdc
          if (natm_metal_rdc(im).gt.0) then
             ten(1:5) = tensor_rdc(:,im)	 
             ibegin = firstatm_metal_rdc(im)
             iend = ibegin + natm_metal_rdc(im) - 1	
             !   
             !  loop over rdc    
             do ir=ibegin,iend
                !
                !    Add contribution of each rdc 
                w = w_rdc(ir)
                k = k_rdc(ir)
                tol1 = sqrt(tols_rdc(ir))
                delx(1:3) = xyz_rdc1(1:3,ir) - xyz_rdc2(1:3,ir)
                !
                !    Local refmac things. If ia12 and ia22 > 0 then this atom is refinable. Otherise not. 
                ia1 = nref_rdc1(ir)/10
                ia2 = nref_rdc2(ir)/10
                ia12 = nref_rdc1(ir)-10*ia1
                ia22 = nref_rdc2(ir)-10*ia2
                
                r2 = sum(delx(1:3)**2)
                r  = sqrt(r2)
                !
                !   Calculate value of pseudocontact shift
                !
                call pcsrdc_5tokens(delx,xx)
                rdc_calc = ten(1)*xx(1)
                rdc_calc = rdc_calc + ten(2)*xx(2)
                rdc_calc = rdc_calc + ten(3)*xx(3)
                rdc_calc = rdc_calc + ten(4)*xx(4)
                rdc_calc = rdc_calc + ten(5)*xx(5)   
                
                rdc_calc = k * rdc_calc
                delta = rdc_calc-obs_rdc(ir) ! MAURO: it is often useful for us keeping trace of final calculated rdc, for example in a file
                !
                !   if difference between calculated and observed rdc is within tolerance 
                !   level then do not do calculations
                !
                if(abs(delta).gt.tol1) then
                   drdx(1:3) = delx(1:3)/r
                   drdc(1:3,1) = -5.0*rdc_calc/r*drdx(1:3)
                   drdc(1,1) = drdc(1,1) + k*(-ten(1)*delx(1) + ten(2)*delx(1) + 2.0*ten(3)*delx(2) + 2.0*ten(4)*delx(3))
                   drdc(2,1) = drdc(2,1) + k*(-ten(1)*delx(2) - ten(2)*delx(2) + ten(3)*delx(1) + ten(5)*delx(3))
                   drdc(3,1) = drdc(3,1) + k*(ten(1)*2.0*delx(3)+ 2.0*ten(4)*delx(1)+2.0*ten(5)*delx(2))
                   drdc(1:3,2) = -drdc(1:3,2)
                   !
                   !   Add contributions of the derivatives of the tensor wrt positions. 
                   !   Following lines are for demonstration purpose only. These contributions will need to be added
                   !   to the gradient vector and sparse matrix used in refmac.
                   !  
                   !   Find position of the contribution of the involved atoms in the gradient vector and
                   !   add contributions
                   ip = 3*(ia1-1)
                   vect(ip:ip+3) = vect(ip:ip+3) + w*delta*drdc(1:3,1)
                   ip = 3*(ia2-1)
                   vect(ip:ip+3) = vect(ip:ip+3) + w*delta*drdc(1:3,2)
                   
                   !
                   !    Find postion of the contribution of the involved atoms in the diagonal of sec.deriv. matrix and add contributions
                   ipos_mat = 6*(ia1-1)
                   call incr_matr_diagonal(max_mat,ipos_mat,am,w,drdc(1:3,1))
                   ipos_mat = 6*(ia2-1)
                   call incr_matr_diagonal(max_mat,ipos_mat,am,w,drdc(1:3,2))
                   !
                   !    Find position of the contributions of the involved atoms to the non-diagonal terms of the sec.deriv matrix
                   !    and add contributions
                   call find_restraint(ndist,ndist,idist,ia1,ia2,isym,n_atom,rest_pos_per_atom, &
                        rest_per_atom,n_target,n_object,nsym_dist,nw_uval,iw_uval,imode)
                   ipos_mat = nmpos + 9*(idist-1)
                   
                   if(imode.eq.0) then
                      call incr_matr_ndiag(max_mat,ipos_mat,am,w,drdc(1:3,1),drdc(1:3,2))
                   else
                      call incr_matr_ndiag(max_mat,ipos_mat,am,w,drdc(1:3,2),drdc(1:3,1))
                   endif
                   
                endif
                fval = fval + w*max(delta**2-tols_rdc(ir),0.0d0)
             enddo
          endif
       end do
    endif
    
    if(pcs_exist.and.n_pcs.gt.0) then
       deallocate(nref_pcs)                           
       deallocate(nref_metal_pcs)               
       deallocate(natm_metal_pcs)               
       deallocate(firstatm_metal_pcs)  
       deallocate(pcs_metal_taken)
       deallocate(tensor_pcs)
       deallocate(xyz_pcs)
       deallocate(obs_pcs)
       deallocate(tols_pcs)
       deallocate(w_pcs)    
    endif
    if(rdc_exist.and.n_rdc.gt.0) then
       deallocate(nref_rdc1)      
       deallocate(nref_rdc2)                          
       deallocate(nref_metal_rdc)               
       deallocate(natm_metal_rdc)               
       deallocate(firstatm_metal_rdc)  
       deallocate(rdc_metal_taken)
       deallocate(tensor_rdc)
       deallocate(larmorfreq)
       deallocate(xyz_rdc1)
       deallocate(xyz_rdc2)
       deallocate(obs_rdc)
       deallocate(tols_rdc)
       deallocate(w_rdc)         
       deallocate(k_rdc)
    endif

    return
  end subroutine pcsrdc_nmr
  

  subroutine estimate_pcsrdc_tensor(xyz_pcs,xyz_metal,obs_pcs,w_pcs, &
                    xyz_rdc1,xyz_rdc2,obs_rdc,w_rdc,kk_rdc,tensor_pcs,tensor_rdc,params,ierr)
    !
    ! Estimate tensor parameters for PCS and RDC contribution. It is a simple linear least-squares fit with weights
    ! Weights are assumed to be known. 
    integer ierr
    real, intent(in) :: xyz_pcs(:,:),xyz_metal(3),obs_pcs(:),w_pcs(:)
    real, intent(in) :: xyz_rdc1(:,:),xyz_rdc2(:,:),obs_rdc(:),w_rdc(:),kk_rdc(:)
    real, intent(out) :: tensor_pcs(:),tensor_rdc(:)
    real, intent(out) :: params(:)
    
    real, allocatable :: tensor(:)  ! MAURO: if 'SEPAR', tensor has dimension 10, 'JOINT' 5, 'SCALE' 11. We need it?

    integer npcs,nrdc
    integer i,j,l
    real k,r,wi
    real*8 :: toler=1.0d-10
    real*8 dfdt(5),d2fdt2(5,5),shifts(5)
    real delx(3),xx(5)

    dfdt(1:5) = 0.0
    d2fdt2(1:5,1:4) = 0.0
    
    npcs = size(xyz_pcs(1,:))
    nrdc = size(xyz_rdc1(1,:))
    
    !
    !   estimation
    select case (nmr_tensor_estim_flag)
    case ('SEPAR')
       if (npcs.gt.0) then
          do i=1,npcs
             wi = w_pcs(i)
             r = sqrt(sum((delx(1:3)**2)))
             k = 1/(4.0*pi*r**5)
             delx = xyz_pcs(1:3,i)-xyz_metal(1:3)
             call pcsrdc_5tokens(delx,xx)
             xx = k*xx
			   
             do j=1,5
                dfdt(j) = dfdt(j) + wi*obs_pcs(i)*xx(j)
                do l=1,5
                   d2fdt2(j,l) = d2fdt2(j,l) + wi*xx(j)*xx(l)
                enddo
             enddo
          end do
          
          call deigen_filter_r90(toler,d2fdt2,5,5,dfdt,shifts)
          tensor_pcs(1:5) = shifts(1:5)
       else
          tensor_pcs(1:5) = 0.0   ! it would be better to assign other kind of values? or nothing?
       endif
       
       if (npcs.gt.0) then
          do i=1,nrdc
             wi = w_rdc(i)
             r = sqrt(sum((delx(1:3)**2)))
             k = kk_rdc(i)/r**5
             delx = xyz_rdc1(1:3,i)-xyz_rdc2(1:3,i)
             call pcsrdc_5tokens(delx,xx)
             xx = k*xx
             
             do j=1,5
                dfdt(j) = dfdt(j) + wi*obs_rdc(i)*xx(j)
                do l=1,5
                   d2fdt2(j,l) = d2fdt2(j,l) + wi*xx(j)*xx(l)
                enddo
             enddo
          end do
          
          call deigen_filter_r90(toler,d2fdt2,5,5,dfdt,shifts)
          tensor_rdc(1:5) = shifts(1:5)
       else
          tensor_rdc(:) = 0.0   ! it would be better to assign other kind of values?
       endif
       !
       !   We may need to refine these parameters further
    case ('JOINT')
      
       
    case ('SCALE')
       
      
    case default
       write(*,*) 'Estimating tensor case ',nmr_tensor_estim_flag,' does not exist'
       call errwrt(1,'Problem with estimating tensor procedure')    
    end select
    
    return
  end subroutine estimate_pcsrdc_tensor
  
  
  subroutine pcsrdc_5tokens(delx,xx)
    !
    !   Use this procedure to calculate the 5 rdc contibution of formula (40a) or (78a)
    !
    real, intent(in) :: delx(3)
    real, intent(out) :: xx(5)
        
    xx(1) = (2*delx(3)*delx(3)-delx(1)**2-delx(2)**2)/2.0
    xx(2) = (delx(2)**2-delx(3)**2)/2.0
    xx(3) = 2.0*delx(1)*delx(2)
    xx(4) = 2.0*delx(1)*delx(3)
    xx(5) = 2.0*delx(2)*delx(3)
    
    return  
  end subroutine pcsrdc_5tokens
  
  !205 ASN  H      -0.379 1  0.10 10.00  1
  !  Convert pcs files to internal file form
  subroutine convert_pcs(n_atom,n_residue,i_resid,res_num_pdb,res_name,atm_name)
    
    integer n_atom,n_residue
    character(len=*) :: res_num_pdb(n_residue)
    character(len=*) :: res_name(n_atom)
    character(len=*) :: atm_name(n_atom)
    integer i_resid(n_atom)
    !
    !   Convert pcs  file to an internal format
    character*512 file_name
    
    character res_name_metal*8,atom_name_local*4,atom_name_metal*4,res_name_local*8
    integer metal_residue,ires_number
    integer imetal_this,imetal_previous
    integer nmetal
    integer, parameter :: nmax_metal = 500
    integer n_refer_metal(nmax_metal),npcs_metal(nmax_metal)
    integer, parameter :: nmax_pcs = 10000
    integer n_refer_atom(nmax_pcs),imetal_pcs(nmax_pcs)
    real pcs_value(nmax_pcs),tol_pcs(nmax_pcs),weight_pcs(nmax_pcs)
    integer n_pcs,ipcs
    !
    !    Other locals
    integer i,ia,jrs

    integer in_file,iout_file,ierr
    logical lexists,eof
    !
    !   Check if PCS file exists. If it is not defined then return, we do not need pcs refinement
    !   If it is defined but does not exist then terminate, We should be using PCS restraints but we cannot
    pcs_exist = .FALSE.
    call ugtenv('pcsin',file_name) ! MAURO: what does it do?
    if(len_trim(file_name).le.0.or.trim(file_name).eq.'PCSIN') return
    inquire(file=file_name,exist=lexists)
    if(.not.lexists) then
       write(*,*)
       write(*,*)
       write(*,*)'PCS file does not exist'
       call errwrt(1,'Problem with PCS file')
    endif
    pcs_exist = .TRUE.
    call open_unform_file(in_file,file_name,ierr)
    
    nmetal = 0
    eof = .FALSE.
    do while(.not.eof)
       line_parse = ' '
       read(in_file,'(a)',iostat=ios)line_parse
       if(ios.ne.0) then
          eof = .TRUE.
          cycle
       endif
       ntok = 500
       call parser(key,line_parse,ibeg,iend,itype,fvalue,cvalue,idec,ntok,lend,lprint)
       if(ntok.eq.3) then     ! ntok = number of columns
          nmetal = nmetal + 1
          metal_residue = nint(fvalue(1))
          res_name_metal = line_parse(ibeg(2):iend(2))
          atom_name_metal = line_parse(ibeg(3):iend(3))
          n_refer_metal(nmetal) = 0
          !
          !    We need to sort out one problem: What happens if we have several chains with the same residue atom name
          !    An example is when we have more than one copies of a molecule in the asymmetric unit
          !    This version deals with only one metal per instruction
          do ia = 1,n_atom
             read(res_num_pdb(ia)(3:6),*)jrs
             if(jrs.eq.metal_residue) then
                !  if(res_name_metal.eq.res_name(i_resid(ia)).and.atom_name_metal.eq.atm_name(ia)) then
                n_refer_metal(nmetal) = ia
                exit
                !  endif
             endif
          enddo

          if(n_refer_metal(nmetal).le.0) then
             write(*,*)
             write(*,*)'Metal ',atom_name_metal,res_name_metal,metal_residue,' could not be found in the pdb file'
             call errwrt(1,'Problem with pcs: Specified metal is not in the pdb file')
          endif
       endif
    enddo
    
    rewind(in_file)

    eof = .FALSE.
    do while(.not.eof)
       line_parse = ' '
       read(in_file,'(a)',iostat=ios)line_parse
       if(ios.ne.0) then
          eof = .TRUE.
          cycle
       endif
       ntok = 500
       call parser(key,line_parse,ibeg,iend,itype,fvalue,cvalue,idec,ntok,lend,lprint)
       if(ntok.le.3) then
          eof = .TRUE.
          cycle
       endif
       if(ntok.lt.8) then
          write(*,*)'Problem with pcs file. There must be 8 items per pcs line'
          call errwrt(1,'Problem with pcs instructions')
       endif
       ipcs = ipcs + 1
       if(ipcs.gt.nmax_pcs) then
          write(*,*)
          write(*,*)'There are too many pcs instructions. The maximum allowed value = ',nmax_pcs
          call errwrt(1,'Problem with PCS: too many instructions')
       endif
       
       if(itype(1).eq.2) then              ! MAURO: what does it mean itype = 2 ? Where can I find info about parser procedure?
          ires_number = nint(fvalue(1))
       else
          write(*,*)'Problem with PCS file. The first item should be an integer'
          call errwrt(1,'Problem with pcs instructions')
       endif
       res_name_local = line_parse(ibeg(2):iend(2))
       atom_name_local = line_parse(ibeg(3):iend(3))
       do ia = 1,n_atom
          read(res_num_pdb(ia)(3:6),*)jrs
          if(jrs.eq.ires_number) then
             if(res_name_local.eq.res_name(i_resid(ia)).and.atom_name_local.eq.atm_name(ia)) then
                n_refer_atom(ipcs) = ia
                exit
             endif
          endif
       enddo
       if(n_refer_atom(ipcs).le.0) then
          write(*,*)
          write(*,*)'Atom ',atom_name_local,res_name_local,ires_number,' could not be found in the pdb file'
          call errwrt(1,'Problem with pcs: Specified atom is not in the pdb file')
       endif
       pcs_value(ipcs) = fvalue(4)
       
       tol_pcs(ipcs) = fvalue(6)
       weight_pcs(ipcs) = fvalue(7)
       imetal_pcs(ipcs) = nint(fvalue(8))
       
    enddo
    close(in_file)
    n_pcs = ipcs
    !
    !   Calculate the number of pcs for each metal
    do ipcs = 1,n_pcs
       npcs_metal(imetal_pcs(ipcs)) = npcs_metal(imetal_pcs(ipcs)) + 1 ! MAURO npcs_metal needs to be initialized to zero?
    enddo

    ! 
    !   Now write pcs to an internal file
    call find_unique_file_name(file_pcs_internal,'.pcs_internal',ierr)
    call open_unform_file(iout_file,file_pcs_internal,ierr)
           
    imetal_previous = 0
    write(iout_file)nmetal,n_pcs
    do i=1,n_pcs
       imetal_this = imetal_pcs(i)
       !   MAURO: I think that we assume that pcs file has data well ordered by metal number.
       !   It should be true. Are we satisfied about this choice? We need to inform
       !   the user about it. How can we do it?
       if(imetal_this.ne.imetal_previous) then
          write(iout_file)n_refer_metal(imetal_this),npcs_metal(imetal_this)
          imetal_previous = imetal_this
       endif
       write(iout_file)n_refer_atom(i),pcs_value(i),tol_pcs(i),weight_pcs(i)
    enddo
    close(iout_file)
  end subroutine convert_pcs
  
  
  subroutine convert_rdc(n_atom,n_residue,i_resid,res_num_pdb,res_name,atm_name)
    !
    !  118  N  118  H    -1.640     1  1.00     0.200 900
    !  Convert rdc files to internal file form    
    !
    
    integer n_atom,n_residue
    character(len=*) :: res_num_pdb(n_residue)
    character(len=*) :: res_name(n_atom)
    character(len=*) :: atm_name(n_atom)
    integer i_resid(n_atom)
    !
    !   Convert rdc  file to an internal format
    character*512 file_name
    
    character res_name_metal*8,atom_name_metal*4,atom_name_local*4
    integer metal_residue,ires_number
    integer imetal_this,imetal_previous
    integer nmetal
    integer, parameter :: nmax_metal = 500
    integer, parameter :: nmax_Larmorfreq = 10
    integer, parameter :: nmax_rdc = 10000
    integer n_refer_metal(nmax_metal)
    integer nrdc_metal_Larmorfreq(nmax_metal*nmax_Larmorfreq)
    integer n_refer_atom1(nmax_rdc),n_refer_atom2(nmax_rdc),imetal_rdc(nmax_rdc)
    real rdc_value(nmax_rdc),tol_rdc(nmax_rdc),weight_rdc(nmax_rdc),Larmorfreq(nmax_rdc)
    real Larmorfreq_this,Larmorfreq_previous
    integer index_ml
    integer n_rdc,irdc
    !
    !    Other locals
    integer i,ia,jrs

    integer in_file,iout_file,ierr
    logical lexists,eof
    !
    !   Check if RDC file exists. If it is not defined then return, we do not need pcs refinement
    !   If it is defined but does not exist then terminate, We should be using RDC restraints but we cannot
    rdc_exist = .FALSE.
    call ugtenv('rdcin',file_name) ! MAURO: what does it do?
    if(len_trim(file_name).le.0.or.trim(file_name).eq.'RDCIN') return
    inquire(file=file_name,exist=lexists)
    if(.not.lexists) then
       write(*,*)
       write(*,*)
       write(*,*)'RDC file does not exist'
       call errwrt(1,'Problem with RDC file')
    endif
    rdc_exist = .TRUE.
    call open_form_file(in_file,file_name,ierr)
    
    nmetal = 0
    eof = .FALSE.
    do while(.not.eof)
       line_parse = ' '
       read(in_file,'(a)',iostat=ios)line_parse
       if(ios.ne.0) then
          eof = .TRUE.
          cycle
       endif
       ntok = 500
       call parser(key,line_parse,ibeg,iend,itype,fvalue,cvalue,idec,ntok,lend,lprint)
       if(ntok.eq.3) then     ! ntok = number of columns
          nmetal = nmetal + 1
          metal_residue = nint(fvalue(1))
          res_name_metal = line_parse(ibeg(2):iend(2))
          atom_name_metal = line_parse(ibeg(3):iend(3))
          n_refer_metal(nmetal) = 0
          !
          !    We need to sort out one problem: What happens if we have several chains with the same residue atom name
          !    An example is when we have more than one copies of a molecule in the asymmetric unit
          !    This version deals with only one metal per instruction
          do ia = 1,n_atom
             read(res_num_pdb(ia)(3:6),*)jrs
             if(jrs.eq.metal_residue) then
                !  if(res_name_metal.eq.res_name(i_resid(ia)).and.atom_name_metal.eq.atm_name(ia)) then
                n_refer_metal(nmetal) = ia
                exit
                !  endif
             endif
          enddo

          if(n_refer_metal(nmetal).le.0) then
             write(*,*)
             write(*,*)'Metal ',atom_name_metal,res_name_metal,metal_residue,' could not be found in the pdb file'
             call errwrt(1,'Problem with rdc: Specified metal is not in the pdb file')
          endif
       endif
    enddo
    
    rewind(in_file)

    eof = .FALSE.
    do while(.not.eof)
       line_parse = ' '
       read(in_file,'(a)',iostat=ios)line_parse
       if(ios.ne.0) then
          eof = .TRUE.
          cycle
       endif
       ntok = 500
       call parser(key,line_parse,ibeg,iend,itype,fvalue,cvalue,idec,ntok,lend,lprint)
       if(ntok.le.3) then
          eof = .TRUE.
          cycle
       endif
       if(ntok.lt.9) then
          write(*,*)'Problem with rdc file. There must be 9 items per rdc line'
          call errwrt(1,'Problem with rdc instructions')
       endif
       irdc = irdc + 1
       if(irdc.gt.nmax_rdc) then
          write(*,*)
          write(*,*)'There are too many rdc instructions. The maximum allowed value = ',nmax_rdc
          call errwrt(1,'Problem with RDC: too many instructions')
       endif
       
       !
       ! look for first atom
       if(itype(1).eq.2) then              ! MAURO: what does it mean itype = 2 ? Where can I find info about parser procedure?
          ires_number = nint(fvalue(1))
       else
          write(*,*)'Problem with RDC file. The first item should be an integer'
          call errwrt(1,'Problem with rdc instructions')
       endif       
       
       atom_name_local = line_parse(ibeg(2):iend(2))
       do ia = 1,n_atom
          read(res_num_pdb(ia)(3:6),*)jrs
          if(jrs.eq.ires_number) then
             if(atom_name_local.eq.atm_name(ia)) then
                n_refer_atom1(irdc) = ia
                exit
             endif
          endif
       enddo
       if(n_refer_atom1(irdc).le.0) then
          write(*,*)
          write(*,*)'Atom ',atom_name_local,ires_number,' could not be found in the pdb file'
          call errwrt(1,'Problem with rdc: Specified atom is not in the pdb file')
       endif
       
       !
       ! look for second atom
       if(itype(3).eq.2) then        
          ires_number = nint(fvalue(3))
       else
          write(*,*)'Problem with RDC file. The third item should be an integer'
          call errwrt(1,'Problem with rdc instructions')
       endif
       
       atom_name_local = line_parse(ibeg(4):iend(4))
       do ia = 1,n_atom
          read(res_num_pdb(ia)(3:6),*)jrs
          if(jrs.eq.ires_number) then
             if(atom_name_local.eq.atm_name(ia)) then
                n_refer_atom2(irdc) = ia
                exit
             endif
          endif
       enddo
       if(n_refer_atom2(irdc).le.0) then
          write(*,*)
          write(*,*)'Atom ',atom_name_local,ires_number,' could not be found in the pdb file'
          call errwrt(1,'Problem with rdc: Specified atom is not in the pdb file')
       endif       
       
       rdc_value(irdc) = fvalue(5)
       
       imetal_rdc(irdc) = nint(fvalue(6))
       tol_rdc(irdc) = fvalue(7)
       weight_rdc(irdc) = fvalue(8)
       Larmorfreq(irdc) = fvalue(9)
       
    enddo
    close(in_file)
    n_rdc = irdc
    
    !   MAURO: I think that we assume that rdc file has data well ordered by metal number.
    !   It should be true. Are we satisfied about this choice? We need to inform
    !   the user about it. How can we do it?
    !
    !   Calculate the number of rdc for each pair metal-frequency
    imetal_previous = 0
    Larmorfreq_previous = -1
    index_ml = 0
    do irdc = 1,n_rdc
       imetal_this = imetal_rdc(irdc)
       Larmorfreq_this = Larmorfreq(irdc)
       if((imetal_this.ne.imetal_previous).or.(Larmorfreq_this.ne.Larmorfreq_previous)) then
          index_ml = index_ml + 1
          imetal_previous = imetal_this
          Larmorfreq_previous = Larmorfreq_this
       endif
       nrdc_metal_Larmorfreq(index_ml) = nrdc_metal_Larmorfreq(index_ml) + 1  ! MAURO nrdc_metal needs to be initialized to zero?
    enddo

    ! 
    !   Now write rdc to an internal file
    call find_unique_file_name(file_rdc_internal,'.rdc_internal',ierr)
    call open_unform_file(iout_file,file_rdc_internal,ierr)
           
    imetal_previous = 0
    Larmorfreq_previous = -1
    index_ml = 0
    write(iout_file)nmetal,n_rdc
    do i=1,n_rdc
       imetal_this = imetal_rdc(i)
       Larmorfreq_this = Larmorfreq(i)
       if((imetal_this.ne.imetal_previous).or.(Larmorfreq_this.ne.Larmorfreq_previous)) then
          index_ml = index_ml + 1
          write(iout_file)n_refer_metal(imetal_this),nrdc_metal_Larmorfreq(index_ml),Larmorfreq_this
          imetal_previous = imetal_this
          Larmorfreq_previous = Larmorfreq_this
       endif
       write(iout_file)n_refer_atom1(i),n_refer_atom2(i),rdc_value(i),tol_rdc(i),weight_rdc(i)
    enddo
    close(iout_file)
  end subroutine convert_rdc
  
end module NMRTargets
