module dnarna
  use atomcom
  implicit none

  logical :: basepair_flag = .FALSE.

  !
  !   Base pairs. In fact organisation is more general. Restraints between any pair of residues 
  !   can be dealt with
  !
  character(len=512) :: file_basepairs_dict=' ',user_basepair_file=' '
  character(len=512) :: file_basepairs=' '

  integer npairs
  character(len=8), allocatable :: res_pairs(:,:)
  character(len=17), allocatable :: pairs_label(:)
  integer, allocatable :: bond_start(:),angle_start(:),chiral_start(:),torsion_start(:)

  integer nbond_pairs
  character(len=4), allocatable :: bond_atoms(:,:)
  integer, allocatable :: bond_refer(:,:)
  real, allocatable :: bond_value(:,:)

  integer nangle_pairs
  character(len=4), allocatable :: angle_atoms(:,:)
  integer, allocatable :: angle_refer(:,:)
  real, allocatable :: angle_value(:,:)

  integer nchiral_pairs
  character(len=4), allocatable :: chiral_atoms(:,:)
  integer, allocatable :: chiral_refer(:,:)
  real, allocatable :: chiral_value(:,:)

  integer ntorsion_pairs
  character(len=4), allocatable :: torsion_atoms(:,:)
  integer, allocatable :: torsion_refer(:,:)
  real, allocatable :: torsion_value(:,:)
  !
  !
  integer nreal_pairs
  integer, allocatable :: real_pairs(:,:)
  !
  !  Some local files
  integer, private :: is_symm(4,4)
  integer, private :: ir_1(2),ii(2)
  integer, private :: ia_out(4)
  !
  !---  Things for parser
  integer, private :: ntok,itk,itk_in,itk_out,itok
  integer, private,parameter :: maxtok = 500
  integer, private ::  ibeg(maxtok),iend(maxtok),idec(maxtok),itype(maxtok)
  character, private :: line_parse*600,key*4
  character, private :: cvalue(maxtok)*4
  real, private :: fvalue(maxtok)
  logical, private :: lend,lprint
  integer, private :: iend_r,lend_f
  integer, private :: ios
  !
  !
  integer res_bp(2,2)
  character(len=4) :: chain_bp(2,2)
  character(len=1) :: ins_bp(2,2)
  !
  !  File numbers
  integer, private :: ierr_l
  integer, private :: iun_l,iun_p,iun_bp

  real, private :: rtodeg=180.0/(4.0*atan(1.0)),degtor=(4.0*atan(1.0))/180.0

contains

  subroutine dna_rna_basepair(ibond_file,iangle_file,itorsion_file,ichiral_file,    &
       nbonds_bp,nangles_bp,ntorsions_bp,nchirals_bp)
    !
    !   Define basepair restraints and append them to the files ibond_file, etc
    !   It is assumed that pointers are at the end of the files
    !   The number of bonds, angles, torsions and chirals are returned

    integer ibond_file,iangle_file,itorsion_file,ichiral_file
    integer nbonds_bp,nangles_bp,ntorsions_bp,nchirals_bp
    integer nbond_s,nbond_e,nangle_s,nangle_e,nchiral_s,nchiral_e,ntorsion_s,ntorsion_e
    !
    character res_name1*8,res_name2*8
    !
    !  some more locals, counting etc
    integer i,k,m,ll,ia,ib,ibb,ip,ip_ref,ir,ir1,ir2
    integer iat_s1,iat_e1
    integer icount
    integer is_symm(4,4)
    !
    !   Body

    nbonds_bp    = 0
    nangles_bp   = 0
    ntorsions_bp = 0
    nchirals_bp  = 0
    if(.not.basepair_flag.or.len_trim(file_basepairs_dict).le.0.and.len_trim(user_basepair_file).le.0) return
    !
    !   Read base pair info from the library
    call read_basepair_size

    if(npairs.le.0) then
       write(*,*)
       write(*,*)
       write(*,*)'Problem==> The basepair file seems to be empty:'//trim(file_basepairs_dict)
       call errwrt(1,'Problem when readin base pairs')
    endif
    allocate(res_pairs(2,npairs))
    allocate(pairs_label(npairs))
    allocate(bond_start(npairs+1))
    allocate(angle_start(npairs+1))
    allocate(chiral_start(npairs+1))
    allocate(torsion_start(npairs+1))
    
    allocate(bond_atoms(2,nbond_pairs))
    allocate(bond_refer(2,nbond_pairs))
    allocate(bond_value(2,nbond_pairs))
    
    allocate(angle_atoms(3,nangle_pairs))
    allocate(angle_refer(3,nangle_pairs))
    allocate(angle_value(2,nangle_pairs))
 
    allocate(chiral_atoms(4,nchiral_pairs))
    allocate(chiral_refer(4,nchiral_pairs))
    allocate(chiral_value(2,nchiral_pairs))

    allocate(torsion_atoms(4,ntorsion_pairs))
    allocate(torsion_refer(4,ntorsion_pairs))
    allocate(torsion_value(2,ntorsion_pairs))

    call read_basepair_values

    if(len_trim(file_basepairs).le.0) then
       call find_unique_file_name(file_basepairs,'.basepairs')
    endif
    !
    !  Read instructions and find the list of potential basepairs
    call read_basepair_instructs

    is_symm = 0
    do i=1,4
       is_symm(1,1:4) = 1
    enddo
    !
    !   Now add new bond, angle etc info (we do not use planes here)
    call open_unform_file(iun_bp,file_basepairs,ierr_l)
    read(iun_bp)nreal_pairs
    allocate(real_pairs(3,nreal_pairs))

    do i=1,nreal_pairs
       read(iun_bp)real_pairs(1:3,i)
       ir_1(1) = real_pairs(1,i)
       ir_1(2) = real_pairs(2,i)
       call get_mon_name(ir_1(1),res_name1)
       call get_mon_name(ir_1(2),res_name2)
       ip_ref = real_pairs(3,i)
       if(res_name1.eq.res_pairs(1,ip_ref).and.res_name2.eq.res_pairs(2,ip_ref)) then
          ii(1) = 1
          ii(2) = 2
       else
          ii(1) = 2
          ii(2) = 1
       endif
       !
       !  bonds
       if(nbond_pairs.gt.0) then
          nbond_s = bond_start(ip_ref)
          nbond_e = bond_start(ip_ref+1)-1
          if(nbond_s.le.nbond_e) then
             do ip=nbond_s,nbond_e
                icount = 0
                do ll=1,2
                   iat_s1 = iratm_first(ir_1(ii(ll)))
                   iat_e1 = iat_s1 + natm_res(ir_1(ii(ll))) - 1
                   do ia=iat_s1,iat_e1
                      do k=1,2
                         if(bond_refer(k,ip).eq.ll.and.atm_name(ia).eq.bond_atoms(k,ip)) then
                            icount = icount + 1
                            ia_out(k) = ia
                            ibb = ip
                            exit
                         endif
                      enddo
                   enddo
                enddo
                if(icount.eq.2) then
                   nbonds_bp = nbonds_bp + 1
                   write(ibond_file)1
                   write(ibond_file)ia_out(1:2),bond_value(1:2,ibb),is_symm(1:4,1),1,1
                endif
             enddo
          endif
          !
          !   Write out angles
          nangle_s = angle_start(ip_ref)
          nangle_e   = angle_start(ip_ref+1)-1
          if(nangle_s.le.nangle_e) then
             do ip=nangle_s,nangle_e
                icount = 0
                do ll=1,2
                   iat_s1 = iratm_first(ir_1(ii(ll)))
                   iat_e1 = iat_s1 + natm_res(ir_1(ii(ll))) - 1
                   do ia=iat_s1,iat_e1
                      do k=1,3
                         if(angle_refer(k,ip).eq.ll.and.atm_name(ia).eq.angle_atoms(k,ip)) then
                            icount = icount + 1
                            ia_out(k) = ia
                            ibb = ip
                            exit
                         endif
                      enddo
                   enddo
                enddo
                if(icount.eq.3) then
                   nangles_bp = nangles_bp + 1
                   write(iangle_file)ia_out(1:3),(is_symm(1:4,m),m=1,3),angle_value(1:2,ibb),1,1
                endif
             enddo
          endif
          !
          !   torsions
          ntorsion_s = torsion_start(ip_ref)
          ntorsion_e   = torsion_start(ip_ref+1)-1
          if(ntorsion_s.le.ntorsion_e) then
             do ip=ntorsion_s,ntorsion_e
                icount = 0
                do ll=1,2
                   iat_s1 = iratm_first(ir_1(ii(ll)))
                   iat_e1 = iat_s1 + natm_res(ir_1(ii(ll))) - 1
                   do ia=iat_s1,iat_e1
                      do k=1,4
                         if(torsion_refer(k,ip).eq.ll.and.atm_name(ia).eq.torsion_atoms(k,ip)) then
                            icount = icount + 1
                            ia_out(k) = ia
                            ibb = ip
                            exit
                         endif
                      enddo
                   enddo
                enddo
                if(icount.eq.4) then
                   ntorsions_bp = ntorsions_bp + 1
                   write(itorsion_file)'ResPairs',ia_out(1:4),(is_symm(1:4,m),m=1,4),1,torsion_value(1:2,ibb),1,1,1
                endif
             enddo
          endif
          !
          !   Chirals
          nchiral_s = chiral_start(ip_ref)
          nchiral_e   = chiral_start(ip_ref+1)-1
          if(nchiral_s.le.nchiral_e) then
             do ip=nchiral_s,nchiral_e
                icount = 0
                do ll=1,2
                   iat_s1 = iratm_first(ir_1(ii(ll)))
                   iat_e1 = iat_s1 + natm_res(ir_1(ii(ll))) - 1
                   do ia=iat_s1,iat_e1
                      do k=1,4
                         if(chiral_refer(k,ip).eq.ll.and.atm_name(ia).eq.chiral_atoms(k,ip)) then
                            icount = icount + 1
                            ia_out(k) = ia
                            ibb = ip
                            exit
                         endif
                      enddo
                   enddo
                enddo
                if(icount.eq.4) then
                   nchirals_bp = nchirals_bp + 1
                   write(ichiral_file)ia_out(1:4),(is_symm(1:4,m),m=1,4),chiral_value(1:2,ibb),0,1,1
                endif
             enddo
          endif
       endif
    enddo
    close(iun_bp,status='DELETE')
    deallocate(real_pairs)

    deallocate(res_pairs)
    deallocate(pairs_label)
    deallocate(bond_start)
    deallocate(angle_start)
    deallocate(chiral_start)
    deallocate(torsion_start)
    
    deallocate(bond_atoms)
    deallocate(bond_refer)
    deallocate(bond_value)
    
    deallocate(angle_atoms)
    deallocate(angle_refer)
    deallocate(angle_value)
 
    deallocate(chiral_atoms)
    deallocate(chiral_refer)
    deallocate(chiral_value)

    deallocate(torsion_atoms)
    deallocate(torsion_refer)
    deallocate(torsion_value)

  end subroutine dna_rna_basepair

  subroutine read_basepair_instructs
    use restr_files

    integer i,k,ll,i1,i2,ia,ib,ic,it,ir,ir1,ir2,ir_s1,ir_s2,ip,is
    integer ix,itemp
    integer ibase,icount
    integer nbond_s,nbond_e,nangle_s,nangle_e,nchiral_s,nchiral_e,ntorsion_s,ntorsion_e
    integer iat_s1,iat_e1
    integer npairs_l
    integer res_num,res_num2
    logical auto_bp
    real vnorm1,vnorm2,v12
    real xyz_1(3,4),a(3,3),v1(3),v2(3)
    real :: small_bond=0.001

    real dist
    real angle,delta,delta1
    real valueT,volC
    real det3
    external det3

    character chnnamp*4,chnnamp2*4,ins_code*1,ins_code2*1
    character res_name1*8,res_name2*8
    !
    logical anti_flag
    !
    !----body
    auto_bp = .FALSE.

    call open_form_file(iun_p,keywords_file,ierr_l)
    if(ierr_l.ne.0) then
       write(*,*)'Problem with keywords file'
       call errwrt(1,'Can not continue')
    endif

    call open_unform_file(iun_bp,file_basepairs,ierr_l)

    npairs_l = 0
    ios = 0
    line_parse = ' '
    anti_flag = .FALSE.
    call header('Base pair information')
    write(*,*)('-',i=1,100)
    write(*,'(10x,a5,1x,a8,5x,a7,1x,a7,1x,a3,1x,a5,1x,a8,5x,a7,1x,a7,1x,a8)')'Chain','ResName','ResNum','InsCode','and',    &
         'Chain','ResName','ResNum','InsCode','BP label'
    !
    !   Read users' instructions
    do while(ios.eq.0)
       line_parse = ' '
       read(iun_p,'(a)',iostat=ios)line_parse
       if(ios.ne.0) exit
       ntok = 500
       call parser(key,line_parse,ibeg,iend,itype,fvalue,cvalue,idec,ntok,lend,lprint)
       if(len_trim(line_parse).le.0) cycle
       call ccpupc(key)
       if(key.eq.'END') exit
       do  itk=2,ntok
          call ccpupc(cvalue(itk))
       enddo
       if(key.eq.'REST'.and.ntok.ge.2) then
          if(cvalue(2)(1:2).eq.'BP'.or.cvalue(2).eq.'PAIR'.or.cvalue(2).eq.'BASEP') then
             itk = 3
             anti_flag = .FALSE.
             ins_bp(1:2,1:2) = '.'
             !
             !   If auto then set auto_bp = TRUE. Remaining basepairs will be defined automatically
             if(ntok.lt.3) then
                auto_bp = .TRUE.
             elseif(cvalue(itk).eq.'AUTO') then
                auto_bp = .TRUE.
             elseif(cvalue(itk).eq.'BETW') then
                !
                !   Read instructions.
                i1 = 1
                itk = itk+1
                ! 
                !   Next key is dummy. It is just to make instructions human readable
                if(cvalue(itk).eq.'RESI'.or.cvalue(itk).eq.'MONO') itk = itk + 1
                if(ntok-itk.ge.10) then
                   do while(itk.le.ntok)
                      !
                      !   residue range is defined: from chain <chain> residue <residue> to chain <chain> residue <residue>
                      !   ins code can also be defined
                      if(cvalue(itk).eq.'FROM') then
                         itk = itk+1
                         is = 1
                      elseif(cvalue(itk).eq.'TO') then
                         itk = itk + 1
                         is = 2
                      elseif(cvalue(itk).eq.'AND') then
                         !
                         !   Start residue range for the second part of pairs
                         itk = itk + 1
                         i1 = 2
                      else if(cvalue(itk).eq.'CHAI') then
                         itk = itk + 1
                         chain_bp(is,i1) = line_parse(ibeg(itk):iend(itk))
                         itk = itk + 1
                      elseif(cvalue(itk).eq.'RESI') then
                         itk = itk + 1
                         res_bp(is,i1) = nint(fvalue(itk))
                         itk = itk + 1
                      elseif(cvalue(itk)(1:3).eq.'INS') then
                         itk = itk + 1
                         ins_bp(is,i1) = line_parse(ibeg(itk):iend(itk))
                         itk = itk + 1
                         !
                         !   Residues are anti paralel. In principle residue ranges should define it
                      elseif(cvalue(itk).eq.'ANTI') then
                         anti_flag = .TRUE.
                         itk = itk + 1
                      endif
                   enddo
                endif
                if(res_bp(1,2).gt.res_bp(2,2)) anti_flag = .TRUE.
                if(anti_flag) then
                   !
                   !   If antiparallel then make sure that the range for the second pairs is appropriate 
                   !   I.e. the second residue number is smaller than first
                   if(res_bp(2,2).gt.res_bp(1,2)) then
                      itemp = res_bp(2,2)
                      res_bp(2,2) = res_bp(1,2)
                      res_bp(1,2) = itemp
                   endif
                endif
                if(any(chain_bp(1:2,1:2).eq.' ').or.any(res_bp(1:2,1:2).le.-1000)) then
                   write(*,*)'Error==> Instructions for base pair restraints. Correct syntax is:'
                   write(*,*)'rest bp between from chain <chain> residue <residue> to chain <chain> residue <residue> -'
                   write(*,*)'         and from chain <chain> residue <residue> to chain <chain> residue <residue> [anti>' 
                   write(*,*)'                     or '
                   write(*,*)'rest bp auto'
                   call errwrt(1,'Invalid base pair instructions')
                endif
                do ir=1,n_residue
                   call get_chain_namepdb(chnnamp,ir)
                   ins_code = res_num_pdb(ir)(7:7)
                   if(ins_code.eq.' ')ins_code='.'
                   if(chnnamp.eq.chain_bp(1,1)) then
                      read(res_num_pdb(ir)(3:6),'(i5)')res_num
                      if(res_num.eq.res_bp(1,1)) then
                         if(ins_bp(1,1).eq.'.'.or.ins_code.eq.ins_bp(1,1)) then 
                            ir_s1 = ir
                            exit
                         endif
                      endif
                   endif
                enddo
                !
                !   another one
                do ir=1,n_residue
                   call get_chain_namepdb(chnnamp,ir)
                   ins_code = res_num_pdb(ir)(7:7)
                   if(ins_code.eq.' ')ins_code = '.'
                   if(chnnamp.eq.chain_bp(1,2)) then
                      read(res_num_pdb(ir)(3:6),'(i5)')res_num
                      if(res_num.eq.res_bp(1,2)) then
                         if(ins_bp(1,2).eq.'.'.or.ins_code.eq.ins_bp(1,2)) then
                            ir_s2 = ir
                            exit
                         endif
                      endif
                   endif
                enddo
                ir2=ir_s2
                !
                !  Now find actual pairs
                do ir1=ir_s1,n_residue
                   call get_chain_namepdb(chnnamp,ir1)
                   call get_chain_namepdb(chnnamp2,ir2)

                   ins_code = res_num_pdb(ir1)(7:7)
                   if(ins_code.eq.' ')ins_code='.'
                   ins_code2= res_num_pdb(ir2)(7:7)
                   if(ins_code2.eq.' ') ins_code2='.'
                   read(res_num_pdb(ir1)(3:6),'(i5)')res_num
                   read(res_num_pdb(ir2)(3:6),'(i5)')res_num2

                   if(chnnamp.ne.chain_bp(2,1).or.chnnamp2.ne.chain_bp(2,2)) exit
                   if((anti_flag.and.ir2.lt.1).or.(.not.anti_flag.and.ir2.gt.n_residue)) exit
                   if(res_num.gt.res_bp(2,1)) exit
                   if((anti_flag.and.res_num2.lt.res_bp(2,2)).or.(.not.anti_flag.and.res_num2.gt.res_bp(2,2))) exit
                   !
                   !   Find reference to the basepair info
                   call get_mon_name(ir1,res_name1)
                   call get_mon_name(ir2,res_name2)
                   ibase = 0
                   !
                   !  Is the current pair in the list of standard pairs
                   do i=npairs,1,-1
                      if(res_name1.eq.res_pairs(1,i).and.res_name2.eq.res_pairs(2,i).or. &
                           res_name1.eq.res_pairs(2,i).and.res_name2.eq.res_pairs(1,i)) then
                         ibase = i
                         exit
                      endif
                   enddo
                   if(ibase.gt.0) then
                      !
                      !   Write to the log file that these residues are going to be restrained
                      npairs_l = npairs_l + 1
                      write(iun_bp)ir1,ir2,ibase
                      
                      !  Report pair infomration
                      write(*,'(12x,a4,3x,a8,1x,i7,2x,a1,2x,a3,2x,a4,2x,a8,1x,i7,3x,a1,1x,a8)')            &
                     chnnamp,res_name1,res_num,ins_code,' - ',chnnamp2,res_name2,res_num2,ins_code2,pairs_label(ibase)
                   endif
                   if(ins_bp(2,1).ne.'.'.and.ins_code.eq.ins_bp(2,1).and.res_num.eq.res_bp(2,1)) exit
                   if(ins_bp(2,2).ne.'.'.and.ins_code2.eq.ins_bp(2,2).and.res_num2.eq.res_bp(2,2)) exit
                   if(anti_flag) then
                      ir2 = ir2 - 1
                   else
                      ir2 = ir2 + 1
                   endif
                enddo
             endif
          endif
       endif
    enddo
    close(iun_p)
    !
    !   Automatic pair definition
    if(auto_bp) then
       if(len_trim(file_residue_pairs).le.0) then
          !
          !   If not defined find all residue pairs that are close to each other
          call make_residue_pairs(n_atom,n_atom,xyz_crd(1:3,1:n_atom),occup(1:n_atom),     &
               i_resid(1:n_atom),maxnso,cs_nsym,cs_m_cs,cs_v_cs,cs_cell,                   &
               file_residue_pairs,ierr_l)
       endif
       !
       !  Make sure that we use right pairs. I.e. remove pairs that are not within tolerance
       call open_unform_file(iun_l,file_residue_pairs,ierr_l)
       read(iun_l)nreal_pairs
       allocate(real_pairs(2,nreal_pairs))
       do ip=1,nreal_pairs
          read(iun_l)real_pairs(1:2,ip)
          !
          ! Check whether it is real pair
          ir_1(1) = real_pairs(1,ip)
          ir_1(2) = real_pairs(2,ip)
          call get_chain_namepdb(chnnamp,ir_1(1))
          call get_chain_namepdb(chnnamp2,ir_1(2))
                
          ins_code = res_num_pdb(ir_1(1))(7:7)
          if(ins_code.eq.' ')ins_code='.'
          ins_code2= res_num_pdb(ir_1(2))(7:7)
          if(ins_code2.eq.' ')ins_code2='.'
          read(res_num_pdb(ir_1(1))(3:6),'(i5)')res_num
          read(res_num_pdb(ir_1(2))(3:6),'(i5)')res_num2

          call get_mon_name(ir_1(1),res_name1)
          call get_mon_name(ir_1(2),res_name2)
          pairs: do i=npairs,1,-1
             if(res_name1.eq.res_pairs(1,i).and.res_name2.eq.res_pairs(2,i).or.    &
                  res_name1.eq.res_pairs(2,i).and.res_name2.eq.res_pairs(1,i)) then
                if(res_name1.eq.res_pairs(1,i).and.res_name2.eq.res_pairs(2,i)) then
                   ii(1) = 1
                   ii(2) = 2
                else
                   ii(1) = 2
                   ii(2) = 1
                endif

                nbond_s = bond_start(i)
                nbond_e = bond_start(i+1)-1
                !
                !   Check if this pair is real pair. I.e. if bonds, angles etc are within tolerance limit then
                !   consider this pair for restraints
                if(nbond_e.ge.nbond_s) then
                   do ib=nbond_s,nbond_e
                      icount = 0
                      do ll=1,2
                         iat_s1 = iratm_first(ir_1(ii(ll)))
                         iat_e1 = iat_s1 + natm_res(ir_1(ii(ll))) - 1
                          do ia=iat_s1,iat_e1
                            do k=1,2
                               if(bond_refer(k,ib).eq.ll.and.atm_name(ia).eq.bond_atoms(k,ib)) then
                                  icount = icount + 1
                                  xyz_1(1:3,k) = xyz_crd(1:3,ia)
                                  exit 
                               endif
                            enddo
                         enddo
                      enddo                      
                      if(icount.lt.2) cycle pairs
                      dist = sqrt(sum((xyz_1(1:3,1)-xyz_1(1:3,2))**2))
                      if(abs(dist-bond_value(1,ib)).gt.5.0*bond_value(2,ib)) cycle pairs
                   enddo
                endif
                !
                !  Chirals
                nchiral_s = chiral_start(i)
                nchiral_e = chiral_start(i+1) - 1
                if(nchiral_e.ge.nchiral_s) then
                   do ic = nchiral_s,nchiral_e
                      icount = 0
                      do ll=1,2
                         iat_s1 = iratm_first(ir_1(ii(ll)))
                         iat_e1 = iat_s1 + natm_res(ir_1(ii(ll))) - 1
                         do ia=iat_s1,iat_e1
                            do k=1,4
                               if(chiral_refer(k,ic).eq.ll.and.atm_name(ia).eq.chiral_atoms(k,ic)) then
                                  icount = icount + 1
                                  xyz_1(1:3,k) = xyz_crd(1:3,ia)
                                  exit
                               endif
                            enddo
                         enddo
                      enddo
                      if(icount.lt.4) cycle pairs
                      do ix=1,3
                         a(1:3,ix) = xyz_1(1:3,ix+1)-xyz_1(1:3,1)
                      enddo
                      volC = det3(a)

                      if(abs(volC-chiral_value(1,ic)).gt.4.0*chiral_value(2,ic)) cycle pairs
                   enddo
                endif
                !
                !  Angles
                nangle_s = angle_start(i)
                nangle_e = angle_start(i+1) - 1
                if(nangle_e.ge.nangle_s) then
                   do it = nangle_s,nangle_e
                      icount = 0
                      do ll=1,2
                         iat_s1 = iratm_first(ir_1(ii(ll)))
                         iat_e1 = iat_s1 + natm_res(ir_1(ii(ll))) - 1
                         do ia=iat_s1,iat_e1
                            do k=1,3
                               if(angle_refer(k,it).eq.ll.and.atm_name(ia).eq.angle_atoms(k,it)) then
                                  icount = icount + 1
                                  xyz_1(1:3,k) = xyz_crd(1:3,ia)
                                  exit
                               endif
                            enddo
                         enddo
                      enddo
                      if(icount.lt.3) cycle pairs
                      v1(1:3) = xyz_1(1:3,3)-xyz_1(1:3,2)
                      v2(1:3) = xyz_1(1:3,1)-xyz_1(1:3,2)
                      vnorm1 = max(small_bond,sqrt(sum((v1(1:3)**2))))
                      vnorm2 = max(small_bond,sqrt(sum((v2(1:3)**2))))
                      v12 = dot_product(v1,v2)
                      angle = acos(max(-1.0,min(1.0,v12/(vnorm1*vnorm2))))*rtodeg
                      delta = angle_value(1,it)-angle
                      if(abs(delta).gt.4.0*angle_value(2,it)) cycle pairs
                   enddo
                endif
                
                !
                !  Torsions. Do we need to use period?
                ntorsion_s = torsion_start(i)
                ntorsion_e = torsion_start(i+1) - 1
                if(ntorsion_e.ge.ntorsion_s) then
                   do it = ntorsion_s,ntorsion_e
                      icount = 0
                      do ll=1,2
                         iat_s1 = iratm_first(ir_1(ii(ll)))
                         iat_e1 = iat_s1 + natm_res(ir_1(ii(ll))) - 1
                         do ia=iat_s1,iat_e1
                            do k=1,4
                               if(torsion_refer(k,it).eq.ll.and.atm_name(ia).eq.torsion_atoms(k,it)) then
                                  icount = icount + 1
                                  xyz_1(1:3,k) = xyz_crd(1:3,ia)
                                  exit
                               endif
                            enddo
                         enddo
                      enddo
                      if(icount.lt.4) cycle pairs
                      call torshn(xyz_1,valueT)
                      delta = (torsion_value(1,it)-valueT)*degtor
                      delta1 = atan2(sin(delta),cos(delta))*rtodeg

                      if(abs(delta1).gt.3.0*torsion_value(2,it)) cycle pairs
                   enddo
                endif
                npairs_l = npairs_l + 1
                write(iun_bp)real_pairs(1:2,ip),i
                call get_chain_namepdb(chnnamp,ir_1(1))
                call get_chain_namepdb(chnnamp2,ir_1(2))
                
                ins_code = res_num_pdb(ir_1(1))(7:7)
                if(ins_code.eq.' ')ins_code='.'
                ins_code2= res_num_pdb(ir_1(2))(7:7)
                if(ins_code2.eq.' ')ins_code2='.'
                read(res_num_pdb(ir_1(1))(3:6),'(i5)')res_num
                read(res_num_pdb(ir_1(2))(3:6),'(i5)')res_num2
                write(*,'(12x,a4,3x,a8,1x,i7,4x,a1,5x,a3,3x,a4,3x,a8,1x,i7,4x,a1,6x,a8)')            &
                     chnnamp,res_name1,res_num,ins_code,' - ',chnnamp2,res_name2,res_num2,ins_code2,pairs_label(i)
             endif
          enddo pairs
       enddo
       write(*,*)('-',i=1,100)
       write(*,*)
       close(iun_l)
       deallocate(real_pairs)
    endif
    !
    !  Read and write piars file. For safety we need the number of pairs in the beginning of the file'
    allocate(real_pairs(3,npairs_l))
    rewind(iun_bp)
    do ip=1,npairs_l
       read(iun_bp)real_pairs(1:3,ip)
    enddo
    rewind(iun_bp)
    write(iun_bp)npairs_l
    do ip=1,npairs_l
       write(iun_bp)real_pairs(1:3,ip)
    enddo
    deallocate(real_pairs)
    close(iun_bp)
    !
    !  Sort and remove redundant pairs
  end subroutine read_basepair_instructs
  !
  !   Read info about base pairs from the library
  subroutine read_basepair_size
    !  
    !   This routine read the sizes of basepair bonds, angles etc.
    
    integer ipass

    !
    !  There might be two files. One file is standard file and another one user supplied.
    !
    do ipass=1,2
       if(ipass.eq.2.and.len_trim(user_basepair_file).le.0) then 
          exit
       elseif(ipass.eq.2.and.len_trim(user_basepair_file).gt.0) then
          call open_form_file(iun_l,user_basepair_file,ierr_l)
       else
          call open_form_file(iun_l,file_basepairs_dict,ierr_l)
       endif
       if(ierr_l.gt.0) then
          write(*,*)'Problem in opening basepair file'//trim(file_basepairs_dict)
          stop
       endif
       npairs = 0
       ios = 0
       line_parse = ' '
       do while(ios.eq.0)
          key = ' '
          line_parse=' '
          read(iun_l,'(a)',iostat=ios)line_parse
          if(ios.ne.0) exit
          if(len_trim(line_parse).le.0) cycle
          
          call parser(key,line_parse,ibeg,iend,itype,fvalue,cvalue,idec,ntok,lend,lprint)
          call ccpupc(key)
          if(key.eq.'END') exit
          do  itk=2,ntok
             call ccpupc(cvalue(itk))
          enddo
          
          if(key.eq.'MONO') then
             npairs = npairs + 1
          elseif(key.eq.'BOND') then
             nbond_pairs = nbond_pairs + 1
          elseif(key.eq.'ANGL') then
             nangle_pairs = nangle_pairs + 1
          elseif(key.eq.'CHIR') then
             nchiral_pairs = nchiral_pairs + 1
          elseif(key.eq.'TORS') then
             ntorsion_pairs = ntorsion_pairs+1
          endif
       enddo
       close(iun_l)
    enddo

  end subroutine read_basepair_size

  subroutine read_basepair_values
    !
    !    Read information about basepairs: bonds, angles etc. Instructions are general. 
    !    Any pair of residues can be added
    integer iun_l,ios

    integer ic,ipass

    character(len=4) :: atom_l(4)
    character mon1*8,mon2*8
    character c1*8
    real values(2)
    integer irefer(4)
    !
    !   Parser things
    npairs         = 0
    nbond_pairs    = 0
    nangle_pairs   = 0
    nchiral_pairs  = 0
    ntorsion_pairs = 0
    !
    !  There might be two files. One file is standard file and another one user supplied.
    !
    do ipass=1,2
       if(ipass.eq.2.and.len_trim(user_basepair_file).le.0) then 
          exit
       elseif(ipass.eq.2.and.len_trim(user_basepair_file).gt.0) then
          call open_form_file(iun_l,user_basepair_file,ierr_l)
       else
          if(len_trim(file_basepairs_dict).le.0) cycle
          call open_form_file(iun_l,file_basepairs_dict,ierr_l)
       endif

       if(ierr_l.gt.0) then
          write(*,*)'Problem in opening basepair file'//trim(file_basepairs_dict)
          stop
       endif
       ios = 0
       line_parse = ' '
       do while(ios.eq.0)
          line_parse= ' '
          read(iun_l,'(a)',iostat=ios)line_parse
          if(ios.ne.0) exit
          if(len_trim(line_parse).le.0) cycle
          line_parse = adjustl(line_parse)
          key = ' '
          call parser(key,line_parse,ibeg,iend,itype,fvalue,cvalue,idec,ntok,lend,lprint)
          call ccpupc(key)
          do  itk=2,ntok
             call ccpupc(cvalue(itk))
          enddo
          ! 
          !   Instruction about monomers (residue) pairs. Resiude names and label for the pair
          if(key.eq.'MONO') then
             npairs = npairs + 1
             bond_start(npairs) = nbond_pairs+1
             angle_start(npairs) = nangle_pairs+1
             chiral_start(npairs) = nchiral_pairs+1
             torsion_start(npairs) = ntorsion_pairs+1
             itk=2
             if(ntok.lt.3) then
                write(*,*)'There is a problem in the basepair file'//trim(file_basepairs_dict)
                write(*,*)trim(line_parse)
                write(*,*)'Monomer info is not complete. The correct syntax is:'
                write(*,*)'monomer <mon1> <mon2> [label <label>]'
                call errwrt(1,'Problem with the basepair file')
             endif
             res_pairs(1,npairs) = line_parse(ibeg(itk):iend(itk))
             itk = itk + 1
             res_pairs(2,npairs) = line_parse(ibeg(itk):iend(itk))
             itk = itk + 1
             mon1 = res_pairs(1,npairs)
             mon2 = res_pairs(2,npairs)
             if(itk.lt.ntok) then
                if(cvalue(itk).eq.'LABE') then
                   itk = itk + 1
                   pairs_label(npairs) = line_parse(ibeg(itk):iend(itk))
                endif
             else
                pairs_label(npairs) = trim(res_pairs(1,npairs))//':'//trim(res_pairs(2,npairs))
             endif
          elseif(key.eq.'BOND') then
             irefer = 0
             atom_l = ' '
             values = -1
             ic = 0
             itk = 2
             do while(itk.lt.ntok)
                if(cvalue(itk).eq.'ATOM') then
                   itk = itk + 1
                   ic=ic+1
                   atom_l(ic) = line_parse(ibeg(itk):iend(itk))
                   itk = itk + 1
                   c1 = line_parse(ibeg(itk):iend(itk))
                   itk = itk + 1
                   if(c1.eq.mon1) then
                      irefer(ic) = 1
                   else
                      irefer(ic) = 2
                   endif
                elseif(cvalue(itk).eq.'VALU') then
                   itk = itk + 1
                   values(1) = fvalue(itk)
                   itk = itk + 1
                elseif(cvalue(itk).eq.'SIGM') then
                   itk = itk + 1
                   values(2) = fvalue(itk)
                   itk = itk + 1
                endif
             enddo
             if(any(atom_l(1:2).eq.' ').or.values(1).lt.0.0.or.values(2).lt.0.0) then
                write(*,*)'Problem in the bond file'//trim(file_basepairs_dict)
                write(*,*)trim(line_parse)
                write(*,*)'Bond information is incorrect. The syntax is:'
                write(*,*)'bond atom <atname> <resname> atom <atname> <resname> value <value> sigma <sigma>'
             else
                nbond_pairs = nbond_pairs + 1
                bond_atoms(1:2,nbond_pairs) = atom_l(1:2)
                bond_refer(1:2,nbond_pairs) = irefer(1:2)
                bond_value(1:2,nbond_pairs) = values(1:2)
             endif
          elseif(key.eq.'ANGL') then
             irefer = 0
             atom_l = ' '
             values = -1
             ic = 0
             itk = 2
             do while(itk.lt.ntok)
                if(cvalue(itk).eq.'ATOM') then
                   itk = itk + 1
                   ic=ic+1
                   atom_l(ic) = line_parse(ibeg(itk):iend(itk))
                   itk = itk + 1
                   c1 = line_parse(ibeg(itk):iend(itk))
                   itk = itk + 1
                   if(c1.eq.mon1) then
                      irefer(ic) = 1
                   else
                      irefer(ic) = 2
                   endif
                elseif(cvalue(itk).eq.'VALU') then
                   itk = itk + 1
                   values(1) = fvalue(itk)
                   itk = itk + 1
                elseif(cvalue(itk).eq.'SIGM') then
                   itk = itk + 1
                   values(2) = fvalue(itk)
                   itk = itk + 1
                endif
             enddo
             if(any(atom_l(1:3).eq.' ').or.values(1).lt.0.0.or.values(2).lt.0.0) then
                write(*,*)'Problem in the bond file'//trim(file_basepairs_dict)
                write(*,*)trim(line_parse)
                write(*,*)'Angle information is incorrect. The syntax is:'
                write(*,*)'angle atom <atname> <resname> atom <atname> <resname> atom '//     &
                     '<atname> <resname> value <value> sigma <sigma>'
             else
                nangle_pairs = nangle_pairs + 1
                angle_atoms(1:3,nangle_pairs) = atom_l(1:3)
                angle_refer(1:3,nangle_pairs) = irefer(1:3)
                angle_value(1:2,nangle_pairs) = values(1:2)
             endif
          elseif(key.eq.'CHIR') then
             irefer = 0
             atom_l = ' '
             values = -100.0
             ic = 0
             itk = 2
             do while(itk.lt.ntok)
                if(cvalue(itk).eq.'ATOM') then
                   itk = itk + 1
                   ic=ic+1
                   atom_l(ic) = line_parse(ibeg(itk):iend(itk))
                   itk = itk + 1
                   c1 = line_parse(ibeg(itk):iend(itk))
                   itk = itk  + 1
                   if(c1.eq.mon1) then
                      irefer(ic) = 1
                   else
                      irefer(ic) = 2
                   endif
                elseif(cvalue(itk).eq.'VALU') then
                   itk = itk + 1
                   values(1) = fvalue(itk)
                   itk = itk + 1
                elseif(cvalue(itk).eq.'SIGM') then
                   itk = itk + 1
                   values(2) = fvalue(itk)
                   itk = itk + 1
                endif
             enddo
             if(any(atom_l(1:4).eq.' ').or.values(1).lt.-20.0.or.values(2).lt.0.0) then
                write(*,*)'Problem in the base pair'//trim(file_basepairs_dict)
                write(*,*)trim(line_parse)
                write(*,*)'Chiral information is incorrect. The syntax is:'
                write(*,*)'chiral atom <atname> <resname> atom <atname> <resname> '//       &
                     'atom <atname> <resname> atom <atname> <resname> value <value> sigma <sigma>'
             else
                nchiral_pairs = nchiral_pairs + 1
                chiral_atoms(1:4,nchiral_pairs) = atom_l(1:4)
                chiral_refer(1:4,nchiral_pairs) = irefer(1:4)
                chiral_value(1:2,nchiral_pairs) = values(1:2)
             endif
          elseif(key.eq.'TORS') then
             irefer = 0
             atom_l = ' '
             values = -1000.0
             ic = 0
             itk = 2
             do while(itk.lt.ntok)
                if(cvalue(itk).eq.'ATOM') then
                   itk = itk + 1
                   ic=ic+1
                   atom_l(ic) = line_parse(ibeg(itk):iend(itk))
                   itk = itk + 1
                   c1 = line_parse(ibeg(itk):iend(itk))
                   itk = itk + 1
                   if(c1.eq.mon1) then
                      irefer(ic) = 1
                   else
                      irefer(ic) = 2
                   endif
                elseif(cvalue(itk).eq.'VALU') then
                   itk = itk + 1
                   values(1) = fvalue(itk)
                   itk = itk + 1
                elseif(cvalue(itk).eq.'SIGM') then
                   itk = itk + 1
                   values(2) = fvalue(itk)
                   itk = itk + 1
                endif
             enddo
             if(any(atom_l(1:4).eq.' ').or.values(1).lt.-360.0.or.values(2).lt.0.0) then
                write(*,*)'Problem in the base pair'//trim(file_basepairs_dict)
                write(*,*)trim(line_parse)
                write(*,*)'Torsion information is incorrect. The syntax is:'
                write(*,*)'torsion atom <atname> <resname> atom <atname> <resname> '//             &
                     'atom <atname> <resname> atom <atname> <resname> value <value> sigma <sigma>'
             else
                ntorsion_pairs = ntorsion_pairs + 1
                torsion_atoms(1:4,ntorsion_pairs) = atom_l(1:4)
                torsion_refer(1:4,ntorsion_pairs) = irefer(1:4)
                torsion_value(1:2,ntorsion_pairs) = values(1:2)
             endif
          endif
       enddo
       close(iun_l)
    enddo
    if(npairs.le.0) then
       write(*,*)
       write(*,*)
       write(*,*)'There is no basepair file.'
       call errwrt(1,'Problem with the base file(s)')
    endif
    !
    !   Add a dummy number that is one after the last pairs
    !
    bond_start(npairs+1) = nbond_pairs+1
    angle_start(npairs+1) = nangle_pairs+1
    chiral_start(npairs+1) = nchiral_pairs+1
    torsion_start(npairs+1) = ntorsion_pairs+1

  end subroutine read_basepair_values


end module dnarna
