C
C
C     This code is distributed under the terms and conditions of the
C     CCP4 licence agreement as `Part 2' (Annex 2) software.
C     A copy of the CCP4 licence can be obtained by writing to the
C     CCP4 Secretary, Daresbury Laboratory, Warrington WA4 4AD, UK.
C
C
C------------------------------------------------------------------
C
C----Operation on vectors
C
      FUNCTION DOT_R(N,NMAX,VEC1,VEC2)
C
C-----dot product of vectors
      REAL    VEC1(*),VEC2(*)
      INTEGER N,NMAX,I
C
      DOT_R = 0.0
      DO    I=1,N
        DOT_R = DOT_R + VEC1(I)*VEC2(I)
      ENDDO
      RETURN
      END

      FUNCTION DOT1_R(N,NMAX,VEC1,VEC2)
C
C-----dot product of vectors. Second version. Intermediate
C-----summation in double precision
      REAL             VEC1(*),VEC2(*)
      DOUBLE PRECISION TEMP
      INTEGER          N,NMAX,I
C
      TEMP = 0.0D0
      DO    I=1,N
        TEMP = TEMP + VEC1(I)*VEC2(I)
      ENDDO
      DOT1_R = REAL(TEMP)
      RETURN
      END
C
      SUBROUTINE CROSS_R(A,B,C)
      IMPLICIT NONE
C     =======================
C
C     compute vector product A = B x C
C     .. Array Arguments ..
      REAL   A(3),B(3),C(3)
C
C_END_CROSS
C     ..
      A(1) = B(2)*C(3) - C(2)*B(3)
      A(2) = B(3)*C(1) - C(3)*B(1)
      A(3) = B(1)*C(2) - C(1)*B(2)

      RETURN
      END
C
      SUBROUTINE CROSS_WITH_DERIVS(A,DADB,DADC,B,C)
      IMPLICIT NONE
C     =======================
C
C     compute vector product A = B x C and derivative of elements
C     of A wrt elements of B and C
C
C     .. Array Arguments ..
      REAL             A(3),B(3),C(3),DADB(3,3),DADC(3,3)
C
C_END_CROSS
C     ..
      A(1) = B(2)*C(3) - C(2)*B(3)
      A(2) = B(3)*C(1) - C(3)*B(1)
      A(3) = B(1)*C(2) - C(1)*B(2)

      DADB(1,1) =  0.0
      DADB(1,2) =  C(3)
      DADB(1,3) = -C(2)
      DADB(2,1) = -C(3)
      DADB(2,2) =  0.0
      DADB(2,3) =  C(1)
      DADB(3,1) =  C(2)
      DADB(3,2) = -C(1)
      DADB(3,3) =  0.0

      DADC(1,1) =  0.0
      DADC(1,2) = -B(3)
      DADC(1,3) =  B(2)
      DADC(2,1) =  B(3)
      DADC(2,2) =  0.0
      DADC(2,3) = -B(1)
      DADC(3,1) = -B(2)
      DADC(3,2) =  B(1)
      DADC(3,3) =  0.0

      END
C
      REAL FUNCTION FIND_CHIR_SIGN(XYZ1,XYZ2,XYZ3,XYZ4)
      IMPLICIT NONE
C
C---Find sign of chiral center
      REAL XYZ1(3),XYZ2(3),XYZ3(3),XYZ4(3)
      REAL  A(3,3)
      REAL DET3
      EXTERNAL DET3
C
      A(1,1) = XYZ2(1) - XYZ1(1)
      A(2,1) = XYZ2(2) - XYZ1(2)
      A(3,1) = XYZ2(3) - XYZ1(3)

      A(1,2) = XYZ3(1) - XYZ1(1)
      A(2,2) = XYZ3(2) - XYZ1(2)
      A(3,2) = XYZ3(3) - XYZ1(3)

      A(1,3) = XYZ4(1) - XYZ1(1)
      A(2,3) = XYZ4(2) - XYZ1(2)
      A(3,3) = XYZ4(3) - XYZ1(3)

      FIND_CHIR_SIGN = DET3(A)

      RETURN
      END
C
      REAL FUNCTION DET3(A)
      IMPLICIT NONE
C
C---Determinant of 3x3 matrix
      REAL A(9)
      DET3 = A(1)*(A(5)*A(9)-A(8)*A(6))
     .     - A(4)*(A(2)*A(9)-A(8)*A(3))
     .     + A(7)*(A(2)*A(6)-A(5)*A(3))
      RETURN
      END
C
      SUBROUTINE DET3_WITH_DERIVS(MAT_IN,DETER,DDETDE)
      IMPLICIT NONE
C
C-----Calculates determinant of 3x3 amtrix and its derivatives wrt 
C-----elements of the matrix. Matrix assumed to general real matrix
      REAL MAT_IN(3,3),DDETDE(3,3),DETER
C

      DDETDE(1,1) =  MAT_IN(2,2)*MAT_IN(3,3) - MAT_IN(2,3)*MAT_IN(3,2)
      DDETDE(1,2) = -MAT_IN(2,1)*MAT_IN(3,3) + MAT_IN(2,3)*MAT_IN(3,1)
      DDETDE(1,3) =  MAT_IN(2,1)*MAT_IN(3,2) - MAT_IN(2,2)*MAT_IN(3,1)
      DDETDE(2,1) = -MAT_IN(1,2)*MAT_IN(3,3) + MAT_IN(1,3)*MAT_IN(3,2)
      DDETDE(2,2) =  MAT_IN(1,1)*MAT_IN(3,3) - MAT_IN(1,3)*MAT_IN(3,1)
      DDETDE(2,3) = -MAT_IN(1,1)*MAT_IN(3,2) + MAT_IN(2,2)*MAT_IN(3,1)
      DDETDE(3,1) =  MAT_IN(1,2)*MAT_IN(2,3) - MAT_IN(2,2)*MAT_IN(1,3)
      DDETDE(3,2) = -MAT_IN(1,1)*MAT_IN(2,3) + MAT_IN(2,1)*MAT_IN(1,3)
      DDETDE(3,3) =  MAT_IN(1,1)*MAT_IN(2,2) - MAT_IN(1,2)*MAT_IN(2,1)
c
C---Find determinant
      DETER = MAT_IN(1,1)*DDETDE(1,1) + MAT_IN(1,2)*DDETDE(1,2)+ 
     +        MAT_IN(1,3)*DDETDE(1,3)
      RETURN
      END
C
      SUBROUTINE PLANE_AND_DERIVS_R(NPLANE,X_INP,VM,D,DVMDX,DDDX,
     &           IERROR)
      IMPLICIT NONE
C
C----Finds equation of lsq plane going through given coordinates and
C---derivatives of coefficients wrt positional parameters
      INTEGER NPLANE,IERROR
      REAL D
      REAL X_INP(3,*),VM(3),DVMDX(3,*),DDDX(3,*)
C
C----Local parameters
      INTEGER I,J,K,I1,INFO1,RANK_L,LWORK,NPLANE3
      REAL YYS2(3,3)
      REAL XS(3),X_CEN(3,50),RHS(3,150),
     &     WORKSPACE(100),EIGENV(3)
      REAL TOLER_L,FNPLANE,SFMIN,SLAMCH,DEL_X,DEL_Y,DEL_Z
      LOGICAL ERROR
      DATA TOLER_L/1.0E-6/
      EXTERNAL SLAMCH,MAT2VECT,MAT2VEC
      
      LWORK = 100
C     
C--Find machine parameters
      SFMIN = SLAMCH('S')
C
C
C---Return while it is not too late
      IERROR = 0
      IF(NPLANE.LE.0) THEN
        IERROR = 1
        RETURN
      ENDIF
C
C---Calculate the centre
      FNPLANE = REAL(NPLANE)
      NPLANE3 = NPLANE*3
      DO   I=1,3
        XS(I) = 0.0
        DO  J=1,NPLANE
          XS(I) = XS(I) + X_INP(I,J)
        ENDDO
        XS(I) = XS(I)/FNPLANE
      ENDDO
C
c--Relative (to the center) coordinates
      DO I=1,3
        DO   J=1,NPLANE
          X_CEN(I,J) = X_INP(I,J)-XS(I)
        ENDDO
      ENDDO
C
C--Setup matrix 3x3. It includes normalisation also.
      DO   I=1,3
         DO  J=1,3
           YYS2(I,J) = 0.0
           DO  K=1,NPLANE
             YYS2(I,J) = YYS2(I,J) + X_CEN(I,K)*X_CEN(J,K)
           ENDDO
         ENDDO
       ENDDO
cd       DO   I=1,3
cd         DO   J=1,3
cd           YYS1(I,J) = YYS2(I,J)
cd         ENDDO
cd       ENDDO
C
c---Calculate plane equations
      CALL SSYEV_MO('V','U',3,YYS2,3,EIGENV,WORKSPACE,LWORK,INFO1)

C
C---Eigenvector corresponding to minimum eigenvector which is first one.
C---Be careful here, may be other way around
       VM(1) = YYS2(1,1)
       VM(2) = YYS2(2,1)
       VM(3) = YYS2(3,1)
C
C---
       D = VM(1)*XS(1) + VM(2)*XS(2)+VM(3)*XS(3)
C
cd       YYS1(1,4) = VM(1)
cd       YYS1(2,4) = VM(2)
cd       YYS1(3,4) = VM(3)
C
C---Now calculate all right hand sides. Check right hand side carefully
       I1 = 0
       DO    I=1,NPLANE
         I1 = I1 + 1
         DEL_X =  2.0*VM(1)*(X_CEN(1,I)*VM(1)+X_CEN(2,I)*VM(2)+
     &                              X_CEN(3,I)*VM(3))
         DVMDX(1,I1) = -(2.0*VM(1)*X_CEN(1,I)+VM(2)*X_CEN(2,I)+
     &                            VM(3)*X_CEN(3,I)) + DEL_X*VM(1)

         DVMDX(2,I1) = -VM(1)*X_CEN(2,I) + DEL_X*VM(2)
         DVMDX(3,I1) = -VM(1)*X_CEN(3,I) + DEL_X*VM(3)
         DEL_Y = 2.0*VM(2)*(X_CEN(1,I)*VM(1)+X_CEN(2,I)*VM(2)+
     &                              X_CEN(3,I)*VM(3))
cd         RHS(4,I1) = 0
         I1 = I1 + 1
         DVMDX(1,I1) = -VM(2)*X_CEN(1,I) + DEL_Y*VM(1)
         DVMDX(2,I1) = -(VM(1)*X_CEN(1,I)+2.0*VM(2)*X_CEN(2,I)+
     &                                VM(3)*X_CEN(3,I)) + DEL_Y*VM(2)
         DVMDX(3,I1) = -VM(2)*X_CEN(3,I) + DEL_Y*VM(3)
cd         RHS(4,I1) = 0.0
         DEL_Z = 2.0*VM(1)*(X_CEN(1,I)*VM(1)+X_CEN(2,I)*VM(2)+
     &                              X_CEN(3,I)*VM(3))
         I1 = I1 + 1
         DVMDX(1,I1) = -VM(3)*X_CEN(1,I) + DEL_Z*VM(1)
         DVMDX(2,I1) = -VM(3)*X_CEN(2,I) + DEL_Z*VM(2)
         DVMDX(3,I1) = -(VM(1)*X_CEN(1,I)+    VM(2)*X_CEN(2,I)+
     &                           2.0*VM(3)*X_CEN(3,I)) + DEL_Z*VM(3)
cd         RHS(4,I1) = 0.0
      ENDDO
      EIGENV(1) = 0.0
      EIGENV(2) = EIGENV(2)-EIGENV(1)
      EIGENV(3) = EIGENV(3)-EIGENV(1)
      DO    I=1,NPLANE3
        CALL MAT2VECT(3,3,YYS2,DVMDX(1,I),RHS(1,I),ERROR)
        DO   J=1,3
          IF(EIGENV(J).GT.AMAX1(EIGENV(3)*TOLER_L,SFMIN)) THEN
            RHS(J,I)=RHS(J,I)/EIGENV(J)
          ELSE
            RHS(J,I) = 0.0
          ENDIF
        ENDDO
        CALL MAT2VEC(3,3,YYS2,RHS(1,I),DVMDX(1,I),ERROR)
      ENDDO  
C
C---Now call SVD to solve equations to find derivatives.
cd      LWORK = NWORKSPACE - 3
cd      CALL SGELSS(3,3,NPLANE3,YYS1,3,RHS,3,,WORKSPACE(1),TOLER_L,
cd     &            RANK_L,WORKSPACE(4),LWORK,INFO)
cd      WRITE(*,*)RANK_L
cd      DO   I=1,NPLANE3
cd        DVMDX(1,I) = RHS(1,I)
cd        DVMDX(2,I) = RHS(2,I)
cd        DVMDX(3,I) = RHS(3,I)
cd      ENDDO
C
cd      WRITE(*,*)'Solved '
cd      STOP

C---Calculate derivatives of d wrt positions
      DO   I=1,NPLANE
         I1 = 3*I-2
         DDDX(1,I) = DVMDX(1,I1  )*XS(1) + DVMDX(2,I1  )*XS(2) + 
     &               DVMDX(3,I1  )*XS(3) + VM(1)/FNPLANE
         DDDX(2,I) = DVMDX(1,I1+1)*XS(1) + DVMDX(2,I1+1)*XS(2) + 
     &               DVMDX(3,I1+1)*XS(3) + VM(2)/FNPLANE
         DDDX(3,I) = DVMDX(1,I1+2)*XS(1) + DVMDX(2,I1+2)*XS(2) + 
     &               DVMDX(3,I1+2)*XS(3) + VM(3)/FNPLANE
      ENDDO
       
C
C---Return
      RETURN

      END

C     =================================
      SUBROUTINE PLANE_R(N,X,VM,D)
C     =================================
C
C    Fit a least-squares plane to a set of points
C    Uses eigenvalue analysis. Find eigenvalues of normal
C    matrix and eigenvector corresponding to the minimum eigenvalue 
C    is coefficients of plane 
C
      implicit none
C
C
C
C     .. Scalar Arguments ..
      REAL     D
      INTEGER  N
C     ..
C     .. Array Arguments ..
      REAL   VM(3),X(3,50)
C     ..
C     .. Local Scalars ..
      INTEGER I,J,K,J1,I_MIN,LWORK,INFO
      REAL    EIGEN_MIN ,VNORM 
C     ..
C     .. Local Arrays ..
      REAL    XS(3),E_V(3),E_WORK(200),XXS1(3,3)
C     ..
C     .. Intrinsic Functions ..
      INTRINSIC ABS,SQRT
C     ..
C
C---- Set up the a matrix
      LWORK = 200
      DO    I = 1,3
        XS(I) = 0.0
        DO    K = 1,N
          XS(I) = XS(I) + X(I,K)
        ENDDO
      ENDDO 
      J1 = 0
      DO    I = 1,3
        DO    J = 1,3
          XXS1(I,J) = 0.0
          DO    K = 1,N
            XXS1(I,J) = X(I,K)*X(J,K) + XXS1(I,J)
          ENDDO
          XXS1(I,J) = XXS1(I,J) - XS(I)*XS(J)/FLOAT(N)
        ENDDO
      ENDDO
C
C---Use lapack.
      CALL SSYEV_MO('V','U',3,XXS1,3,E_V,E_WORK,LWORK,INFO)
      VM(1) = XXS1(1,1)
      VM(2) = XXS1(2,1)
      VM(3) = XXS1(3,1)
      VNORM = SQRT(VM(1)**2+VM(2)**2+VM(3)**2)
      DO   I=1,3
        VM(I) = VM(I)/VNORM
      ENDDO
      D     = (VM(1)*XS(1)+VM(2)*XS(2)+VM(3)*XS(3))/FLOAT(N)
C---Find eigenvalues of XXS
C
      RETURN
      END      

      SUBROUTINE INIT_VEC(SIZE,VEC,VALUE)
C
      REAL    VEC(*),VALUE
      INTEGER SIZE

cd      IF(SIZE.GT.0) THEN
        DO    I=1,SIZE
          VEC(I) = VALUE
        ENDDO
cd      ENDIF
      RETURN
      END
C
      SUBROUTINE INIT_VECN(SIZE,VECN,VALUEN)
C
      INTEGER SIZE,VECN(*),VALUEN
      
      DO    I=1,SIZE
        VECN(I) = VALUEN
      ENDDO
      RETURN
      END
C
C
      SUBROUTINE VECCOPY_R(N,NMAX,VEC_OUT,ALPHA,VEC_IN)
C
C---Copies vector VEC_IN to VEC_OUT. It also applies VEC_IN by
C---alpha if alpha .NE. 1.0
C---N     is current size of vector
C---NMAX  is maximum size of vector
      REAL VEC_IN(*),VEC_OUT(*),ALPHA
      INTEGER N,NMAX
C
      IERROR = 0
      IF(N.GT.NMAX) THEN
        IERROR = 1
        write(*,*)'Size mismatch in VECCOPY_R'
        stop
      ENDIF
      IF(ALPHA.NE.1.0) THEN
        DO   I=1,N
          VEC_OUT(I) = ALPHA*VEC_IN(I)
        ENDDO
      ELSE
        DO    I=1,N
          VEC_OUT(I) = VEC_IN(I)
        ENDDO
      ENDIF
      RETURN
      END
C
      SUBROUTINE AVECPVEC_R(N,NMAX,ALPHA,VEC1,VEC2,VEC_OUT,IERROR)
C
C   VEC_OUT = ALPHA*VEC1 + VEC2
      REAL VEC1(*),VEC2(*),VEC_OUT(*)
      REAL ALPHA
      INTEGER N,NMAX,I,IERROR

      IF(N.LE.0.OR.N.GT.NMAX)THEN
        write(*,*),'Size mismatch In AVECPVEC'
        stop
        IERROR = 1
      ENDIF
      IF(ALPHA.EQ.0.0) THEN
        DO    I=1,N
          VEC_OUT(I) = VEC2(I)
        ENDDO
      ELSEIF(ALPHA.NE.1.0) THEN
        DO   I=1,N
          VEC_OUT(I) = VEC2(I) + ALPHA*VEC1(I)
        ENDDO
      ELSE
        DO   I=1,N
          VEC_OUT(I) = VEC2(I) + VEC1(I)
        ENDDO
      ENDIF
      IERROR = 0
      RETURN
      END
C
C----Operation on matrices and vectors
C
      SUBROUTINE MAT2IDENT(NP,N,AMAT,ERROR)
C
C----Initialises matrix to identity matrix
C
C------Input and output parameters
C
C----NP     is maximum size of matrix
C----N      is current size of matrix
C----AMAT   is matrix to be initialised
C----ERROR  is .TRUE. if N.GT.NP or N.LT.0
C----------------------------------------------
      REAL    AMAT(NP,NP)
      INTEGER NP,N,I,J
      REAL    SZERO,SUNIT
      LOGICAL ERROR
      DATA    SZERO/0.0E0/,SUNIT/1.0E0/
C
      ERROR = .FALSE.
      IF(N.LE.0 .OR. N.GT.NP) THEN
        ERROR = .TRUE.
        RETURN
      ENDIF

      DO    I=1,N
        DO    J=1,N
          IF(I.EQ.J) THEN
             AMAT(I,J) = SUNIT
          ELSE
             AMAT(I,J) = SZERO
          ENDIF
        ENDDO
      ENDDO
      RETURN
      END
C
      SUBROUTINE MAT2VEC(NP,N,AMAT,AVEC_IN,AVEC_OUT,ERROR)
C
C---Multiplies matrix AMAT which has size NxN to AVEC_IN
C---Resultant vector is in AVEC_OUT
C
C---NP - maximal size of matrix and vectors
C---N  - current size of matrix and vectors
C---AMAT     - Matrix will applied to AVEC_IN
C---AVEC_IN  - Input vector which will be multiplied to AMAT
C---AVEC_OUT - resultant output vector
C---ERROR    - simple error checks
C--------------------------------------------------------------------
      REAL AMAT(NP,NP),AVEC_IN(NP),AVEC_OUT(NP)
      LOGICAL ERROR
C
      ERROR = .FALSE.
      IF(N.LE.0 .OR. N.GT.NP) THEN
        ERROR = .TRUE.
        RETURN
      ENDIF
      DO    I=1,N
        AVEC_OUT(I) = 0.0
      ENDDO
      DO    J=1,N
        DO    I=1,N
          AVEC_OUT(I) = AVEC_OUT(I) + AMAT(I,J)*AVEC_IN(J)
        ENDDO
      ENDDO
      RETURN
      END
C
      SUBROUTINE MAT2VECT(NP,N,AMAT,AVEC_IN,AVEC_OUT,ERROR)
C
C---Multiplies transposed matrix AMAT which has size NxN to AVEC_IN
C---Resultant vector is in AVEC_OUT
C
C------Input and output parameters
C
C---NP       - maximal size of matrix and vectors
C---N        - current size of matrix and vectors
C---AMAT     - Matrix will applied to AVEC_IN
C---AVEC_IN  - Input vector which will be multiplied to AMAT
C---AVEC_OUT - resultant output vector
C---ERROR    - simple error checks
C--------------------------------------------------------------------
      REAL    AMAT(NP,NP),AVEC_IN(NP),AVEC_OUT(NP)
      INTEGER NP,N,I,J
      LOGICAL ERROR
C
      ERROR = .FALSE.
      IF(N.LE.0 .OR. N.GT.NP) THEN
        ERROR = .TRUE.
        RETURN
      ENDIF
      DO    I=1,N
        AVEC_OUT(I) = 0.0
        DO    J=1,N
          AVEC_OUT(I) = AVEC_OUT(I) + AMAT(J,I)*AVEC_IN(J)
        ENDDO
      ENDDO
      RETURN
      END
C
      SUBROUTINE IMAT2VECT(NP,N,AMAT,AVEC_IN,AVEC_OUT,ERROR)
C
C---Multiplies transposed matrix AMAT which has size NxN to AVEC_IN
C---Resultant vector is in AVEC_OUT
C
C------Input and output parameters
C
C---NP       - maximal size of matrix and vectors
C---N        - current size of matrix and vectors
C---AMAT     - Matrix will applied to AVEC_IN
C---AVEC_IN  - Input vector which will be multiplied to AMAT
C---AVEC_OUT - resultant output vector
C---ERROR    - simple error checks
C--------------------------------------------------------------------
      INTEGER    AMAT(NP,NP),AVEC_IN(NP),AVEC_OUT(NP)
      INTEGER NP,N,I,J
      LOGICAL ERROR
C
      ERROR = .FALSE.
      IF(N.LE.0 .OR. N.GT.NP) THEN
        ERROR = .TRUE.
        RETURN
      ENDIF
      DO    I=1,N
        AVEC_OUT(I) = 0
        DO    J=1,N
          AVEC_OUT(I) = AVEC_OUT(I) + AMAT(J,I)*AVEC_IN(J)
        ENDDO
      ENDDO
      RETURN
      END
c
      SUBROUTINE MATTRANS(NP,N,AMAT,AMAT_OUT,ERROR)
C
C---Transpose matrix AMAT and stroes in AMAT_OUT
C
C-----Input and output parameters
C
C---NP       - maximal size
C---N        - current size
C---AMAT     - Input matrix to be transposed
C---AMAT_OUT - resultant output matrix
C---ERROR    - simple error checks
C--------------------------------------------------------------------
      REAL AMAT(NP,NP),AMAT_OUT(NP,NP)
      INTEGER NP,N,I1,I2
      LOGICAL ERROR
C
      ERROR = .FALSE.
      IF(N.LE.0 .OR. N.GT.NP) THEN
        ERROR = .TRUE.
        RETURN
      ENDIF
      DO    I1=1,N-1
        DO    I2=I1+1,N
          AMAT_OUT(I1,I2) = AMAT(I2,I1)
          AMAT_OUT(I2,I1) = AMAT(I1,I2)
        ENDDO
      ENDDO
      DO    I1=1,N
        AMAT_OUT(I1,I1) = AMAT(I1,I1)
      ENDDO
      RETURN
      END
C
      SUBROUTINE MAT2MAT(NP,N,AMAT1,AMAT2,AMAT_OUT,ERROR)
C
C---Multiples AMAT1 to AMAT2 and writes results to AMAT_OUT
C
C---NP       - maximal size
C---N        -  current size
C---AMAT1    - Input left matrix
C---AMAT2    - Input right matrix
C---AMAT_OUT - resultant output matrix
C---ERROR    - simple error checks
C--------------------------------------------------------------------
      REAL AMAT1(NP,NP),AMAT2(NP,NP),AMAT_OUT(NP,NP)
      INTEGER N,NP,I,J,K
      LOGICAL ERROR
C
      ERROR = .FALSE.
      IF(N.LE.0. .OR. N.GT.NP) THEN
        ERROR = .TRUE.
        RETURN
      ENDIF
      DO    J=1,N
        DO    I=1,N
          AMAT_OUT(I,J) = 0.0
        ENDDO
      ENDDO
c
      DO   J=1,N
        DO   K=1,N
          DO    I=1,N
            AMAT_OUT(I,K) = AMAT_OUT(I,K) + AMAT1(I,J)*AMAT2(J,K)
          ENDDO
        ENDDO
      ENDDO
      RETURN
      END
C
      SUBROUTINE MATT2MAT(NP,N,AMAT1,AMAT2,AMAT_OUT,ERROR)
C
C---Multiples transpose of AMAT1 to AMAT2 and writes results to AMAT_OUT
C
C----Input and output parameters
C
C---NP       - maximal size
C---N        -  current size
C---AMAT1    - Input left matrix
C---AMAT2    - Input right matrix
C---AMAT_OUT - resultant output matrix
C---ERROR    - simple error checks
C--------------------------------------------------------------------
      REAL AMAT1(NP,NP),AMAT2(NP,NP),AMAT_OUT(NP,NP)
      INTEGER NP,N,I,J,K
      LOGICAL ERROR
C
      ERROR = .FALSE.
      IF(N.LE.0 .OR. N.GT.NP) THEN
        ERROR = .TRUE.
        RETURN
      ENDIF
      DO    J=1,N
        DO    I=1,N
          AMAT_OUT(I,J) = 0.0
        ENDDO
      ENDDO
      DO   J=1,N
        DO   I=1,N
          DO    K=1,N
            AMAT_OUT(I,K) = AMAT_OUT(I,K) + AMAT1(J,I)*AMAT2(J,K)
          ENDDO
        ENDDO
      ENDDO
      RETURN
      END
C
      SUBROUTINE MATT2MATT(NP,N,AMAT1,AMAT2,AMAT_OUT,ERROR)
C
C---Multiples transpose of AMAT1 to transpose of AMAT2 and writes results
C---to AMAT_OUT
C
C-----Input and output parameters
C
C---NP       - maximal size
C---N        -  current size
C---AMAT1    - Input left matrix
C---AMAT2    - Input right matrix
C---AMAT_OUT - resultant output matrix
C---ERROR    - simple error checks
C--------------------------------------------------------------------
      REAL AMAT1(NP,NP),AMAT2(NP,NP),AMAT_OUT(NP,NP)
      INTEGER NP,N,I,J,K
      LOGICAL ERROR
C
      ERROR = .FALSE.
      IF(N.LE.0 .OR. N.GT.NP) THEN
        ERROR = .TRUE.
        RETURN
      ENDIF
      DO    J=1,N
        DO    I=1,N
          AMAT_OUT(I,J) = 0.0
        ENDDO
      ENDDO
      DO   J=1,N
        DO   K=1,N
          DO    I=1,N
            AMAT_OUT(I,K) = AMAT_OUT(I,K) + AMAT1(J,I)*AMAT2(K,J)
          ENDDO
        ENDDO
      ENDDO
      RETURN
      END
C
      SUBROUTINE MAT2MATT(NP,N,AMAT1,AMAT2,AMAT_OUT,ERROR)
C
C---Multiples AMAT1 to transpose of AMAT2 and writes results
C---to AMAT_OUT
C
C-----Input and output parameters
C
C---NP       - maximal size
C---N        -  current size
C---AMAT1    - Input left matrix
C---AMAT2    - Input right matrix
C---AMAT_OUT - resultant output matrix
C---ERROR    - simple error checks
C--------------------------------------------------------------------
      REAL AMAT1(NP,NP),AMAT2(NP,NP),AMAT_OUT(NP,NP)
      LOGICAL ERROR
C
      ERROR = .FALSE.
      IF(N.LE.0 .OR. N.GT.NP) THEN
        ERROR = .TRUE.
        RETURN
      ENDIF
      DO   I1=1,N
        DO   I2=1,N
          AMAT_OUT(I1,I2) = 0.0
          DO    I3=1,N
            AMAT_OUT(I1,I2) =
     +          AMAT_OUT(I1,I2) + AMAT1(I1,I3)*AMAT2(I2,I3)
          ENDDO
        ENDDO
      ENDDO
      RETURN
      END
C
      SUBROUTINE PUT_K_MATRIX(NP,N,NMAX,R_IN,R_OUT,K,ERROR)
C
C---Puts matrix R_IN into K-s position of R_OUT
C
C---Input and output parameters:
C
C NP    - maximum size of matrix R_IN(NP,NP)
C N     - current size of R_IN and R_OUT(NP,NP,NMAX)
C NMAX  - maximum number of matrices in R_OUT
C K     - R_IN will be put into k-s position of R_OUT
C R_IN  - input matrix with maximum size NP R_IN(NP,NP)
C R_OUT - output matrix R_OUT(NP,NP,NMAX)
C ERROR - simple error checks
C----------------------------------------------------------
      REAL    R_IN(NP,NP),R_OUT(NP,NP,NMAX)
      INTEGER NP,N,NMAX,K,I1,I2
      LOGICAL ERROR
C
      ERROR = .FALSE.
      IF(K.LE.0 .OR. K.GT.NMAX .OR. N.LE.0 .OR. N.GT.NP) THEN
        ERROR = .TRUE.
        RETURN
      ENDIF
      DO    I1 = 1,N
        DO    I2 = 1,N
          R_OUT(I1,I2,K) = R_IN(I1,I2)
        ENDDO
      ENDDO
      RETURN
      END
C
      SUBROUTINE GET_K_MATRIX(NP,N,NMAX,R_IN,R_OUT,K,ERROR)
C
C---Gets K-s matrix from R_IN and puts into R_OUT
C
C---Input and output parameters:
C
C NP    - maximum size of matrix R_IN(NP,NP,NMAX)
C N     - current size of R_IN and R_OUT(NP,NP)
C NMAX  - maximum number of matrices in R_OUT
C K     - R_IN will be put into k-s position of R_OUT
C R_IN  - input matrix with maximum size NP R_IN(NP,NP,NMAX)
C R_OUT - output matrix R_OUT(NP,NP)
C ERROR - simple error checks
C----------------------------------------------------------
      REAL    R_IN(NP,NP,NMAX),R_OUT(NP,NP)
      INTEGER NP,N,NMAX,K,I1,I2
      LOGICAL ERROR
C
      ERROR = .FALSE.
      IF(K.LE.0 .OR. K.GT.NMAX .OR. N.LE.0 .OR. N.GT.NP) THEN
        ERROR = .TRUE.
        RETURN
      ENDIF
      DO    I1 = 1,N
        DO    I2 = 1,N
          R_OUT(I1,I2) = R_IN(I1,I2,K)
        ENDDO
      ENDDO
      RETURN
      END
C
C---Solution of linear equations and matrix inversion problems
C
C
C_BEGIN_GAUSSJ
C
      SUBROUTINE GAUSSJ(A,N,NP,B,M,MP,ERROR)
C     ===========================
C
C
C---- Purpose
C     =======
C
C           Linear equation solution by Gauss-Jordan method
C
C---- Usage
C     ======
C
C           CALL GAUSSJ(A,N,NP,B,M,MP)
C
C---- Description of parameters
C     =========================
C
C    A - input matrix of order N*N, stored in an array of NP*NP
C    On output: replaced by  resultant inverse.
C
C    N - order of matrix A
C
C    NP - dimension of array in which A is stored.
C
C    B - input matrix of order N*M containing the M right hand side vectors,
C        stored in an arry of dimensions NP * MP.
C    On output: replaced by  corresponding set of solution vectors.
C
C---- Remarks
C     =======
C
C---- Method
C     ======
C
C     The standard gauss-jordan method is used. the determinant
C     is also calculated. a determinant of zero indicates that
C     the matrix is singular.
C
C
C_END_GAUSSJ
C
C---- Search for largest element
C
C     .. Scalar Arguments ..
         Parameter (NMAX=500)
C     ..
C     .. Array Arguments ..
      REAL A(NP,NP),B(NP,MP)
      INTEGER IPIV(NMAX),INDXR(NMAX),INDXC(NMAX)
      LOGICAL ERROR
C     The integer arrays IPIV INDXR and INDXC are used forthe pivot bookkeeping.
C     ..
C     .. Local Scalars ..
C     ..
C     .. Intrinsic Functions ..
      INTRINSIC ABS
C     ..
C
      IROW = 1
      ICOL = 1
      ERROR=.FALSE.
C
      DO 11 J = 1,N
       IPIV(J)=0
  11  CONTINUE
C
      DO 22 I = 1,N
        BIG = 0.0
        DO 13 J = 1,N
          IF(IPIV(J).NE.1) THEN
          DO 12 K = 1,N
            IF(IPIV(K).EQ.0) THEN
              IF (ABS(A(J,K)) .GE. BIG) THEN
                BIG = ABS(A(J,K))
                IROW = J
                ICOL = K
              END IF
            ELSE IF(IPIV(K).GT.1) THEN
               ERROR=.TRUE.
               RETURN
            END IF
  12      CONTINUE
          END IF
  13    CONTINUE
      IPIV(ICOL)=IPIV(ICOL) + 1
C
      IF (IROW.NE.ICOL) THEN
        DO 14 L = 1,N
          DUM = A(IROW,L)
          A(IROW,L) = A(ICOL,L)
          A(ICOL,L) = DUM
  14    CONTINUE
C
        DO 15 L = 1,M
          DUM = B(IROW,L)
          B(IROW,L) = B(ICOL,L)
          B(ICOL,L) = DUM
  15    CONTINUE
      END IF 
C
C  Divide pivot row by pivot element
      INDXR(I) = IROW
      INDXC(I) = ICOL
      IF ( A(ICOL,ICOL).EQ.0) THEN
        ERROR=.TRUE.
        RETURN
      END IF 
      PIVINV = 1.0/A(ICOL,ICOL)
      A(ICOL,ICOL) = 1.0
C
        DO 16 L = 1,N
          A(ICOL,L) = A(ICOL,L)*PIVINV
  16    CONTINUE
C
        DO 17 L = 1,M
          B(ICOL,L) = B(ICOL,L)*PIVINV
  17    CONTINUE
C
        DO 21 LL = 1,N
          IF(LL.NE.ICOL) THEN
            DUM = A(LL,ICOL)
              A(LL,ICOL) = 0.0
C
              DO 18 L = 1,N
                A(LL,L) = A(LL,L) -  A(ICOL,L)*DUM
  18          CONTINUE
C
              DO 19 L = 1,M
                B(LL,L) = B(LL,L) -  B(ICOL,L)*DUM
  19          CONTINUE
          END IF
  21    CONTINUE
  22  CONTINUE
C
      DO 24 L = N,1,-1
        IF(INDXR(L).NE.INDXC(L) ) THEN
          DO 23 K = 1,N
            DUM = A(K,INDXR(L))
              A(K,INDXR(L)) = A(K,INDXC(L))
              A(K,INDXR(L)) = DUM
  23      CONTINUE
        END IF
  24  CONTINUE
      RETURN
      END
C
      SUBROUTINE MATINV3_R(MAT_IN,MAT_OUT)
      IMPLICIT NONE
C
C---simple marix inversion for 3x3 matrix
      REAL MAT_IN(3,3),MAT_OUT(3,3)
c
C---Local array
      REAL A_MINOR(3,3),DETER,SIGN
      INTEGER I,J
C
C---First calculate minors of matrix

      A_MINOR(1,1) = MAT_IN(2,2)*MAT_IN(3,3) - MAT_IN(2,3)*MAT_IN(3,2)
      A_MINOR(1,2) = MAT_IN(2,1)*MAT_IN(3,3) - MAT_IN(2,3)*MAT_IN(3,1)
      A_MINOR(1,3) = MAT_IN(2,1)*MAT_IN(3,2) - MAT_IN(2,2)*MAT_IN(3,1)
      A_MINOR(2,1) = MAT_IN(1,2)*MAT_IN(3,3) - MAT_IN(1,3)*MAT_IN(3,2)
      A_MINOR(2,2) = MAT_IN(1,1)*MAT_IN(3,3) - MAT_IN(1,3)*MAT_IN(3,1)
      A_MINOR(2,3) = MAT_IN(1,1)*MAT_IN(3,2) - MAT_IN(1,2)*MAT_IN(3,1)
      A_MINOR(3,1) = MAT_IN(1,2)*MAT_IN(2,3) - MAT_IN(2,2)*MAT_IN(1,3)
      A_MINOR(3,2) = MAT_IN(1,1)*MAT_IN(2,3) - MAT_IN(2,1)*MAT_IN(1,3)
      A_MINOR(3,3) = MAT_IN(1,1)*MAT_IN(2,2) - MAT_IN(1,2)*MAT_IN(2,1)
c
C---Find determinant
      DETER = MAT_IN(1,1)*A_MINOR(1,1) - MAT_IN(1,2)*A_MINOR(1,2)+ 
     +        MAT_IN(1,3)*A_MINOR(1,3)
      SIGN = 1.0
      DO    I=1,3
        DO   J=1,3
          MAT_OUT(I,J) = SIGN*A_MINOR(J,I)/DETER
          SIGN         = -SIGN
        ENDDO
      ENDDO
      RETURN
      END

C
      SUBROUTINE CGSOLVE(NV,NMAX,NCYCL,TOLER,A,V,DV,R,SK,P,F,WORKSPACE,
     +                  LWORK,SCALE1,MATMUL1,SCALE_SHIFTS,SCALE1_INV,
     +                  IERROR)
C
C     Solves normal equation (a*dv=v) by conjugate gradient
C     This routine is efficient if matrix is sparse. Routines
C     SCALE1, MATMUL, SCALE_SHIFTS should be supplied
C
C     see for example: Golub & Loan, Matrix Computations 
C                      Press et al. Numericl recipes
C
C----If matrix AM and vector V are going to be used after this routine
C----they should be saved by the calling subroutine
C   
      REAL TOLER1
      PARAMETER (TOLER1 = 1.0E-12)
      REAL EPS_LOC
      PARAMETER (EPS_LOC = 1.0E-8)
C
      REAL      A(*),V(*),DV(*),SK(*),P(*),F(*),R(*),WORKSPACE(*)
      REAL      TOLER
      INTEGER   NV,NMAX,IERROR
      EXTERNAL  SCALE1,MATMUL1,SCALE_SHIFTS,SCALE1_INV
C
      IF(TOLER.LE.0.0) TOLER = TOLER1
C
C----Add alpha |x|^2 here alpha = lambda_max*tolerance. It is to stabilise
C----linear equation solution. 
      IERROR = 0
      IF(NV.GT.NMAX.OR.NV.LE.0) THEN
        IERROR = 1
        write(*,*)'Warning==> CGSOLVE Size mismatch'
        RETURN
      ENDIF
C    
C----Call predoconditioner. Should be supplied
      CALL SCALE1(NV,NMAX,F,A,SK,V,WORKSPACE,LWORK,IERROR)
      IF(IERROR.GT.0) THEN
        write(*,*)'Warning==> In CGSOLVE, after SCALE1'
        RETURN
      ENDIF
C
C---Initialise
      DO    I = 1,NV
        DV(I)  = 0.0
        R(I)   = -V(I)
        P(I)   = R(I)
      ENDDO
      R1SUM = DOT_R(NV,NMAX,R,R)
      IF(R1SUM.LE.TOLER1)RETURN
C
C----First cycle of iteration 
      CALL MATMUL1(NV,NMAX,P,A,F,WORKSPACE,LWORK,IERROR)
      IF(IERROR.GT.0) THEN
        write(*,*)'Warning==> In CGSOLVE, After MATMUL1 1'
        RETURN
      ENDIF
      ALPHA = DOT_R(NV,NMAX,P,F)
      ALPHA = R1SUM/ALPHA
      CALL AVECPVEC_R(NV,NMAX,ALPHA,P,DV,DV,IERROR)
      CALL AVECPVEC_R(NV,NMAX,-ALPHA,F,R,R,IERROR)
      IF(IERROR.GT.0) THEN
        write(*,*)'Warning==> In CGSOLVE After AVECPVEC_R 1'
        RETURN
      ENDIF
      R2SUM    = DOT_R(NV,NMAX,R,R)
      VNORM    = DOT_R(NV,NMAX,V,V)
      TEST_LIM = TOLER*SQRT(VNORM)
      IF(R1SUM.LE.0.0) THEN
        IERROR = 1
        write(*,*)'Error==> CGSOLVE R1SUM le 0.0'
        stop
      ENDIF
C
C----Loop over iterations.
      IF(SQRT(R2SUM).LE.TEST_LIM) GOTO 120
      FVAL0 = 0.0
      DO    ITER = 1,NCYCL
        BETA = R2SUM/R1SUM
        CALL AVECPVEC_R(NV,NMAX,BETA,P,R,P,IERROR)
        IF(IERROR.GT.0) THEN
          write(*,*)'Warning==> In CGSOLVE, After AVECPVEC_R'
          RETURN
        ENDIF
C
C----Multiply matrix by vector. It is external subroutine and should be 
C----supplied
        CALL MATMUL1(NV,NMAX,P,A,F,WORKSPACE,LWORK,IERROR)
        IF(IERROR.GT.0) THEN
          write(*,*)'Warning==> In CGSOLVE after MATMUL1'
          RETURN
        ENDIF
        ALPHA = DOT_R(NV,NMAX,P,F)
        ALPHA = R2SUM/ALPHA
        CALL AVECPVEC_R(NV,NMAX,ALPHA,P,DV,DV,IERROR)
        CALL AVECPVEC_R(NV,NMAX,-ALPHA,F,R,R,IERROR)
        IF(IERROR.GT.0) THEN
          write(*,*)'Warning==> In CGSOLVE, After AVECPVEC_R '
          RETURN
        ENDIF
        R1SUM = R2SUM
        R2SUM = DOT_R(NV,NMAX,R,R)
cd        WRITE(*,'(A,G12.3)')'R2sum ',R2SUM
      CALL MATMUL1(NV,NMAX,DV,A,F,WORKSPACE,LWORK,IERROR)

      FVAL1 = 0.5*DOT_R(NV,NMAX,F,DV) + DOT_R(NV,NMAX,DV,V)
      WRITE(*,*)FVAL1,FVAL0,SQRT(R2SUM),TEST_LIM
      IF(2.0*ABS(FVAL0-FVAL1).LE.
     &            TOLER1*(ABS(FVAL0)+ABS(FVAL1)+EPS_LOC)) GOTO 120
      FVAL0 = FVAL1
        IF(SQRT(R2SUM).LE.TEST_LIM) GOTO 120
      ENDDO
C
  120 CONTINUE
C
      CALL SCALE_SHIFTS(NV,NMAX,SK,DV,IERROR)
      IF(IERROR.GT.0) THEN
        write(*,*)'Warning==> In CGSOLVE after SCALE_SHIFTS'
        RETURN
      ENDIF
      CALL SCALE1_INV(NV,NMAX,F,A,SK,V,WORKSPACE,LWORK,IERROR)
      RETURN
      END
C
      SUBROUTINE CGSOLVE1(NV,NMAX,NCYCL,TOLER,A,V,DV,R,SK,P,F,WORKSPACE,
     +                  LWORK,SCALE1,MATMUL1,SCALE_SHIFTS,SCALE1_INV,
     +                  IERROR)
C
C     Solves normal equation (A*DV=V) by conjugate gradient
C     This routine is efficient if matrix is sparse. Routines
C     SCALE1, MATMUL, SCALE_SHIFTS should be supplied
C
C     see for example: Golub & Loan, Matrix Computations 
C                      Press et al. Numericl recipes
C
C----If matrix A and vector V are going to be used after this routine
C----they should be saved by the calling subroutine
C   
      REAL TOLER1
      PARAMETER (TOLER1 = 1.0E-6)
      REAL EPS_LOC
      PARAMETER (EPS_LOC = 1.0E-8)
      REAL      A(*),V(*),DV(*),SK(*),P(*),F(*),R(*),WORKSPACE(*)
      REAL      TOLER,DIEN
      REAL      FVAL0,FVAL1
      INTEGER   NV,NMAX,IERROR,I
      EXTERNAL  SCALE1,MATMUL1,SCALE_SHIFTS,SCALE1_INV
C
      IF(TOLER.LE.0.0) TOLER = TOLER1
C
C----Add alpha |x|^2 here alpha = lambda_max*tolerance. It is to stabilise
C----linear equation solution. 
      IERROR = 0
      IF(NV.GT.NMAX.OR.NV.LE.0) THEN
        IERROR = 1
        write(*,*)'Warning==> CGSOLVE Size mismatch'
        RETURN
      ENDIF
C    
C----Call predoconditioner. Should be supplied
      CALL SCALE1(NV,NMAX,F,A,SK,V,WORKSPACE,LWORK,IERROR)
      IF(IERROR.GT.0) THEN
        write(*,*)'Warning==> In CGSOLVE, after SCALE1'
        RETURN
      ENDIF
C
C---Initialise
      DO    I = 1,NV
        DV(I)  = 0.0
        R(I)   = -V(I)
        P(I)   = R(I)
      ENDDO
      FVAL0 = 0.0
c
      R1SUM = DOT_R(NV,NMAX,R,R)
      IF(R1SUM.LE.TOLER1)RETURN
C
C----First cycle of iteration 
      CALL MATMUL1(NV,NMAX,P,A,F,WORKSPACE,LWORK,IERROR)
      IF(IERROR.GT.0) THEN
        write(*,*)'Warning==> In CGSOLVE, After MATMUL1 1'
        RETURN
      ENDIF
      ALPHA = DOT_R(NV,NMAX,P,F)
      ALPHA = R1SUM/ALPHA
      CALL AVECPVEC_R(NV,NMAX,ALPHA,P,DV,DV,IERROR)
      CALL AVECPVEC_R(NV,NMAX,-ALPHA,F,R,R,IERROR)
      IF(IERROR.GT.0) THEN
        write(*,*)'Warning==> In CGSOLVE After AVECPVEC_R 1'
        RETURN
      ENDIF

      R2SUM    = DOT_R(NV,NMAX,R,R)
      VNORM    = DOT_R(NV,NMAX,V,V)
      TEST_LIM = TOLER*SQRT(VNORM)
      IF(R1SUM.LE.0.0) THEN
        IERROR = 1
        write(*,*)'Error==> CGSOLVE R1SUM le 0.0'
        stop
      ENDIF
C
      IF(SQRT(R2SUM).LE.TEST_LIM) GOTO 120

c      DIEN = 1.5
C----Loop over iterations.
      DO    ITER = 1,NCYCL
        BETA = R2SUM/R1SUM
        CALL AVECPVEC_R(NV,NMAX,BETA,P,R,P,IERROR)
        IF(IERROR.GT.0) THEN
          write(*,*)'Warning==> In CGSOLVE, After AVECPVEC_R'
          RETURN
        ENDIF
C
C----Multiply matrix by vector. It is external subroutine and should be 
C----supplied
        CALL MATMUL1(NV,NMAX,P,A,F,WORKSPACE,LWORK,IERROR)
        IF(IERROR.GT.0) THEN
          write(*,*)'Warning==> In CGSOLVE after MATMUL1'
          RETURN
        ENDIF
cd        DO I = 1,NV
cd           F(I) = F(I) + DIEN*P(I)
cd        ENDDO
c        DIEN = DIEN/2.0
        ALPHA = DOT_R(NV,NMAX,P,F)
        ALPHA = R2SUM/ALPHA
        CALL AVECPVEC_R(NV,NMAX,ALPHA,P,DV,DV,IERROR)
        CALL AVECPVEC_R(NV,NMAX,-ALPHA,F,R,R,IERROR)
        IF(IERROR.GT.0) THEN
          write(*,*)'Warning==> In CGSOLVE, After AVECPVEC_R '
          RETURN
        ENDIF
        R1SUM = R2SUM
        R2SUM = DOT_R(NV,NMAX,R,R)
c        WRITE(*,*)'R2sum ',SQRT(R2SUM),TEST_LIM
      CALL MATMUL1(NV,NMAX,DV,A,F,WORKSPACE,LWORK,IERROR)

      FVAL1 = 0.5*DOT_R(NV,NMAX,F,DV) + DOT_R(NV,NMAX,DV,V)
      WRITE(*,*)FVAL1,FVAL0,SQRT(R2SUM),TEST_LIM
      IF(2.0*ABS(FVAL0-FVAL1).LE.
     &            TOLER1*(ABS(FVAL0)+ABS(FVAL1)+EPS_LOC)) GOTO 120
      FVAL0 = FVAL1
        IF(SQRT(R2SUM).LE.TEST_LIM) GOTO 120
      ENDDO
C
  120 CONTINUE
      WRITE(*,*)'R2sum ',SQRT(R2SUM),TEST_LIM
C
      CALL SCALE_SHIFTS(NV,NMAX,SK,DV,IERROR)
      IF(IERROR.GT.0) THEN
        write(*,*)'Warning==> In CGSOLVE after SCALE_SHIFTS'
        RETURN
      ENDIF
      CALL SCALE1_INV(NV,NMAX,F,A,SK,V,WORKSPACE,LWORK,IERROR)
      RETURN
      END
C
C---find initil solution using block diagonal matrix
C
      SUBROUTINE CGSOL_RM(NV,NCYCL,TOLER,A,V,DV,
     &     SCALE1,MATMUL1,SCALE_SHIFTS,SCALE1_INV,
     &     IERROR,GAMMA_FLAG,GAMMA,STEP_FLAG,STEP,nmodel)
c
      IMPLICIT NONE
      INTEGER MCYCLE
      PARAMETER (MCYCLE = 100 0)
      INTEGER   NV,IERROR,I,NCYCL,ITER,GAMMA_CYC,MAX_GAMMA_CYC
c
      REAL      TOLER1,A(*),V(*),DV(*)
      real      TOLER,RHO(0:MCYCLE),TEST_LIM
      real      ALPHA,BETA,VNORM,DOT_R,GAMMA,GAMMA_SAVE,STEP
c
      integer nmax,nmodel
      REAL FVAL0,FVAL1
      PARAMETER (TOLER1 = 0.5E-7)
      REAL EPS_LOC
      PARAMETER (EPS_LOC = 0.5E-7)
      real,    allocatable :: sk(:)
      real,    allocatable :: r(:)
      real,    allocatable :: p(:)
      real,    allocatable :: f(:)
      real,    allocatable :: dv_save(:)
c
      LOGICAL   GAMMA_FLAG,CONVER_FLAG,STEP_FLAG
c
      EXTERNAL  SCALE1,MATMUL1,SCALE_SHIFTS,SCALE1_INV,AVECPVEC_R

C---Consider finding initial solution (e.g. Using diagonal terms only). But 
C---this should be outside of this subroutine.
C
c-----initializations
cd      NCYCL = 200
      if(ncycl.le.0) ncycl = mcycle
      nmax = nv
      allocate(sk(nv))
      allocate(r(nv))
      allocate(p(nv))
      allocate(f(nv))
      allocate(dv_save(nv))
c
      IERROR = 0
      IF(TOLER.LE.0.0) TOLER = TOLER1
      CONVER_FLAG = .FALSE.
      IF (.NOT.GAMMA_FLAG) THEN
         GAMMA = 0.0
      ENDIF 
      DO I = 1,NV
         F(I)  = 0.0
         DV(I)  = 0.0
         DV_SAVE(I) = 0.0
      ENDDO 
c      do i=1,100
c         write(*,*)a(i),v(i),dv(i)
c      enddo
c      stop
c   
c-----preconditioning
      CALL SCALE1(NV,NMAX,F,A,SK,V,IERROR,nmodel)
c
c-----calculation of the 2-norm for the gradient
      VNORM = SQRT(DOT_R(NV,NMAX,V,V))
      TEST_LIM = TOLER*VNORM
c-----set initial guess for solution and calculate the gradient of the object
c-----function at the current solution
      MAX_GAMMA_CYC = 500 
      IF (.NOT.STEP_FLAG) THEN
         STEP = 0.050
      ENDIF 
      DO GAMMA_CYC = 1,MAX_GAMMA_CYC
         IF (GAMMA_CYC.NE.1) GAMMA = GAMMA + STEP
         write(*,*) 'Trying gamma equal ', GAMMA
         
 99      CONTINUE
 89      CONTINUE
C
C---Initial lues. DV is the solution vector.
C
         CALL MATMUL1(NV,NMAX,DV,A,R,IERROR,nmodel)
         DO I = 1,NV
            R(I) = V(I) - R(I)
         ENDDO
         CALL AVECPVEC_R(NV,NMAX,-GAMMA,DV,R,R,IERROR)
c
         RHO(0) = DOT_R(NV,NMAX,R,R)

         IF (RHO(0).LT.TOLER1) GOTO 120
C
         FVAL0 = 0.0
         DO ITER = 1,NCYCL
            IF (ITER.EQ.1) THEN
               DO I = 1,NV
                  P(I) = R(I)
               ENDDO
               BETA = 0.0
            ELSE
               BETA = RHO(ITER-1)/RHO(ITER-2)
               CALL AVECPVEC_R(NV,NMAX,BETA,P,R,P,IERROR)
            ENDIF 
c
            CALL MATMUL1(NV,NMAX,P,A,F,IERROR,nmodel)
            CALL AVECPVEC_R(NV,NMAX,GAMMA,P,F,F,IERROR)
c
            ALPHA = RHO(ITER-1)/DOT_R(NV,NMAX,P,F)
            CALL AVECPVEC_R(NV,NMAX,ALPHA,P,DV,DV,IERROR)
            CALL AVECPVEC_R(NV,NMAX,-ALPHA,F,R,R,IERROR)
C
C--Value of the functional
            RHO(ITER) = DOT_R(NV,NMAX,R,R)
            IF(SQRT(RHO(ITER)).GT. 2.0*SQRT(RHO(ITER-1))) THEN
               WRITE(*,*) 'Not converging with gamma equal ', GAMMA
               STEP = STEP*1.1
               GOTO 110
            ENDIF

             IF (SQRT(RHO(ITER)).LT.TEST_LIM) THEN
               IF (GAMMA_FLAG.EQV..FALSE.) THEN
                  write(*,*) 'Convergence reached with no gamma cycles '
                  GOTO 120
               ELSEIF (CONVER_FLAG) THEN
                  write(*,*) SQRT(DOT_R(NV,NMAX,DV,DV))
                  write(*,*) 'Convergence reached with gamma equal ',
     +                        GAMMA
                  STEP = STEP*1.01
                  GOTO 120
               ELSE
                  CONVER_FLAG = .TRUE.
                  STEP_FLAG = .TRUE.
                  GAMMA_SAVE = GAMMA
                  DO I = 1,NV
                     DV_SAVE(I) = DV(I)
                  ENDDO
                  GAMMA = AMAX1(0.0,GAMMA - STEP/5.0)
                  STEP = STEP/1.1
                  IF (GAMMA.LT.0.0) stop
                  write(*,*) 'Gamma decreased to ', GAMMA
cd                  GOTO 99
                  GOTO 120
               ENDIF
            ENDIF
         ENDDO
 110     CONTINUE
         GAMMA_FLAG = .TRUE.
         IF (.NOT.CONVER_FLAG) THEN
            DO I = 1,NV
               DV(I)  = 0.0
            ENDDO
         ELSE
            DO I = 1,NV
               DV(I) = DV_SAVE(I)
            ENDDO
            GAMMA = GAMMA_SAVE
            write(*,*) 'Back to gamma equal ', GAMMA
            GOTO 89
         ENDIF
      ENDDO 
c
  120 CONTINUE
c
      DO  I=1,NV
         DV(I) = DV(I)* (1.0+GAMMA)
      ENDDO
c

      CALL SCALE_SHIFTS(NV,NMAX,SK,DV,IERROR)
      CALL SCALE1_INV(NV,NMAX,F,A,SK,V,IERROR,nmodel)

      deallocate(sk)
      deallocate(p)
      deallocate(f)
      deallocate(r)
      deallocate(dv_save)
c
      RETURN
      END
C
      SUBROUTINE DEIGEN_FILTER_INVERSE_R(RANK,D2F,NPARAM,NMAXPAR,
     &             D_INV,WORKSPACE,NWORKSPACE)
C
C---Solve linear equations using eigenvalue filtering (psuedo inversion)
C
      IMPLICIT NONE
      INTEGER NPARAM,NMAXPAR,NWORKSPACE
      REAL*8 D2F(NMAXPAR,NMAXPAR),D_INV(NMAXPAR,NMAXPAR)
      REAL*8 WORKSPACE(*)
      REAL*8 RANK
C
      INTEGER LWORK,INFO,I,J,K
      LOGICAL ERROR
      REAL*8  SFMIN,DLAMCH
      EXTERNAL SLAMCH,SSYEV_MO,MAT2VECT,MAT2VEC
      
C     
C--Find machine parameters
      SFMIN = DLAMCH('S')
C
C---find eigenvalues and eigenvectors of right hand side matrix
      LWORK = NWORKSPACE - NMAXPAR
c      CALL DSYEV_MO('V','U',NPARAM,D2F,NMAXPAR,WORKSPACE(1),
c     &    WORKSPACE(NMAXPAR+1),LWORK,INFO)
      CALL DSYEV('V','U',NPARAM,D2F,NMAXPAR,WORKSPACE(1),
     &    WORKSPACE(NMAXPAR+1),LWORK,INFO)
C
C---Now solve equations. 0 eigenvalues are not used in the inversion
C
      DO   I=1,NPARAM
         IF(WORKSPACE(I).GT.
     &     ABS(WORKSPACE(NPARAM))*RANK) THEN
           WORKSPACE(I)        = 1.0D0/WORKSPACE(I)
         ELSE
           WORKSPACE(I)        = 1.0D0/ABS(WORKSPACE(NPARAM))
         ENDIF
      ENDDO
C
C--Now psuedoinverse
C
      DO   I=1,NPARAM
        DO   K=1,NPARAM
           D_INV(I,K) = 0.0
           DO  J=1,NPARAM
              D_INV(I,K) = D_INV(I,K) + D2F(I,J)*D2F(K,J)*WORKSPACE(J)
           ENDDO
        ENDDO
      ENDDO

cd      CALL MAT2VEC(NMAXPAR,NPARAM,D2F,WORKSPACE(NPARAM+1),SHIFTS,ERROR)
      RETURN
      END
C
      SUBROUTINE EIGEN_FILTER_R(RANK,D2F,NPARAM,NMAXPAR,DF,SHIFTS,
     &             WORKSPACE,NWORKSPACE)
C
C---Solve linear equations using eigenvalue filtering (psuedo inversion)
C
      IMPLICIT NONE
      INTEGER NPARAM,NMAXPAR,NWORKSPACE
      REAL D2F(NMAXPAR,NMAXPAR),DF(NMAXPAR),SHIFTS(NMAXPAR)
      REAL WORKSPACE(*)
      REAL RANK
C
      INTEGER LWORK,INFO,I
      LOGICAL ERROR
      REAL  SFMIN,SLAMCH
      EXTERNAL SLAMCH,SSYEV_MO,MAT2VECT,MAT2VEC
      
C     
C--Find machine parameters
      SFMIN = SLAMCH('S')
C
C---find eigenvalues and eigenvectors of right hand side matrix
      LWORK = NWORKSPACE - NMAXPAR
      CALL SSYEV_MO('V','U',NPARAM,D2F,NMAXPAR,WORKSPACE(1),
     &    WORKSPACE(NMAXPAR+1),LWORK,INFO)
C
C---Now solve equations. 0 eigenvalues are not used in the inversion
C
      CALL MAT2VECT(NMAXPAR,NPARAM,D2F,DF,WORKSPACE(NPARAM+1),ERROR)
      DO   I=1,NPARAM
         IF(WORKSPACE(I).GT.
     &     AMAX1(ABS(WORKSPACE(NPARAM))*RANK,SFMIN)) THEN
           WORKSPACE(NPARAM+I) = WORKSPACE(NPARAM+I)/WORKSPACE(I)
         ELSE
           WORKSPACE(NPARAM+I) = 0.0
         ENDIF
      ENDDO
      CALL MAT2VEC(NMAXPAR,NPARAM,D2F,WORKSPACE(NPARAM+1),SHIFTS,ERROR)
      RETURN
      END
C
      SUBROUTINE EIGEN_FILTER_R90(RANK,D2F,NPARAM,NMAXPAR,DF,SHIFTS,
     &     ierr)
C
C---  Solve linear equations using eigenvalue filtering (psuedo inversion)
      IMPLICIT NONE
      INTEGER NPARAM,NMAXPAR,NWORKSPACE
      REAL D2F(NMAXPAR,NMAXPAR),DF(NMAXPAR)
      real rank
c
c---  outputs
      integer ierr
      real SHIFTS(NMAXPAR)
      logical error
C
c---- locals
      INTEGER INFO,I
c
c---  allocatables
      integer nwork
      real, allocatable :: workspace(:)
      real, allocatable :: evalue(:)
      real, allocatable :: evector(:,:)
c
c---  externals
      REAL  SFMIN,SLAMCH
      EXTERNAL SLAMCH,SSYEV_MO,MAT2VECT,MAT2VEC
c
c---  Body
      if(nparam.eq.1) then
         shifts(1) = df(1)/d2f(1,1)
      else
c
c--   work out the case with two parameters also and may be the case with three
c--   
         nwork = 2*nparam*nparam
         allocate(workspace(nwork))
         allocate(evalue(nparam))
         allocate(evector(nparam,nparam))
c
         evector(1:nparam,1:nparam) = d2f(1:nparam,1:nparam)
C     
C---  Find machine parameters
         SFMIN = SLAMCH('S')
C
C---  find eigenvalues and eigenvectors of right hand side matrix

         CALL SSYEV_mo('V','U',NPARAM,evector,nparam,evalue,
     &        WORKSPACE,nwork,INFO)
C
C---  Now solve equations. 0 eigenvalues are not used in the inversion
         workspace(1:nparam) = matmul(transpose(evector),df(1:nparam))
         DO   I=1,NPARAM
            IF(evalue(i).GT.
     &           max(ABS(evalue(NPARAM))*RANK,SFMIN)) THEN
               WORKSPACE(I) = WORKSPACE(I)/evalue(I)
            ELSE
               WORKSPACE(I) = 0.0
            ENDIF
         ENDDO
         shifts(1:nparam) = matmul(evector,workspace(1:nparam))
c
c---  Free memory
         deallocate(workspace)
         deallocate(evalue)
         deallocate(evector)
      endif
      RETURN
      END
c
      SUBROUTINE EIGEN_FILTER_INV_R90(RANK,D2F,NPARAM,NMAXPAR,DF,SHIFTS,
     &     d2f_inv,ierr)
C
C---  Solve linear equations using eigenvalue filtering (psuedo inversion)
      IMPLICIT NONE
      INTEGER NPARAM,NMAXPAR,NWORKSPACE
      REAL D2F(NMAXPAR,NMAXPAR),DF(NMAXPAR)
      real rank
c
c---  outputs
      integer ierr
      real SHIFTS(NMAXPAR)
      logical error
      real d2f_inv(nmaxpar,nmaxpar)
C
c---- locals
      INTEGER INFO,I
      integer j
c
c---  allocatables
      integer nwork
      real, allocatable :: workspace(:)
      real, allocatable :: evalue(:)
      real, allocatable :: evector(:,:)
c
c---  externals
      REAL  SFMIN,SLAMCH
      EXTERNAL SLAMCH,SSYEV_MO,MAT2VECT,MAT2VEC
c
c---  Body
      if(nparam.eq.1) then
         d2f_inv(1,1) = 1.0/d2f(1,1)
         shifts(1) = df(1)*d2f_inv(1,1)
      else
c
c--   work out the case with two parameters also and may be the case with three
c--   
         nwork = 2*nparam*nparam
         allocate(workspace(nwork))
         allocate(evalue(nparam))
         allocate(evector(nparam,nparam))
c
         evector(1:nparam,1:nparam) = d2f(1:nparam,1:nparam)
C     
C---  Find machine parameters
         SFMIN = SLAMCH('S')
C
C---  find eigenvalues and eigenvectors of right hand side matrix

         CALL SSYEV('V','U',NPARAM,evector,nparam,evalue,
     &        WORKSPACE,nwork,INFO)
         if(info.ne.0) then
            write(*,*)nparam
            write(*,*)df(1:nparam)
            write(*,*)d2f(1:nparam,1:nparam)
            stop
         endif
C
C---  Now solve equations. 0 eigenvalues are not used in the inversion
c         workspace(1:nparam) = matmul(transpose(evector),df(1:nparam))
         DO   I=1,NPARAM
            IF(evalue(i).GT.
     &           max(ABS(evalue(NPARAM))*RANK,SFMIN)) THEN
c               WORKSPACE(I) = WORKSPACE(I)/evalue(I)
               evalue(i) = 1.0/evalue(i)
c               workspace(i) = workspace(i)*evalue(i)
            ELSE
               evalue(i) = 0.0
c               WORKSPACE(I) = 0.0
            ENDIF
         ENDDO
         do i=1,nparam
            do j=1,nparam
               d2f_inv(i,j) = evector(j,i)*evalue(i)
            enddo
         enddo
         d2f_inv(1:nparam,1:nparam) = 
     &        matmul(evector,d2f_inv(1:nparam,1:nparam))
         shifts(1:nparam) = matmul(d2f_inv(1:nparam,1:nparam),
     &        df(1:nparam))
c
c---  Free memory
         deallocate(workspace)
         deallocate(evalue)
         deallocate(evector)
      endif
      RETURN
      END
c
      SUBROUTINE DEIGEN_FILTER_R(RANK,D2F,NPARAM,NMAXPAR,DF,SHIFTS,
     &             WORKSPACE,NWORKSPACE)
C
C---Solve linear equations using eigenvalue filtering (psuedo inversion)
C---Double precision version
C
      IMPLICIT NONE
      INTEGER NPARAM,NMAXPAR,NWORKSPACE
      REAL*8 D2F(NMAXPAR,NMAXPAR),DF(NMAXPAR),SHIFTS(NMAXPAR)
      REAL*8 WORKSPACE(*)
      REAL*8 RANK
C
      INTEGER LWORK,INFO,I,J
      LOGICAL ERROR
      REAL*8  SFMIN,DLAMCH
      EXTERNAL DLAMCH,DSYEV_MO
cd,MAT2VECT,MAT2VEC
      
C     
C--Find machine parameters
      SFMIN = DLAMCH('S')
C
C---find eigenvalues and eigenvectors of right hand side matrix
      LWORK = NWORKSPACE - NPARAM
c      CALL DSYEV_MO('V','U',NPARAM,D2F,NMAXPAR,WORKSPACE(1),
c     &    WORKSPACE(NMAXPAR+1),LWORK,INFO)
      CALL DSYEV('V','U',NPARAM,D2F,NMAXPAR,WORKSPACE(1),
     &    WORKSPACE(NMAXPAR+1),LWORK,INFO)
C
C---Now solve equations. 0 eigenvalues are not used in the inversion
C
cd      CALL MAT2VECT(NMAXPAR,NPARAM,D2F,DF,WORKSPACE(NPARAM+1),ERROR)
      DO    I=1,NPARAM
         WORKSPACE(NPARAM+I) = 0.0D0
         DO   J=1,NPARAM
           WORKSPACE(NPARAM+I) = WORKSPACE(NPARAM+I) + D2F(J,I)*DF(J)
         ENDDO
      ENDDO
C
      DO   I=1,NPARAM
         IF(WORKSPACE(I).GT.
     &     DMAX1(DABS(WORKSPACE(NPARAM))*RANK,SFMIN)) THEN
           WORKSPACE(NPARAM+I) = WORKSPACE(NPARAM+I)/WORKSPACE(I)
         ELSE
           WORKSPACE(NPARAM+I) = 0.0D0
         ENDIF
      ENDDO
cd      CALL MAT2VEC(NMAXPAR,NPARAM,D2F,WORKSPACE(NPARAM+1),SHIFTS,ERROR)
      DO   I=1,NPARAM
        SHIFTS(I) = 0.0D0
        DO  J=1,NPARAM
          SHIFTS(I) = SHIFTS(I) + D2F(I,J)*WORKSPACE(NPARAM+J)
        ENDDO
      ENDDO
      RETURN
      END
C
      SUBROUTINE DEIGEN_FILTER_R90(RANK,D2F,NPARAM,NMAXPAR,DF,SHIFTS)
C
C---Solve linear equations using eigenvalue filtering (psuedo inversion)
C---Double precision version
C
      IMPLICIT NONE
      INTEGER NPARAM,NMAXPAR,NWORKSPACE
      REAL*8 D2F(NMAXPAR,NMAXPAR),DF(NMAXPAR),SHIFTS(NMAXPAR)
      REAL*8 RANK
C
      INTEGER LWORK,INFO,I,J
      LOGICAL ERROR
      real*8, allocatable :: workspace(:)
      real*8, allocatable :: evalue(:)
      REAL*8  SFMIN,DLAMCH
      EXTERNAL DLAMCH,DSYEV_MO
cd,MAT2VECT,MAT2VEC
      
C     
C--Find machine parameters
      lwork = 4*nparam**2
      allocate(workspace(lwork))
      allocate(evalue(nparam))
      SFMIN = DLAMCH('S')
C
C---find eigenvalues and eigenvectors of right hand side matrix
c      CALL DSYEV_MO('V','U',NPARAM,D2F,NMAXPAR,WORKSPACE(1),
c     &    WORKSPACE(NMAXPAR+1),LWORK,INFO)
c      write(*,*)nparam,lwork
      CALL DSYEV('V','U',NPARAM,D2F(1:nparam,1:nparam),nparam,
     &     evalue,workspace,lwork,info)
C
C---Now solve equations. 0 eigenvalues are not used in the inversion
C
cd      CALL MAT2VECT(NMAXPAR,NPARAM,D2F,DF,WORKSPACE(NPARAM+1),ERROR)
      workspace(1:nparam)=matmul(transpose(d2f(1:nparam,1:nparam)),
     &  df(1:nparam))
C
      DO   I=1,NPARAM
         IF(evalue(I).GT.
     &     max(abs(evalue(NPARAM))*RANK,SFMIN)) THEN
           WORKSPACE(I) = WORKSPACE(I)/evalue(I)
         ELSE
           WORKSPACE(I) = 0.0D0
         ENDIF
      ENDDO
      shifts(1:nparam)=matmul(d2f(1:nparam,1:nparam),
     &     workspace(1:nparam))

      deallocate(workspace)
      deallocate(evalue)
      RETURN
      END
c
      SUBROUTINE  EUL2MATR(ALPHA1,ALPHA2,ALPHA3,ROT_TEMP)
C
C---Converts euler angles to rotation matrix
      REAL ROT_TEMP(3,3),ALPHA1,ALPHA2,ALPHA3
C
C---Local variables
      REAL SIN1,COS1,SIN2,COS2,SIN3,COS3

      SIN1 = SIN(ALPHA1)
      SIN2 = SIN(ALPHA2)
      SIN3 = SIN(ALPHA3)

      COS1 = COS(ALPHA1)
      COS2 = COS(ALPHA2)
      COS3 = COS(ALPHA3)
      ROT_TEMP(1,1) =  COS1*COS2*COS3 - SIN1*SIN3
      ROT_TEMP(1,2) = -COS1*COS2*SIN3 - SIN1*COS3
      ROT_TEMP(1,3) =  COS1*SIN2
      ROT_TEMP(2,1) =  SIN1*COS2*COS3 + COS1*SIN3
      ROT_TEMP(2,2) = -SIN1*COS2*SIN3 + COS1*COS3
      ROT_TEMP(2,3) =  SIN1*SIN2
      ROT_TEMP(3,1) = -SIN2*COS3
      ROT_TEMP(3,2) =  SIN2*SIN3
      ROT_TEMP(3,3) =  COS2

      RETURN
      END

      SUBROUTINE MATR2EUL(R,AL1,AL2,AL3,ERROR)
C     ========================
C
C---- This s/r works out Euler rotation angles from a given rotation matrix.
C
      REAL AL1,AL2,AL3
      REAL R(3,3)
      LOGICAL ERROR
C
      REAL EPS_LOC
      REAL RINV(3,3),RTRANS(3,3)
      DATA EPS_LOC/1.0E-6/
C
C---First check if it is rotation matrix
      ERROR = .FALSE.
      CALL MATINV3_R(R,RINV)
      CALL MATTRANS(3,3,R,RTRANS,ERROR)

      ADIFF = 0.0
      DO   I=1,3
         DO   J=1,3
           ADIFF = ADIFF + (RINV(J,I)-RTRANS(J,I))**2
         ENDDO
      ENDDO
      IF(SQRT(ABS(ADIFF)).GT.EPS_LOC) THEN
         ERROR = .TRUE.
         write(*,*)'Warning==> Matrix is not rotation matrix'
         RETURN
      ENDIF
      IF(ABS(DET3(R)-1.0).GT.EPS_LOC) THEN
         ERROR = .TRUE.
         write(*,*)'Warning==> Matrix is not rotationa matrix'
      ENDIF
C
      IF(SQRT(ABS(R(3,1)**2+R(3,2)**2)).LT.EPS_LOC) THEN
C
C---AL2 = 0.0 or pi. Either AL1+AL3 or AL1 + AL3 can be defined
        IF(R(3,3).GT.1.0-EPS_LOC) THEN
          AL2 = 0.0
        ELSE
          AL2 = 4.0*ATAN2(1.0,1.0)
        ENDIF
        AL3 = 0.0
        AL1 = ATAN2(R(2,1),R(1,1))
      ELSE
        AL3 = ATAN2(R(3,2),-R(3,1))
        IF(ABS(R(3,1)).GT.EPS_LOC) THEN
           AL2 = ATAN2(-R(3,1)/COS(AL3),R(3,3))
        ELSE
           AL2 = ATAN2(R(3,2)/SIN(AL3),R(3,3))
        ENDIF
        AL1 = ATAN2(R(2,3),R(1,3))
      ENDIF

      RETURN
      END
C
      SUBROUTINE MATR2EUL_UNSAFE(R,AL1,AL2,AL3)
C     ========================
C
C---- This s/r works out Euler rotation angles from a given rotation matrix.
C
      REAL AL1,AL2,AL3
      REAL R(3,3)
C
      REAL EPS_LOC
      DATA EPS_LOC/1.0E-8/
C
      IF(SQRT(ABS(R(3,1)**2+R(3,2)**2)).LT.EPS_LOC) THEN
C
C---AL2 = 0.0 or pi. Either AL1+AL3 or AL1 - AL3 can be defined
        IF(R(3,3).GT.0.5) THEN
          AL2 = 0.0
        ELSE
          AL2 = 4.0*ATAN2(1.0,1.0)
        ENDIF
        AL3 = 0.0
        AL1 = ATAN2(R(2,1),R(1,1))
      ELSE         
        AL3 = ATAN2(R(3,2),-R(3,1))
        IF(ABS(R(3,1)).GT.EPS_LOC) THEN
           AL2 = ATAN2(-R(3,1)/COS(AL3),R(3,3))
        ELSE
           AL2 = ATAN2(R(3,2)/SIN(AL3),R(3,3))
        ENDIF
        AL1 = ATAN2(R(2,3),R(1,3))
      ENDIF

      RETURN
      END
c
      subroutine trace_mat(np,n,amat,trace,error)
      implicit none
c
c--Calculate the trace of the matrix
      integer n,np
      real amat(np,np)
      logical error
c
c---outputs
      real trace
c
      integer i
      if(n.le.0.or.n.gt.np) then
         error = .TRUE.
         write(*,*)'Problem in trace_mat'
         return
      endif
      trace = 0.0
      do  i=1,n
         trace = trace + amat(i,i)
      enddo
      end
c
      subroutine trace_matmul(np,n,amat1,amat2,trace,error)
      implicit none
c
c--Calculate the trace of matrix multiplication
      integer n,np
      real amat1(np,np),amat2(np,np)
      logical error
c
c---  outputs
      real trace
c
      integer i,j
c
      trace = 0.0
      if(n.le.0.or.np.le.0.or.n.gt.np) then
         error = .TRUE.
         write(*,*)'Problem in trace_matmul'
         return
      endif
      do  i=1,n
         do  j=1,n
            trace = trace + amat1(i,j)*amat2(j,i)
         enddo
      enddo
      end
c
      subroutine matrix_invanddet(np,n,amat,amat_out,det,error)
c
      implicit none
      integer n,np
      real amat(np,np)
c
c--outputs
      real amat_out(np,np)
      real det
      logical error
c
c--locals
      real a11,a12,a13,a22,a23,a33
      integer lwork,info1
      integer i,j,k
      real evec(100,100),eval(100),worksp(3*100*100)
      real eps_loc
      data eps_loc/5.0e-8/
c
c--body
      lwork = 100
      error = .FALSE.
      if(n.le.0.or.np.le.0.or.n.gt.np) then
         error = .TRUE.
         write(*,*)'Error in matrix inversions'
         return
      endif
      if(n.eq.1) then
         if(abs(amat(1,1)).gt.0.0) then
            amat_out(1,1) = 1/amat(1,1)
            det = amat(1,1)
         else
            error = .TRUE.
            write(*,*)'Error  in matrix inversion. 0 determinant'
         endif
      else if(n.eq.2) then
         det = amat(1,1)*amat(2,2) - amat(1,2)*amat(2,1)
         if(abs(det).gt.0.0) then
            amat_out(1,1) = amat(2,2)/det
            amat_out(1,2) = -amat(2,1)/det
            amat_out(2,1) = amat_out(1,2)
            amat_out(2,2) = amat(1,1)/det
         else
            error = .TRUE.
            write(*,*)'Error in matrix inversion. 0 determinant'
         endif
      else if(n.eq.3) then
         a11 =   amat(2,2)*amat(3,3)-amat(2,3)*amat(3,2)
         a12 = -(amat(2,1)*amat(3,3)-amat(3,1)*amat(2,3))
         a13 =   amat(2,1)*amat(2,1)-amat(2,2)*amat(3,1)
         a22 =   amat(1,1)*amat(3,3)-amat(2,1)*amat(3,3)
         a23 = -(amat(1,1)*amat(3,2)-amat(1,2)*amat(3,1))
         a33 =   amat(1,1)*amat(2,2)-amat(1,2)*amat(2,1)
         det = amat(1,1)*a11+amat(1,2)*a12+amat(1,3)*a13
         if(abs(det).gt.0.0) then
            amat_out(1,1) = a11/det
            amat_out(1,2) = a12/det
            amat_out(1,3) = a13/det
            amat_out(2,1) = amat_out(1,2)
            amat_out(2,2) = a22/det
            amat_out(2,3) = a23/det
            amat_out(3,1) = amat_out(1,3)
            amat_out(3,2) = amat_out(2,3)
            amat_out(3,3) = a33/det
         else
            error = .TRUE.
            write(*,*)'Error in matrix inversion. 0 determinant'
         endif
      else
c
c--Use eigenvalue filtering for pseudo inversion. It is slow but should work 
c---for more general problems
         evec(1:n,1:n) = amat(1:n,1:n)
         call ssyev_mo('V','U',n,amat,np,eval,worksp,lwork,info1)
         det = 1.0
         do  i=2,n
            if(eval(i).le.eval(1)*eps_loc) then
               eval(i) = 0.0
            else
               det = det*eval(i)
               eval(i) = 1.0/sqrt(eval(i))
            endif
         enddo
         det = det*eval(1)
         eval(1) = 1.0/sqrt(eval(1))
         do  i=1,n
            do  j=1,n
               amat(i,j) = amat(i,j)*eval(i)
            enddo
         enddo
         do  i=1,n
            do  k=1,n
               amat_out(i,k) = 0.0
               do   j=1,n
                  amat_out(i,k) = amat_out(i,k) + 
     &                 amat(j,i)*amat(j,k)
               enddo
            enddo
         enddo
         amat(1:n,1:n) = evec(1:n,1:n)
      endif
      return
      end
c
      subroutine solve_linear(np,n,rank,df,df2,b,error)
c
c--   Solves linear equation
      implicit none
c
c---  Inputs
      integer np,n
      real rank
      real df(np),df2(np,np)
c
c---  Outputs
      real b(np)
      logical error
c
c--   locals
      real d1,a11,a12,a13,a22,a23,a33
      real evec(100,100),eval(100),worksp(3*100*100),btmp(100)
      integer lwork,info1
      integer i,j
c
c--   body

      lwork = 30000
      error = .FALSE.
      if(n.gt.np.or.n.le.0.or.n.gt.100) then
         write(*,*)'Error in solve_linear. Size mismatch'
         error = .TRUE.
         return
      endif
      if(n.eq.1) then
         if(abs(df2(1,1)).gt.0.0) then
            b = df(1)/df2(1,1)
            return
         else
            write(*,*)'Error in solve_linear. 0 determinant'
            error = .TRUE.
         endif
      else if(n.eq.2) then
         d1 = df2(1,1)*df2(2,2) - df2(1,2)*df2(2,1)
         if(abs(d1).gt.0.0) then
            b(1) = (df(1)*df2(2,2) - df(2)*df2(2,1))/d1
            b(2) = (df2(1,1)*df(2) - df2(1,2)*df(1))/d1
            return
         else
            write(*,*)'Error in solve_linear. 0 determinant'
            error = .TRUE.
            return
         endif
      else if(n.eq.3) then
         a11 =   df2(2,2)*df2(3,3)-df2(2,3)*df2(3,2)
         a12 = -(df2(2,1)*df2(3,3)-df2(3,1)*df2(2,3))
         a13 =   df2(2,1)*df2(2,1)-df2(2,2)*df2(3,1)
         a22 =   df2(1,1)*df2(3,3)-df2(2,1)*df2(3,3)
         a23 = -(df2(1,1)*df2(3,2)-df2(1,2)*df2(3,1))
         a33 =   df2(1,1)*df2(2,2)-df2(1,2)*df2(2,1)
         d1  = df2(1,1)*a11+df2(1,2)*a12+df2(1,3)*a13
         if(abs(d1).gt.0.0) then
            b(1) = (a11*df(1)+a12*df(2)+a13*df(3))/d1
            b(2) = (a12*df(1)+a22*df(2)+a23*df(3))/d1
            b(3) = (a13*df(1)+a23*df(2)+a33*df(3))/d1
            return
         else
            write(*,*)'Error in solve_linear. 0 determinant'
            error = .TRUE.
            return
         endif
      else
c
c--dimension is more than 3. Use eigen value filtering
         evec(1:n,1:n) = df2(1:n,1:n)
         call ssyev_mo('V','U',n,evec,100,eval,worksp,lwork,info1)
         do  i=1,n
            btmp(i) = 0.0
            do  j=1,n
               btmp(i) = btmp(i) + evec(j,i)*df(j)
            enddo
         enddo
         do  i=1,n
            if(eval(i).gt.eval(n)*rank) then
               btmp(i) = btmp(i)/eval(i)
            else
               btmp(i) = 0.0
            endif
         enddo
         do  i=1,n
            b(i) = 0
            do  j=1,n
               b(i) = b(i) + evec(i,j)*btmp(j)
            enddo
         enddo
      endif
      return
      end
c
      subroutine seig3dim(amat,eigenv,ierror)
      implicit none
c
c--input and output
      real amat(3,3)
c
c--output
      integer i,j
      integer ierror
      real eigenv(3)
c
c---locals
      real eigenvec(3,3)
      logical error
c
c--body
      ierror = 0
      call Z4_eigv3sym(amat,eigenv,eigenvec,error)
      if(error) then
         ierror = 1
         return
      endif
      do i=1,3
         do j=1,3
            amat(i,j) = eigenvec(j,i)
         enddo
      enddo
      return
      end
c
      subroutine calc_inv_determinant(np,amat,a_inv,det)
      implicit none
c
      integer np
      integer ndim
      real amat(np,np),a_inv(np,np)
      real det
c
      real toler,emax,sfmin
      integer i,j,info
      integer nwork
      real, allocatable :: work(:)
      real, allocatable :: eigen(:)
      real, allocatable :: einv(:)
      real, allocatable :: evector(:,:)
c
      real slamch
      external slamch
c
c---body
      toler = 1.0e-32
      if(np.eq.1) then
         if(abs(amat(1,1)).gt.0.0) then
            det = amat(1,1)
            a_inv(1,1) = 1.0/amat(1,1)
            ndim = 1
         else 
            det = 1.0
            ndim = 0
         endif
      else
         sfmin = slamch('S')
         nwork = 2.0*np*np
         allocate(work(nwork))
         allocate(eigen(np))
         allocate(einv(np))
         allocate(evector(np,np))
         evector(1:np,1:np) = amat(1:np,1:np)
         info = 0
         call ssyev('N','U',np,evector,np,eigen,work,nwork,info)
         det = 1.0
         emax = max(maxval(abs(eigen(1:np)))*toler,sfmin)
c
c---Analyse cases with 0 eigenvalues. We should calculate determinant
c---  corresponding to non-zero eigenvalue. Dimensioality also should be return
c
        do   i=1,np
            if(eigen(i).gt.emax) then
               einv(i) = 1.0/eigen(i)
            else
               write(*,*)'Error==> matrix has zero determinant'
               stop
c               einv(i) = 0.0
c               stop
            endif
         enddo
         do i=1,np
            do j=1,np
               a_inv(i,j) = evector(j,i)*einv(i)
            enddo
         enddo
         a_inv(1:np,1:np) = matmul(evector,a_inv(1:np,1:np))
c
         ndim = 0
         do i=1,np
            if(abs(eigen(i)).gt.emax) then
               det = det*eigen(i)
               ndim = ndim + 1
            endif
         enddo
         deallocate(eigen)
         deallocate(einv)
         deallocate(evector)
      endif
c
      end
c
      real function calc_determinant(np,amat)
      implicit none
c
      integer np
      real amat(np,np)
c
      integer i,info
      integer nwork
      real, allocatable :: work(:)
      real, allocatable :: eigen(:)
c
c---body
      if(np.eq.1) then
         calc_determinant = amat(1,1)
      else if(np.eq.2) then
         calc_determinant = amat(1,1)*amat(2,2)-amat(1,2)**2
c      else if(np.eq.3) then
      else
         nwork = np*np
         allocate(work(nwork))
         allocate(eigen(np))
         info = 0
         call ssyev('N','U',np,amat,eigen,work,nwork,info)
         calc_determinant = 1.0
c
c---Analyse cases with 0 eigenvalues. We should calculate determinant
c---corresponding to non-zero eigenvalue. Dimensioality also should be returned.
         do i=1,np
            calc_determinant = calc_determinant*eigen(i)
         enddo
      endif
c
      end


