// tabulated sin,cos; I0,I1 on <0,200000> and exp on <-50,50>
// the (relative) precision in worst case is 10^-6 - don't use if highly precise results are requested
// approx. 2-15 x quicker than standard C implementations (depending on platform, compiler, optimization, function etc)
// Pavol Skubak, 2003
#include <cstdlib>
#include <cstring>
#include <math.h>
#include <fstream>
#include <iostream>


using namespace std;

const double PI = 3.14159265358979323846 ;
const double TWO_PI = 2 * 3.14159265358979323846 ;
const double HALF_PI = 0.5 * 3.14159265358979323846 ;
const double TWO_PI_INV = 1 / (2.*3.14159265358979323846) ;
const double PI_INV = 1 / 3.14159265358979323846 ;

// using the commented declarations if linking against fortran main etc
extern "C" double i0( double, int, int );
extern "C" double i0e( double, int, int );
extern "C" double i1( double, int, int );
extern "C" double i1e( double, int, int );
//double i0( double, int, int );
//double i0e( double, int, int );
//double i1( double, int, int );
//double i1e( double, int, int );

template <typename realnum>
class TabFunc
{

  public:
	TabFunc();
	~TabFunc();
	TabFunc( TabFunc<realnum>& );
	TabFunc& operator=( TabFunc<realnum>& );
  
	// ExpM(x) = exp(-x)
	realnum ExpM(realnum x)
	{
	  if ( x<-50. || x>50. ) return exp(-x);
	  return ExpM_nocheck(x);
	}

	// if you want to check whether the point is in <-50,50> region yourself or you now it must be in
	realnum ExpM_nocheck(realnum x)
	{
  	  poin = Exparr + (int) ( 250.5 + x*5. ); 
  	  x_diff = x - *poin;
  	  x_diff_sq = x_diff*x_diff;
  	  return ( *(poin+dimexp1)) * ( 1. - x_diff + 0.5*x_diff_sq -
		0.16666666666666*x_diff_sq*x_diff + 0.041666666666666*x_diff_sq*x_diff_sq ) ;
	}


	realnum Sin(realnum x)
	{
	  ChargeTrig(x);
	  return Sin_charged(x);
	}

	// use only if Cos(x) was called just before (saves time needed for table lookup)
	realnum Sin_charged(realnum x)
	{
  	  return ( *(poin+dimsincos1)) * ( 1. - 0.5*x_diff_sq + 0.041666666666666*x_diff_sq*x_diff_sq ) +
             ( *(poin+2*dimsincos1)) * ( x_diff - 0.16666666666666*x_diff_sq*x_diff ) ;
	}

	realnum Cos(realnum x)
	{
	  ChargeTrig(x);
  	  return Cos_charged(x);
	}

	// use only if Sin(x) was called just before (saves time needed for table lookup)
	realnum Cos_charged(realnum x)
	{
  	  return (*(poin+2*dimsincos1)) * ( 1. - 0.5*x_diff_sq + 0.041666666666666*x_diff_sq*x_diff_sq )+
    	(*(poin+dimsincos1)) * ( - x_diff + 0.16666666666666*x_diff_sq*x_diff ) ;
	}
  
	realnum I0e(realnum x)
	{
	  if ( x < 0. ) return I0e_nocheck(-x);
	  if ( x > 200000. ) return i0e(x,0,0);  
	  return I0e_nocheck(x);
	}

	// if you want to check whether the point is in <0,50> region yourself or you now it must be in
	realnum I0e_nocheck(realnum x)
	{
	  ChargeBess(x);
  	  return I0e_nocheck_charged(x);
	}

	// use only if I1e(x) was called just before (saves time needed for table lookup)
	realnum I0e_nocheck_charged(realnum x)
	{
	  poin = Bess0arr + i;
  	  return ( *(poin+dimbess1) + *(poin+2*dimbess1)*x_diff + *(poin+3*dimbess1)*x_diff_sq
          + *(poin+4*dimbess1)*x_diff_sq*x_diff + *(poin+5*dimbess1)*x_diff_sq*x_diff_sq );
	}

	realnum Sim(realnum x)
	{
  	  return I1e(x)/I0e(x);
	}

	realnum Simoverx(realnum x)
	{
  	  return I1e(x)/(x*I0e(x));
	}

	// use these two versions if you already computed I0e(x) and pass this value as the second argument
	realnum Sim_preI0(realnum x, realnum I0e_x_)
	{
  	  return I1e(x)/I0e_x_;
	}

	realnum Simoverx_preI0(realnum x,realnum I0e_x_)
	{
  	  return I1e(x)/(x*I0e_x_);
	}
  

	realnum I1e(realnum x)
	{
	  if ( x<0. ) return -I1e_nocheck(-x);
	  if ( x>200000. ) return i1e(x,0,0);
  	  return I1e_nocheck(x);
	}

	realnum I1e_nocheck(realnum x)
	{
	  ChargeBess(x);
  	  return I1e_nocheck_charged(x);
	}
  
	// use only if I0e(x) was called just before (saves time needed for table lookup)
	realnum I1e_nocheck_charged(realnum x)
	{
	  poin = Bess1arr + i;
  	  return ( *(poin+dimbess1) + *(poin+2*dimbess1)*x_diff + *(poin+3*dimbess1)*x_diff_sq
	    	 + *(poin+4*dimbess1)*x_diff_sq*x_diff + *(poin+5*dimbess1)*x_diff_sq*x_diff_sq );
	}

	realnum GetBess0coef(int i,int j)	{ return Bess0arr[i+j*dimbess1]; }
	realnum GetBess1coef(int i,int j)	{ return Bess1arr[i+j*dimbess1]; }
	realnum GetExpcoef(int i,int j)		{ return Exparr[i+j*dimexp1]; }
	realnum GetTrigcoef(int i,int j)	{ return SinCosarr[i+j*dimsincos1]; }
	unsigned GetBessdim(int i)			{ if (i==1) return dimbess1; else if (i==2) return dimbess2; else Error();   return 0; }
	unsigned GetExpdim(int i)			{ if (i==1) return dimexp1; else if (i==2) return dimexp2; else Error();   return 0; }
	unsigned GetTrigdim(int i)			{ if (i==1) return dimsincos1; else if (i==2) return dimsincos2; else Error();   return 0; }

  private:

	unsigned i;

	unsigned dimbess1, dimbess2, dimexp1, dimexp2;
	unsigned dimsincos1, dimsincos2;
	realnum x_diff, x_diff_sq;
	realnum *poin;
	realnum *Bess0arr;
	realnum *Bess1arr;
	realnum *Exparr;
	realnum *SinCosarr;

	void ChargeBess(realnum x)
	{
  	  if ( x < 700. )
    	if ( x < 30. )
		  if ( x < 6. ) i = (int) ( .5 + x*5. );
		  else i = (int) ( 30.5 + (x-6.)*1.25 );
    	else
		  if ( x < 140. ) i = (int) ( 60.5 + (x-30.)*0.25 );
		  else i = (int) ( 88.5 + (x-140.)*0.05 );
  	  else
    	if ( x < 30000. )
      	  if ( x < 4000. ) i = (int) ( 116.5 + (x-700.)*0.01 );
      	  else i = (int) ( 149.5 + (x-4000.)*0.002 );
    	else
      	  if ( x < 86000. ) i = (int) ( 201.5 + (x-30000.)*0.001 );
      	  else i = (int) ( 257.5 + (x-86000.)*0.0005 );	  
// using the fact that bessel0 and bessel1 are sampled in the same points here
  	  x_diff = x - *(Bess0arr+i);
  	  x_diff_sq = x_diff*x_diff;
	}

	void ChargeTrig(realnum x)
	{
  	  if ( x > TWO_PI )
    	x -= TWO_PI * (realnum) ((int) (x*TWO_PI_INV));
  	  if ( x < 0. )
    	x -= TWO_PI * (realnum) ((int) (x*TWO_PI_INV)-1);
  	  poin = SinCosarr + (int) ( 0.5 + x*4.7755491881566 );
  	  x_diff = x - *poin;
  	  x_diff_sq = x_diff*x_diff;
	}

    void GetTab();
    void ReadTab();

	void Alloc();
	void DeAlloc();

	void Error() { cerr << "Wrong dimension number requested (only 1 or 2 allowed)" << endl; exit(1); }
};





template <typename realnum>
void TabFunc<realnum>::Alloc()
{
  Bess0arr        = new realnum[dimbess1*dimbess2];
  Bess1arr        = new realnum[dimbess1*dimbess2];
  Exparr          = new realnum[dimexp1*dimexp2];
  SinCosarr       = new realnum[dimsincos1*dimsincos2];
}


template <typename realnum>
void TabFunc<realnum>::DeAlloc()
{
  delete[] SinCosarr;
  delete[] Exparr;
  delete[] Bess1arr;
  delete[] Bess0arr;
}




template <typename realnum>
TabFunc<realnum>::TabFunc()
{
  dimbess1        = 315;
  dimbess2        = 6;
  dimexp1         = 501;
  dimexp2         = 2;
  dimsincos1      = 32;
  dimsincos2      = 3;

  Alloc();  
  GetTab();
}


template <typename realnum>
TabFunc<realnum>::~TabFunc()
{
  DeAlloc();  
}


template <typename realnum>
TabFunc<realnum>::TabFunc( TabFunc<realnum>& tab_in )
{
  dimbess1        = tab_in.dimbess1;
  dimbess2        = tab_in.dimbess2;
  dimexp1         = tab_in.dimexp1;
  dimexp2         = tab_in.dimexp2;
  dimsincos1      = tab_in.dimsincos1;
  dimsincos2      = tab_in.dimsincos2;

  Alloc();
  GetTab();
}


template <typename realnum>
TabFunc<realnum>& TabFunc<realnum>::operator=( TabFunc<realnum>& tab_in )
{
  // this obviously cannot happen with the actual version but maybe useful for the future versions of TabFunc
  if ( dimbess1!=tab_in.dimbess1 || dimbess2!=tab_in.dimbess2 || dimexp1!=tab_in.dimexp1 ||
	  dimexp2!=tab_in.dimexp2 || dimsincos1!=tab_in.dimsincos1 || dimsincos2!=tab_in.dimsincos2 )
  {
	DeAlloc();
	dimbess1        = tab_in.dimbess1;
	dimbess2        = tab_in.dimbess2;
	dimexp1         = tab_in.dimexp1;
	dimexp2         = tab_in.dimexp2;
	dimsincos1      = tab_in.dimsincos1;
	dimsincos2      = tab_in.dimsincos2;
	Alloc();
  }
  
  GetTab();  
}



template <typename realnum>
class likelihood;

template<typename realnum>
void TabFunc<realnum>::ReadTab()
{
  char* func_name = "TabFunc::ReadTab";
  fstream f;
  char filename[10] = "Bess0.tab";
  char path[200] = "\0";  		//strcpy( path, getenv("CPROG"));  strcat( path, "\\refmac5N_\\");
  char path_back[200];  strcpy( path_back, path);
  strcat(path, filename);
  f.open(path); if (!f) likelihood<realnum>::Error( 112, func_name, (void*) &path );
  f.ignore(10000, (int)'\n');
  for ( int i=0; i<dimbess1; i++ )
  {
	for ( int j=0; j<dimbess2; j++ )	f >> Bess0arr[i+j*dimbess1];
	f.ignore(10000, (int)'\n');
  }
  f.close();

//  if ( do_der_Fph || do_der_D )
//  {
	strcpy( filename, "Bess1.tab" );
	strcpy(path,path_back); strcat(path, filename);
	f.open(path); if (!f) likelihood<realnum>::Error( 112, func_name, (void*) &path );
	f.ignore(10000, (int)'\n');
	for ( int i=0; i<dimbess1; i++ )
	{
	  for ( int j=0; j<dimbess2; j++ )	f >> Bess1arr[i+j*dimbess1];
	  f.ignore(10000, (int)'\n');
	}
	f.close();
//  }

  strcpy( filename, "Exp.tab" );
  strcpy(path,path_back);  strcat(path, filename);
  f.open(path); if (!f) likelihood<realnum>::Error( 112, func_name, (void*) &path );
  f.ignore(10000, (int)'\n');
  for ( int i=0; i<dimexp1; i++ )
  {
	for ( int j=0; j<dimexp2; j++ )	f >> Exparr[i+j*dimexp1];
	f.ignore(10000, (int)'\n');
  }
  f.close();
}


template <typename realnum>
void TabFunc<realnum>::GetTab()
{  

  realnum i0e_coef[315][6] =	{
      0. , 1, -1, 0.75, -0.4166666666666667, 0.18229166666666666, 
      0.2, 0.8269385516343292, -0.7446554281055171, 0.5389476192834868, -0.29415190687015436, 0.12733806402422634, 
      0.4, 0.6974021705595334, -0.5606389462637041, 0.3896849158939176, -0.2084661209228444, 0.08918524980225802, 
      0.6, 0.5993272030798961, -0.42716278353837633, 0.2836924339204433, -0.14837311846547452, 0.06264432264760576, 
      0.8, 0.5241489419203691, -0.3296502485560314, 0.2080885652033204, -0.10610144245746096, 0.044141362489204125, 
      1. , 0.4657596075936404, -0.2578491922439319, 0.15389398456907769, -0.07626738330347096, 0.031211735558889665, 
      1.2, 0.4197820789351329, -0.20452521955814978, 0.11483486148440682, -0.05513487810949037, 0.02215352207997983, 
      1.4, 0.38306251539548697, -0.16455492300571137, 0.0865164971522201, -0.04010638138621177, 0.015789794664225275, 
      1.6, 0.35331499772134534, -0.13429550780505198, 0.06585191720621028, -0.02937223078431573, 0.011305411799855279, 
      1.8, 0.32887194969981887, -0.11114567091413166, 0.05066614902921854, -0.02166887559119004, 0.008134860720345838, 
      2. , 0.308508322553671, -0.09323903330473335, 0.03942171099249894, -0.016111952666315606, 0.005885102126255542, 
      2.2, 0.2913173330690491, -0.07922960037962967, 0.031027842950216156, -0.012080877926051337, 0.004282434666941622, 
      2.4, 0.27662232306691364, -0.06814123438733102, 0.02470767424575132, -0.009138935179182267, 0.0031358580543093227, 
      2.6, 0.26391399587590286, -0.059261741550875824, 0.019905538796062924, -0.006977854744274259, 0.0023117798951548738, 
      2.8, 0.25280553370155967, -0.052068122397779526, 0.016222156093533086, -0.005379309846926458, 0.0017165301661618212, 
      3. , 0.2430003541618254, -0.04617364086452452, 0.013369188648307717, -0.004188116432186112, 0.0012842555674301773, 
      3.2, 0.23426883166978596, -0.041290208728257755, 0.011137298893643968, -0.0032935414181482076, 0.0009685253521760327, 
      3.4, 0.22643140123345643, -0.03720155005600448, 0.009373630765202728, -0.0026162378088017066, 0.0007365009253126522, 
      3.6, 0.21934622445812324, -0.033743976026855554, 0.00796588596695727, -0.0020990913755299587, 0.0005648788485284831, 
      3.8, 0.2129001308125339, -0.030792549745461195, 0.006831025920846376, -0.0017007915984152203, 0.00043706235143360465, 
      4. , 0.2070019212239867, -0.028251081721551385, 0.005907226783746972, -0.0013913037712618773, 0.00034118521943556426, 
      4.2, 0.20157738405263376, -0.026044857966172064, 0.005148128670164731, -0.0011486707317821532, 0.0002687280888120265, 
      4.4, 0.1965655589359362, -0.024115325221309658, 0.004518707753738457, -0.000956746659728093, 0.00021354721563793082, 
      4.6, 0.19191591515026402, -0.022416183929628436, 0.003992300101298488, -0.0008035858869637896, 0.00017119099558662865, 
      4.8, 0.18758620420007516, -0.020910498414471096, 0.00354844572847067, -0.0006802932519828428, 0.00013841764379510582, 
      5. , 0.18354081260932836, -0.019568545664786025, 0.0031713189703317934, -0.0005802006097868281, 0.00011285381550045492, 
      5.2, 0.17974948824793158, -0.018366203200092523, 0.0028485796378003025, -0.0004982745326015912, 0.0000927522118843319, 
      5.4, 0.17618634750697945, -0.017283732536518437, 0.0025705274466609335, -0.00043068842669644405, 0.000076818883208765, 
      5.6, 0.17282909512230335, -0.01630485462620551, 0.002329476010482487, -0.000374511988039318, 0.00006408974163783603, 
      5.8, 0.16965840616909, -0.015416042077806308, 0.002119286552695651, -0.00032748471264089016, 0.000053841920261392916, 
      6. , 0.16665743263981653, -0.01460597333131064, 0.0019350183889351447, -0.0002878498582292201, 0.00004552988438388198, 
      6.8, 0.156080172005745, -0.011982572838872885, 0.001387161135426404, -0.00017969874776189998, 0.00002467147011532729, 
      7.6, 0.14729896357061975, -0.010063930258551013, 0.001035309645914903, -0.00011892239131183063, 0.000014436560702326198, 
      8.4, 0.13985490269028533, -0.008609510400554105, 0.0007972846690224855, -0.00008232881653934959, 8.967344814730938e-6, 
      9.2, 0.13343877406765084, -0.007475423910670714, 0.000629589663008745, -0.00005907584273088032, 5.839862212674192e-6, 
     10. , 0.12783333716342862, -0.0065706557789730335, 0.0005075217097502555, -0.00004364979820152201, 3.951896059613096e-6, 
     10.8, 0.12288078404316922, -0.005835027668654111, 0.0004162426513154502, -0.000033048921853812835, 2.760780680815221e-6, 
     11.6, 0.11846329417392376, -0.0052272882996373254, 0.0003464259774698039, -0.00002554603973514355, 1.9812416367339172e-6, 
     12.4, 0.11449085939276077, -0.004718298270600649, 0.00029198532212645206, -0.000020101054491607206, 1.4549820831141758e-6, 
     13.2, 0.11089336208993855, -0.004286943034842186, 0.00024882110093702586, -0.00001606325036232828, 1.0901084423762045e-6, 
     14. , 0.10761525167069508, -0.003917584207552338, 0.0002140960838686659, -0.000013012052493609899, 8.311932295354194e-7, 
     14.8, 0.10461186708079329, -0.003598416520779482, 0.00018579994780604236, -0.000010667873186983451, 6.436838888351434e-7, 
     15.6, 0.10184683506133191, -0.0033203778508996556, 0.00016247858133450893, -8.840216323847266e-6, 5.054128640715264e-7, 
     16.4, 0.09929019059813116, -0.0030764078492835217, 0.00014306081425767558, -7.396434191900975e-6, 4.0179350062081315e-7, 
     17.2, 0.09691699379921238, -0.002860932449479228, 0.00012674461954512162, -6.24235420975533e-6, 3.230089370568137e-7, 
     18. , 0.09470629521276407, -0.002669498340743398, 0.00011292064985393702, -5.3099478395240984e-6, 2.623178146053673e-7, 
     18.8, 0.09264035031183757, -0.0024985091923938607, 0.00010111980091928596, -4.549296592425492e-6, 2.1500585266212915e-7, 
     19.6, 0.09070401516106917, -0.0023450322713158417, 0.000090976585352745, -3.923246761211585e-6, 1.7772156975237596e-7, 
     20.4, 0.08888427583569954, -0.002206654621160792, 0.00008220312080444575, -3.4037849458955216e-6, 1.4804631481846187e-7, 
     21.2, 0.08716987794315037, -0.002081374696615443, 0.00007457037476320261, -2.9695376503584975e-6, 1.2421051164565947e-7, 
     22. , 0.08555103200662495, -0.001967519732201259, 0.00006789445323709259, -2.604018948554332e-6, 1.0490305136622718e-7, 
     22.8, 0.08401917700322054, -0.0018636820400887988, 0.00006202644879204494, -2.2943845375423405e-6, 8.914116086640791e-8, 
     23.6, 0.08256678895488526, -0.0017686694023400784, 0.000056844835548873485, -2.0305339759104507e-6, 7.61802225533563e-8, 
     24.4, 0.08118722476358643, -0.001681466076768015, 0.00005224971023485431, -1.8044558029867153e-6, 6.545041191831358e-8, 
     25.2, 0.07987459386710019, -0.0016012018790757115, 0.00004815838724982896, -1.6097443555230379e-6, 5.651160550493653e-8, 
     26. , 0.0786236520400137, -0.0015271274703474447, 0.000044501997853857445, -1.4412394690674346e-6, 4.9020907460020635e-8, 
     26.8, 0.0774297129601972, -0.0014585944555672997, 0.00004122284167495244, -1.2947551464720375e-6, 4.270900340851147e-8, 
     27.6, 0.07628857413123268, -0.001395039243664159, 0.00003827230729516046, -1.1668733307570317e-6, 3.736276290742732e-8, 
     28.4, 0.07519645448485228, -0.0013359698717732155, 0.00003560922717674858, -1.0547858020262119e-6, 3.281231514057001e-8, 
     29.2, 0.07414994154706872, -0.00128095518270889, 0.00003319856688080436, -9.561719810109032e-7, 2.8921359869916313e-8, 
     30. , 0.0731459464822373, -0.0012296158835897186, 0.00003101037361226411, -8.691037619579234e-7, 2.5579841525340263e-8, 
     34. , 0.06867386860528439, -0.001017564748496222, 0.00002261910354346003, -5.587267186083628e-7, 1.4493317022652025e-8, 
     38. , 0.06493310980894874, -0.00086015832400059, 0.000017093172882849672, -3.774561303637042e-7, 8.752715972577124e-9, 
     42. , 0.06174385590478495, -0.0007395294301855634, 0.000013287448345097874, -2.652889873038661e-7, 5.5618729357352625e-9, 
     46. , 0.05898266978620319, -0.0006446787443469895, 0.000010570146065942004, -1.925769107097297e-7, 3.684219604740968e-9, 
     50. , 0.0565616266474542, -0.0005685027545588028, 8.571515629850202e-6, -1.4360308271156585e-7, 2.5262906805552376e-9, 
     54. , 0.054416172411584675, -0.0005062306009277207, 7.0644730512624265e-6, -1.0954403856060224e-7, 1.7836413322876166e-9, 
     58. , 0.052497698732771415, -0.0004545516586273654, 5.9038390226758e-6, -8.52041191663766e-8, 1.291200328948361e-9, 
     62. , 0.050768811626071025, -0.0004111041332502674, 4.993588953325356e-6, -6.73978937798525e-8, 9.551784458869395e-10, 
     66. , 0.04920021011715339, -0.0003741626310236404, 4.268331886294469e-6, -5.4103667818453105e-8, 7.201086320123007e-10, 
     70. , 0.04776856505404178, -0.00034244043088386966, 3.6823978613112485e-6, -4.3999285622917106e-8, 5.520288380057858e-10, 
     74. , 0.046455040130573846, -0.00031496047139844413, 3.2031764040166277e-6, -3.619708786005427e-8, 4.2950313141124075e-10, 
     78. , 0.04524423729285103, -0.0002909688993094975, 2.8069224278226557e-6, -3.008715654569116e-8, 3.38634116841402e-10, 
     82. , 0.044123429733054574, -0.0002698757620950218, 2.476042759904512e-6, -2.524168700099305e-8, 2.70194498719345e-10, 
     86. , 0.04308199415503663, -0.00025121346446581316, 2.197297660169739e-6, -2.135502391847205e-8, 2.1792563391874464e-10, 
     90. , 0.042110983880794116, -0.00023460712032958436, 1.960582771447922e-6, -1.8205125589323967e-8, 1.7750008494271663e-10, 
     94. , 0.04120280328449694, -0.00021975303612745828, 1.7580879978322528e-6, -1.5628246088743303e-8, 1.4587324555746795e-10, 
     98. , 0.04035095630360352, -0.00020640285661824992, 1.5837063785280325e-6, -1.3501966055623482e-8, 1.2086891253496046e-10, 
    102. , 0.039549849897336176, -0.00019435172067976125, 1.4326119706607743e-6, -1.1733610902342484e-8, 1.009089201087221e-10, 
    106. , 0.038794638805470745, -0.00018342930071706814, 1.3009539965302963e-6, -1.0252214408153215e-8, 8.483357874684849e-11, 
    110. , 0.038081101727626514, -0.00017349294490127953, 1.1856322525288787e-6, -9.00285669538231e-9, 7.178020265848073e-11, 
    114. , 0.03740554167365303, -0.00016442237269977122, 1.0841301517315605e-6, -7.94261944062069e-9, 6.109979015789246e-11, 
    118. , 0.0367647050995827, -0.00015611553143271562, 9.943891947916672e-7, -7.037658788199008e-9, 5.2298995212654575e-11, 
    122. , 0.03615571578235553, -0.00014848533020007937, 9.147135928803951e-7, -6.2610616049368906e-9, 4.4999102100790056e-11, 
    126. , 0.03557602036018383, -0.00014145704358487637, 8.436970904378283e-7, -5.591258187345738e-9, 3.890678621290555e-11, 
    130. , 0.03502334318182804, -0.00013496623150171388, 7.801663081501153e-7, -5.01083457019047e-9, 3.3792907274537987e-11, 
    134. , 0.034495648640881144, -0.00012895706031151172, 7.231365034172932e-7, -4.505635463683544e-9, 2.947706228412228e-11, 
    138. , 0.03399110957144811, -0.00012338093846552683, 6.717767518163442e-7, -4.064080448909491e-9, 2.5816168213593782e-11, 
    140. , 0.03374700269661267, -0.00012074178758821053, 6.479986274093852e-7, -3.864122730123931e-9, 2.419467924619223e-11, 
    160. , 0.03156388368559873, -0.00009879222982766417, 4.638190283789978e-7, -2.4195466567789037e-9, 1.3252900513127663e-11, 
    180. , 0.02975611642715961, -0.00008277132217423058, 3.4536354927110213e-7, -1.6011510641650295e-9, 7.794333754806981e-12, 
    200. , 0.02822715994911191, -0.00007065655427899303, 2.65295791911091e-7, -1.1067948840412511e-9, 4.848356742123694e-12, 
    220. , 0.02691200452415582, -0.00006123346898292623, 2.0898931207997962e-7, -7.925340392636593e-10, 3.1557488826966518e-12, 
    240. , 0.025765057339283914, -0.000053733351047097894, 1.680927382702313e-7, -5.842684374507412e-10, 2.1323880161627784e-12, 
    260. , 0.02475327065060642, -0.00004764839225025394, 1.3758021495322015e-7, -4.41389525623892e-10, 1.4868878697726906e-12, 
    280. , 0.023852026277278286, -0.000042631070028005535, 1.1429287220289519e-7, -3.4046325914777187e-10, 1.064904458017435e-12, 
    300. , 0.023042558415085467, -0.00003843637481651743, 9.617141606861612e-8, -2.673669423879088e-10, 7.804767032312809e-13, 
    320. , 0.022310278358793567, -0.00003488712968736929, 8.183089189074677e-8, -2.1326881536817832e-10, 5.836159463687429e-13, 
    340. , 0.02164365037803266, -0.00003185237036834945, 7.031447472535812e-8, -1.7246687281438633e-10, 4.441782228484237e-13, 
    360. , 0.021033416521385517, -0.000029233421960393746, 6.094543341349234e-8, -1.4117591167275845e-10, 3.433756100637564e-13, 
    380. , 0.020472049172677857, -0.000026954675284095175, 5.323515594492234e-8, -1.1682096739615289e-10, 2.69169560772331e-13, 
    400. , 0.019953356281939987, -0.00002495732303644041, 4.682433781162676e-8, -9.761197536026223e-11, 2.1365906006237667e-13, 
    420. , 0.01947219137431925, -0.000023195011505047075, 4.144440646051961e-8, -8.228014580433096e-11, 1.7151937422438257e-13, 
    440. , 0.019024236981152645, -0.000021630762341795506, 3.689163860162714e-8, -6.991040428852568e-11, 1.3910498187358344e-13, 
    460. , 0.018605840488835717, -0.000020234754800001198, 3.300943691823821e-8, -5.983234697623518e-11, 1.1387364575593528e-13, 
    480. , 0.018213888042629182, -0.000018982702365204912, 2.967596909701775e-8, -5.154772442228506e-11, 9.401463982317004e-14, 
    500. , 0.017845706500153165, -0.000017854647255104855, 2.6795402207832608e-8, -4.468142002223008e-11, 7.822974039395358e-14, 
    520. , 0.017498986346742507, -0.00001683405341206673, 2.429159155770133e-8, -3.894761900144483e-11, 6.556766848214712e-14, 
    540. , 0.01717172047417229, -0.00001590711584213353, 2.210347331034479e-8, -3.412614765427313e-11, 5.5322174792593604e-14, 
    560. , 0.0168621551036, -0.000015062228867386562, 2.018165780499903e-8, -3.0045688159674455e-11, 4.6969589675982526e-14, 
    580. , 0.0165687501046932, -0.000014289572546462292, 1.8485880819152367e-8, -2.6571661537744262e-11, 4.010377099850082e-14, 
    600. , 0.016290146656305977, -0.000013580787966309404, 1.6983076027085597e-8, -2.3597464694075088e-11, 3.442612948184598e-14, 
    620. , 0.0160251406956194, -0.00001292872007615381, 1.5645902328928485e-8, -2.1037945691082527e-11, 2.9703236398281874e-14, 
    640. , 0.015772660969824485, -0.00001232721235622583, 1.4451608203527866e-8, -1.8824573410486378e-11, 2.5747958452837505e-14, 
    660. , 0.01553175077621789, -0.000011770941629025977, 1.3381148277098742e-8, -1.69018085117989e-11, 2.2415988336355053e-14, 
    680. , 0.015301552679867545, -0.000011255284222546486, 1.2418490454718656e-8, -1.5224345655678873e-11, 1.9595111123982267e-14, 
    700. , 0.015081295651531356, -0.000010776206814508762, 1.155006828113675e-8, -1.375500321282308e-11, 1.7198699062137557e-14, 
    800. , 0.01410694500587, -8.81959934e-6, 8.27096e-9, -8.62e-12, 1.e-14, 
    900. , 0.01329992412407, -7.39090148e-6, 6.1608e-9, -5.71e-12, 6.e-15, 
   1000. , 0.01261724045589, -6.31019896e-6, 4.73383e-9, -3.95e-12, 3.e-15, 
   1100. , 0.012029929919, -5.46939386e-6, 3.72998e-9, -2.83e-12, 2.e-15, 
   1200. , 0.01151767184432, -4.80003057e-6, 3.00064e-9, -2.08e-12, 0, 
   1300. , 0.01106573243217, -4.25687004e-6, 2.45636e-9, -1.57e-12, 0, 
   1400. , 0.01066313329436, -3.80894242e-6, 2.04087e-9, -1.22e-12, 0, 
   1500. , 0.01030150409652, -3.43440739e-6, 1.71749e-9, -9.5e-13, 0, 
   1600. , 0.00997433646829, -3.11746748e-6, 1.46154e-9, -7.6e-13, 0, 
   1700. , 0.0096764832224, -2.84644326e-6, 1.25597e-9, -6.2e-13, 0, 
   1800. , 0.00940381292718, -2.61253326e-6, 1.08871e-9, -5.e-13, 0, 
   1900. , 0.00915296591057, -2.40899234e-6, 9.5104e-10, -4.2e-13, 0, 
   2000. , 0.00892117827644, -2.2305735e-6, 8.3657e-10, -3.5e-13, 0, 
   2100. , 0.0087061526069, -2.07314037e-6, 7.405e-10, -2.9e-13, 0, 
   2200. , 0.00850596138603, -1.93339282e-6, 6.5919e-10, -2.5e-13, 0, 
   2300. , 0.00831897378823, -1.80866922e-6, 5.8985e-10, -2.1e-13, 0, 
   2400. , 0.00814379943196, -1.69680169e-6, 5.3031e-10, -1.8e-13, 0, 
   2500. , 0.00797924464011, -1.59600858e-6, 4.7885e-10, -1.6e-13, 0, 
   2600. , 0.00782427804809, -1.50481359e-6, 4.3412e-10, -1.4e-13, 0, 
   2700. , 0.00767800328677, -1.42198416e-6, 3.9503e-10, -1.2e-13, 0, 
   2800. , 0.0075396370822, -1.34648402e-6, 3.607e-10, -1.1e-13, 0, 
   2900. , 0.00740849154695, -1.27743628e-6, 3.304e-10, -0.9e-13, 0, 
   3000. , 0.00728395974655, -1.21409449e-6, 3.0355e-10, -8.e-14, 0, 
   3100. , 0.00716550384867, -1.15581966e-6, 2.7966e-10, -8.e-14, 0, 
   3200. , 0.00705264532598, -1.102062e-6, 2.583e-10, -7.e-14, 0, 
   3300. , 0.00694495680482, -1.0523459e-6, 2.392e-10, -6.e-14, 0, 
   3400. , 0.00684205524276, -1.0062586e-6, 2.22e-10, -5.e-14, 0, 
   3500. , 0.00674359618628, -9.634397e-7, 2.065e-10, -5.e-14, 0, 
   3600. , 0.00664926891215, -9.235737e-7, 1.924e-10, -4.e-14, 0, 
   3700. , 0.00655879229616, -8.863832e-7, 1.797e-10, -4.e-14, 0, 
   3800. , 0.00647191128392, -8.516233e-7, 1.681e-10, -4.e-14, 0, 
   3900. , 0.00638839386274, -8.190774e-7, 1.575e-10, -3.e-14, 0, 
   4000. , 0.00630802845251, -7.885529e-7, 1.479e-10, -3.e-14, 0, 
   4500. , 0.00594724560451, -6.608418e-7, 1.101e-10, -2.e-14, 0, 
   5000. , 0.00564203689874, -5.642319e-7, 8.46e-11, 1.e-14, 0, 
   5500. , 0.0053794588817, -4.890639e-7, 6.67e-11, 1.e-14, 0, 
   6000. , 0.00515043000209, -4.292204e-7, 5.37e-11, 0, 0, 
   6500. , 0.00494836515387, -3.806581e-7, 4.39e-11, 0, 0, 
   7000. , 0.00476835742465, -3.406091e-7, 3.65e-11, 0, 0, 
   7500. , 0.00460666544185, -3.071213e-7, 3.07e-11, 0, 0, 
   8000. , 0.00446037998763, -2.787825e-7, 2.61e-11, 0, 0, 
   8500. , 0.00432720021511, -2.545487e-7, 2.25e-11, 0, 0, 
   9000. , 0.00420527927953, -2.336331e-7, 1.95e-11, 0, 0, 
   9500. , 0.00409311529349, -2.154328e-7, 1.7e-11, 0, 0, 
  10000. , 0.0039894726746, -1.994786e-7, 1.5e-11, 0, 0, 
  10500. , 0.00389332435649, -1.854008e-7, 1.32e-11, 0, 0, 
  11000. , 0.00380380862313, -1.729043e-7, 1.18e-11, 0, 0, 
  11500. , 0.0037201963853, -1.617512e-7, 1.05e-11, 0, 0, 
  12000. , 0.00364186603946, -1.517476e-7, 9.5e-12, 0, 0, 
  12500. , 0.00356828391639, -1.427342e-7, 8.6e-12, 0, 0, 
  13000. , 0.0034989889079, -1.345791e-7, 7.8e-12, 0, 0, 
  13500. , 0.0034335802559, -1.27172e-7, 7.1e-12, 0, 0, 
  14000. , 0.0033717077622, -1.204203e-7, 6.5e-12, 0, 0, 
  14500. , 0.0033130638727, -1.142456e-7, 5.9e-12, 0, 0, 
  15000. , 0.003257377225, -1.085811e-7, 5.4e-12, 0, 0, 
  15500. , 0.0032044073501, -1.033696e-7, 5.e-12, 0, 0, 
  16000. , 0.0031539402934, -9.85622e-8, 4.6e-12, 0, 0, 
  16500. , 0.00310578497, -9.41161e-8, 4.3e-12, 0, 0, 
  17000. , 0.0030597701153, -8.99946e-8, 4.e-12, 0, 0, 
  17500. , 0.003015741717, -8.61653e-8, 3.7e-12, 0, 0, 
  18000. , 0.0029735608438, -8.26001e-8, 3.4e-12, 0, 0, 
  18500. , 0.0029331017979, -7.92741e-8, 3.2e-12, 0, 0, 
  19000. , 0.0028942505375, -7.61655e-8, 3.e-12, 0, 0, 
  19500. , 0.0028569033227, -7.32549e-8, 2.8e-12, 0, 0, 
  20000. , 0.0028209655492, -7.0525e-8, 2.6e-12, 0, 0, 
  20500. , 0.0027863507393, -6.79606e-8, 2.5e-12, 0, 0, 
  21000. , 0.0027529796658, -6.55479e-8, 2.3e-12, 0, 0, 
  21500. , 0.0027207795889, -6.32747e-8, 2.2e-12, 0, 0, 
  22000. , 0.0026896835884, -6.11299e-8, 2.1e-12, 0, 0, 
  22500. , 0.0026596299787, -5.91035e-8, 2.e-12, 0, 0, 
  23000. , 0.0026305617939, -5.71867e-8, 1.9e-12, 0, 0, 
  23500. , 0.0026024263333, -5.53714e-8, 1.8e-12, 0, 0, 
  24000. , 0.0025751747594, -5.365e-8, 1.7e-12, 0, 0, 
  24500. , 0.0025487617415, -5.20161e-8, 1.6e-12, 0, 0, 
  25000. , 0.002523145138, -5.04634e-8, 1.5e-12, 0, 0, 
  25500. , 0.002498285714, -4.89865e-8, 1.4e-12, 0, 0, 
  26000. , 0.0024741468884, -4.75802e-8, 1.4e-12, 0, 0, 
  26500. , 0.0024506945076, -4.624e-8, 1.3e-12, 0, 0, 
  27000. , 0.0024278966418, -4.49615e-8, 1.2e-12, 0, 0, 
  27500. , 0.0024057234027, -4.37408e-8, 1.2e-12, 0, 0, 
  28000. , 0.0023841467787, -4.25744e-8, 1.1e-12, 0, 0, 
  28500. , 0.0023631404857, -4.1459e-8, 1.1e-12, 0, 0, 
  29000. , 0.0023426798326, -4.03914e-8, 1.e-12, 0, 0, 
  29500. , 0.0023227415996, -3.93688e-8, 1.e-12, 0, 0, 
  30000. , 0.002303303927, -3.83887e-8, 1.e-12, 0, 0, 
  31000. , 0.00226584903, -3.65462e-8, 9.e-13, 0, 0, 
  32000. , 0.0022301638569, -3.48466e-8, 8.e-13, 0, 0, 
  33000. , 0.0021961132942, -3.32747e-8, 8.e-13, 0, 0, 
  34000. , 0.0021635762427, -3.18175e-8, 7.e-13, 0, 0, 
  35000. , 0.0021324438022, -3.04637e-8, 7.e-13, 0, 0, 
  36000. , 0.0021026177359, -2.92032e-8, 6.e-13, 0, 0, 
  37000. , 0.0020740091642, -2.80273e-8, 6.e-13, 0, 0, 
  38000. , 0.0020465374492, -2.69283e-8, 5.e-13, 0, 0, 
  39000. , 0.0020201292376, -2.58993e-8, 5.e-13, 0, 0, 
  40000. , 0.0019947176356, -2.49341e-8, 5.e-13, 0, 0, 
  41000. , 0.0019702414954, -2.40275e-8, 4.e-13, 0, 0, 
  42000. , 0.0019466447964, -2.31745e-8, 4.e-13, 0, 0, 
  43000. , 0.0019238761046, -2.23708e-8, 4.e-13, 0, 0, 
  44000. , 0.0019018881013, -2.16125e-8, 4.e-13, 0, 0, 
  45000. , 0.0018806371692, -2.08961e-8, 3.e-13, 0, 0, 
  46000. , 0.001860083028, -2.02184e-8, 3.e-13, 0, 0, 
  47000. , 0.0018401884135, -1.95766e-8, 3.e-13, 0, 0, 
  48000. , 0.001820918793, -1.8968e-8, 3.e-13, 0, 0, 
  49000. , 0.0018022421133, -1.83903e-8, 3.e-13, 0, 0, 
  50000. , 0.0017841285765, -1.78414e-8, 3.e-13, 0, 0, 
  51000. , 0.0017665504398, -1.73192e-8, 3.e-13, 0, 0, 
  52000. , 0.0017494818369, -1.6822e-8, 2.e-13, 0, 0, 
  53000. , 0.0017328986178, -1.63482e-8, 2.e-13, 0, 0, 
  54000. , 0.0017167782053, -1.58962e-8, 2.e-13, 0, 0, 
  55000. , 0.0017010994655, -1.54646e-8, 2.e-13, 0, 0, 
  56000. , 0.0016858425914, -1.50522e-8, 2.e-13, 0, 0, 
  57000. , 0.0016709889978, -1.46579e-8, 2.e-13, 0, 0, 
  58000. , 0.0016565212256, -1.42804e-8, 2.e-13, 0, 0, 
  59000. , 0.0016424228562, -1.39189e-8, 2.e-13, 0, 0, 
  60000. , 0.0016286784328, -1.35724e-8, 2.e-13, 0, 0, 
  61000. , 0.0016152733889, -1.324e-8, 2.e-13, 0, 0, 
  62000. , 0.0016021939839, -1.2921e-8, 2.e-13, 0, 0, 
  63000. , 0.0015894272437, -1.26146e-8, 2.e-13, 0, 0, 
  64000. , 0.0015769609063, -1.23201e-8, 1.e-13, 0, 0, 
  65000. , 0.0015647833727, -1.20368e-8, 1.e-13, 0, 0, 
  66000. , 0.0015528836615, -1.17643e-8, 1.e-13, 0, 0, 
  67000. , 0.0015412513668, -1.15019e-8, 1.e-13, 0, 0, 
  68000. , 0.0015298766205, -1.12491e-8, 1.e-13, 0, 0, 
  69000. , 0.0015187500569, -1.10055e-8, 1.e-13, 0, 0, 
  70000. , 0.0015078627804, -1.07705e-8, 1.e-13, 0, 0, 
  71000. , 0.0014972063351, -1.05437e-8, 1.e-13, 0, 0, 
  72000. , 0.001486772678, -1.03248e-8, 1.e-13, 0, 0, 
  73000. , 0.0014765541529, -1.01134e-8, 1.e-13, 0, 0, 
  74000. , 0.0014665434669, -9.9091e-9, 1.e-13, 0, 0, 
  75000. , 0.0014567336687, -9.7116e-9, 1.e-13, 0, 0, 
  76000. , 0.0014471181281, -9.5205e-9, 1.e-13, 0, 0, 
  77000. , 0.0014376905174, -9.3357e-9, 1.e-13, 0, 0, 
  78000. , 0.0014284447936, -9.1567e-9, 1.e-13, 0, 0, 
  79000. , 0.0014193751823, -8.9834e-9, 1.e-13, 0, 0, 
  80000. , 0.0014104761628, -8.8155e-9, 1.e-13, 0, 0, 
  81000. , 0.0014017424532, -8.6528e-9, 1.e-13, 0, 0, 
  82000. , 0.0013931689982, -8.495e-9, 1.e-13, 0, 0, 
  83000. , 0.0013847509562, -8.3419e-9, 1.e-13, 0, 0, 
  84000. , 0.0013764836877, -8.1934e-9, 1.e-13, 0, 0, 
  85000. , 0.0013683627452, -8.0492e-9, 1.e-13, 0, 0, 
  86000. , 0.0013603838624, -7.9092e-9, 1.e-13, 0, 0, 
  87000. , 0.0013525429453, -7.7733e-9, 1.e-13, 0, 0, 
  89000. , 0.0013372594403, -7.5127e-9, 1.e-13, 0, 0, 
  91000. , 0.0013224825985, -7.2664e-9, 1.e-13, 0, 0, 
  93000. , 0.0013081850307, -7.0333e-9, 1.e-13, 0, 0, 
  95000. , 0.0012943413766, -6.8123e-9, 0, 0, 0, 
  97000. , 0.0012809281159, -6.6027e-9, 0, 0, 0, 
  99000. , 0.0012679233997, -6.4037e-9, 0, 0, 0, 
 101000. , 0.0012553069012, -6.2144e-9, 0, 0, 0, 
 103000. , 0.0012430596819, -6.0343e-9, 0, 0, 0, 
 105000. , 0.0012311640718, -5.8627e-9, 0, 0, 0, 
 107000. , 0.0012196035626, -5.6991e-9, 0, 0, 0, 
 109000. , 0.0012083627111, -5.543e-9, 0, 0, 0, 
 111000. , 0.0011974270523, -5.3938e-9, 0, 0, 0, 
 113000. , 0.0011867830214, -5.2513e-9, 0, 0, 0, 
 115000. , 0.001176417883, -5.1149e-9, 0, 0, 0, 
 117000. , 0.001166319667, -4.9843e-9, 0, 0, 0, 
 119000. , 0.0011564771102, -4.8592e-9, 0, 0, 0, 
 121000. , 0.0011468796039, -4.7392e-9, 0, 0, 0, 
 123000. , 0.0011375171457, -4.6241e-9, 0, 0, 0, 
 125000. , 0.0011283802955, -4.5135e-9, 0, 0, 0, 
 127000. , 0.0011194601356, -4.4073e-9, 0, 0, 0, 
 129000. , 0.0011107482342, -4.3052e-9, 0, 0, 0, 
 131000. , 0.0011022366117, -4.207e-9, 0, 0, 0, 
 133000. , 0.0010939177103, -4.1125e-9, 0, 0, 0, 
 135000. , 0.0010857843651, -4.0214e-9, 0, 0, 0, 
 137000. , 0.0010778297792, -3.9337e-9, 0, 0, 0, 
 139000. , 0.0010700474989, -3.8491e-9, 0, 0, 0, 
 141000. , 0.0010624313921, -3.7675e-9, 0, 0, 0, 
 143000. , 0.001054975628, -3.6887e-9, 0, 0, 0, 
 145000. , 0.0010476746583, -3.6127e-9, 0, 0, 0, 
 147000. , 0.0010405231997, -3.5392e-9, 0, 0, 0, 
 149000. , 0.0010335162179, -3.4682e-9, 0, 0, 0, 
 151000. , 0.0010266489131, -3.3995e-9, 0, 0, 0, 
 153000. , 0.0010199167054, -3.3331e-9, 0, 0, 0, 
 155000. , 0.0010133152227, -3.2688e-9, 0, 0, 0, 
 157000. , 0.0010068402884, -3.2065e-9, 0, 0, 0, 
 159000. , 0.0010004879103, -3.1462e-9, 0, 0, 0, 
 161000. , 0.0009942542704, -3.0878e-9, 0, 0, 0, 
 163000. , 0.0009881357151, -3.0311e-9, 0, 0, 0, 
 165000. , 0.0009821287462, -2.9762e-9, 0, 0, 0, 
 167000. , 0.0009762300129, -2.9228e-9, 0, 0, 0, 
 169000. , 0.0009704363032, -2.8711e-9, 0, 0, 0, 
 171000. , 0.0009647445372, -2.8209e-9, 0, 0, 0, 
 173000. , 0.00095915176, -2.7721e-9, 0, 0, 0, 
 175000. , 0.0009536551352, -2.7247e-9, 0, 0, 0, 
 177000. , 0.0009482519388, -2.6787e-9, 0, 0, 0, 
 179000. , 0.0009429395538, -2.6339e-9, 0, 0, 0, 
 181000. , 0.0009377154645, -2.5904e-9, 0, 0, 0, 
 183000. , 0.0009325772519, -2.548e-9, 0, 0, 0, 
 185000. , 0.0009275225885, -2.5068e-9, 0, 0, 0, 
 187000. , 0.0009225492346, -2.4667e-9, 0, 0, 0, 
 189000. , 0.0009176550332, -2.4277e-9, 0, 0, 0, 
 191000. , 0.0009128379068, -2.3896e-9, 0, 0, 0, 
 193000. , 0.0009080958534, -2.3526e-9, 0, 0, 0, 
 195000. , 0.0009034269432, -2.3165e-9, 0, 0, 0, 
 197000. , 0.0008988293149, -2.2813e-9, 0, 0, 0, 
 199000. , 0.0008943011731, -2.247e-9, 0, 0, 0, 
  };

  realnum i1e_coef[315][6] =	{
      0. , 0, 0.5, -0.5, 0.3125, -0.14583333333333334, 
      0.2, 0.08228312352881212, 0.3332398104614564, -0.3435081013269764, 0.21520034922674605, -0.10027015939487395, 
      0.4, 0.1367632242958292, 0.2187308855241311, -0.23571344687461526, 0.1482748782861865, -0.06899846878739187, 
      0.6, 0.1721644195415197, 0.1402220843025104, -0.16142692147598037, 0.10220417212494923, -0.04751821144067801, 
      0.8, 0.19449869336433767, 0.08652688185060936, -0.11021576216906243, 0.07046400749935527, -0.032751269559528465, 
      1. , 0.20791041534970847, 0.04993877689422352, -0.0749081653413353, 0.048579558932087784, -0.022590757013374214, 
      1.2, 0.21525685937698313, 0.025144503410663877, -0.050569772844064304, 0.03347921021042905, -0.015593565103572618, 
      1.4, 0.2185075923897756, 0.008478071298728879, -0.03380264700641526, 0.02305279727068931, -0.010770516972104191, 
      1.6, 0.21901948991629336, -0.0025916733926313607, -0.022264775146736984, 0.01584941641510538, -0.0074430638791481325, 
      1.8, 0.2177262787856872, -0.009813372855694585, -0.014340477744351626, 0.010870567290193346, -0.005145366607457414, 
      2. , 0.21526928924893765, -0.014395611319735463, -0.008914147006447881, 0.007428455838706563, -0.0035573477957404655, 
      2.2, 0.21208773268941944, -0.01717391447919739, -0.00521479082793784, 0.00504886074171515, -0.0024588902565673733, 
      2.4, 0.20848108867958262, -0.01872588589582838, -0.002709131291795522, 0.0034044970380550515, -0.0016984812433378134, 
      2.6, 0.20465225432502704, -0.01945066395875003, -0.0010280254367598246, 0.0022692648363452153, -0.0011717313482854461, 
      2.8, 0.20073741130378014, -0.019623810210713327, 0.00008422655275369673, 0.0014868108177208475, -0.0008066474634925488, 
      3. , 0.19682671329730086, -0.01943526356790909, 0.000804839351749409, 0.0009489058375345763, -0.0005535284052285028, 
      3.2, 0.1929786229415282, -0.019015610940969818, 0.0012566746391993733, 0.0005805599905558886, -0.0003780265793462429, 
      3.4, 0.18922985117745195, -0.018454288525599022, 0.0015249173387976434, 0.00032976589244890914, -0.0002563796123266678, 
      3.6, 0.18560224843126769, -0.017812204092940986, 0.0016686118403673456, 0.00016042401858400862, -0.00017213037267752203, 
      3.8, 0.1821075810670727, -0.01713049790376847, 0.0017286511256007497, 0.00004745780731916019, -0.00011386869577773798, 
      4. , 0.1787508395024353, -0.01643662815405747, 0.0017333154699613468, -0.00002656289351963409, -0.00007367503894404985, 
      4.2, 0.1755325260864617, -0.0157486006258426, 0.0017021164748182716, -0.00007375837653406797, -0.00004604679705709144, 
      4.4, 0.17245023371462653, -0.015077909713832716, 0.0016484677745541437, -0.00010255779717639052, -0.000027156808371458716, 
      4.6, 0.16949973122063558, -0.014431583727031433, 0.0015815424404070494, -0.00011882190461719871, -0.000014340708664907845, 
      4.8, 0.16667570578560406, -0.013813606957529728, 0.001507565972522093, -0.00012662267680239175, -5.742098838966078e-6, 
      5. , 0.16397226694454234, -0.013225907724122465, 0.001430717140971316, -0.00012878534778500497, -6.665447543127007e-8, 
      5.2, 0.16138328504783905, -0.01266904392449189, 0.0013537560399954907, -0.00012726568506423938, 3.58847744555673e-6, 
      5.4, 0.15890261497046101, -0.01214267764319657, 0.0012784621665716256, -0.00012341289386139792, 5.8531934363729304e-6, 
      5.6, 0.15652424049609784, -0.011645902605240566, 0.0012059400463645675, -0.00011815302148800513, 7.166867740296574e-6, 
      5.8, 0.1542423640912837, -0.011177468972415006, 0.0011368324147729703, -0.00011211703159530809, 7.835763544805814e-6, 
      6. , 0.1520514593085059, -0.010735936553440295, 0.0010714688142473977, -0.00010573032069364705, 8.07306801036637e-6, 
      6.8, 0.14409759916687212, -0.009208250568020049, 0.0008480648921407041, -0.00008101286730059429, 7.06190166732245e-6, 
      7.6, 0.13723503331206874, -0.007993310966721234, 0.000678542471979432, -0.00006117614850257094, 5.3486948222237e-6, 
      8.4, 0.13124539228973123, -0.0070149410625091335, 0.0005502982194044298, -0.00004645943728042584, 3.9143277668109155e-6, 
      9.2, 0.12596335015698013, -0.006216244584653224, 0.0004523621348161144, -0.0000357163938802009, 2.8583199534618423e-6, 
     10. , 0.12126268138445559, -0.005555612359472509, 0.0003765723151456374, -0.000027842213963041873, 2.1060602017966557e-6, 
     10.8, 0.11704575637451511, -0.0050025423660232105, 0.0003170958857540221, -0.00002200579913055195, 1.5718573303808575e-6, 
     11.6, 0.11323600587428644, -0.004534436344697676, 0.0002697878582643212, -0.000017621073188176656, 1.1895000002837622e-6, 
     12.4, 0.10977256112216012, -0.004134327626347731, 0.00023168215865160963, -0.000014281126159136626, 9.125112255814471e-7, 
     13.2, 0.10660641905509637, -0.0037893008329681344, 0.00020063134985003062, -0.000011702816592819992, 7.091532519100732e-7, 
     14. , 0.10369766746314274, -0.0034893920398149925, 0.00017505992638783274, -9.687279575454344e-6, 5.578295522914306e-7, 
     14.8, 0.1010134505600138, -0.003226816625167425, 0.00015379632824513711, -8.093137631674102e-6, 4.4375141671371826e-7, 
     15.6, 0.09852645721043225, -0.002995420688230596, 0.00013595793236292897, -6.818564867540344e-6, 3.566850276109923e-7, 
     16.4, 0.09621378274884763, -0.0027902862207681983, 0.00012087151168198307, -5.7892601894316e-6, 2.8946173312796586e-7, 
     17.2, 0.09405606134973316, -0.002607443210388957, 0.00010801755691580706, -4.9503184614864415e-6, 2.3699595355034658e-7, 
     18. , 0.09203679687202067, -0.002443657041035524, 0.00009699080633535084, -4.2606765811008945e-6, 1.9563426654654534e-7, 
     18.8, 0.0901418411194437, -0.002296269590555247, 0.00008747191114196612, -3.6892731817457503e-6, 1.6271997849929415e-7, 
     19.6, 0.08835898288975333, -0.0021630791006103656, 0.00007920684506911718, -3.2123604822142243e-6, 1.3629861153564501e-7, 
     20.4, 0.08667762121453874, -0.0020422483795518864, 0.00007199176596674184, -2.811599686602592e-6, 1.1491635164674954e-7, 
     21.2, 0.08508850324653493, -0.0019322339470890654, 0.00006566176181214273, -2.472695603786268e-6, 9.748047145180516e-8, 
     22. , 0.08358351227442369, -0.0018317308257271014, 0.00006008239639145735, -2.1844067431050357e-6, 8.316153747882013e-8, 
     22.8, 0.08215549496313174, -0.001739629142504695, 0.00005514329517939537, -1.9378198940610963e-6, 7.132411532001157e-8, 
     23.6, 0.08079811955254518, -0.0016549797312423453, 0.000050753233621157745, -1.7258130857056991e-6, 6.147713201400465e-8, 
     24.4, 0.07950575868681842, -0.0015769666562983203, 0.0000468363428259011, -1.5426541553117262e-6, 5.3237949045308736e-8, 
     25.2, 0.07827339198802448, -0.0015048851045760536, 0.00004332915418325117, -1.3836979334790056e-6, 4.6306100840061426e-8, 
     26. , 0.07709652456966626, -0.0014381234746397298, 0.00004017827944666902, -1.2451558392446993e-6, 4.044391719876944e-8, 
     26.8, 0.0759711185046299, -0.0013761487722174087, 0.0000373385762355502, -1.1239191328484e-6, 3.546209706873499e-8, 
     27.6, 0.07489353488756852, -0.0013184946290738242, 0.00003477168730287028, -1.0174222791221182e-6, 3.120887756601423e-8, 
     28.4, 0.07386048461307906, -0.0012647514174197183, 0.0000324448697706578, -9.235365414552582e-7, 2.7561836013576674e-8, 
     29.2, 0.07286898636435983, -0.0012145580489472813, 0.00003033005093774216, -8.404865415138907e-7, 2.442163762796795e-8, 
     30. , 0.07191633059864758, -0.0011675951363651904, 0.000028403062326397277, -7.667843958617665e-7, 2.170723238702557e-8, 
     34. , 0.06765630385678817, -0.0009723265414093157, 0.000020942923387654022, -5.007534505212241e-7, 1.2559579729530534e-8, 
     38. , 0.06407295148494815, -0.0008259719782348907, 0.000015960804491760294, -3.42445266483804e-7, 7.708793615598952e-9, 
     42. , 0.061004326474599385, -0.0007129545334953746, 0.000012491581383201888, -2.4304149556265975e-7, 4.96213268695242e-9, 
     46. , 0.0583379910418562, -0.0006235384522151194, 9.992415333830162e-6, -1.7784003230117418e-7, 3.32170899293438e-9, 
     50. , 0.055993123892895395, -0.0005513597232991094, 8.140706381727647e-6, -1.3349791999281435e-7, 2.2977139618046705e-9, 
     54. , 0.053909941810656954, -0.000492101654825175, 6.735840935554599e-6, -1.0240947321236982e-7, 1.6342753647764774e-9, 
     58. , 0.05204314707414405, -0.00044274398058199993, 5.648226665166262e-6, -8.003931784190954e-8, 1.1905653090997503e-9, 
     62. , 0.05035770749282076, -0.0004011169553436167, 4.79139527197886e-6, -6.357717999630474e-8, 8.855570443611072e-10, 
     66. , 0.04882604748612975, -0.00036562596725105145, 4.1060208828373745e-6, -5.1223233297342796e-8, 6.708153728676725e-10, 
     70. , 0.04742612462315791, -0.0003350756351612333, 3.5504000044407624e-6, -4.179117026395507e-8, 5.164087950662588e-10, 
     74. , 0.0461400796591754, -0.0003085541185904109, 3.0945851404381997e-6, -3.4479075335276665e-8, 4.032926954628674e-10, 
     78. , 0.04495326839354153, -0.0002853550544538522, 2.7166609581838475e-6, -2.8732620077458193e-8, 3.1903238753051844e-10, 
     82. , 0.04385355397095955, -0.00026492367657521276, 2.400317698908472e-6, -2.4160909004380948e-8, 2.5531982127471875e-10, 
     86. , 0.042830780690570815, -0.0002468188691454737, 2.1332325884151904e-6, -2.048332137672554e-8, 2.0648819394793183e-10, 
     90. , 0.04187637676046453, -0.00023068595478668852, 1.9059673946747457e-6, -1.7495125246083654e-8, 1.685995731047496e-10, 
     94. , 0.040983050248369485, -0.00021623686013178683, 1.7112032595634208e-6, -1.5044753109982878e-8, 1.388707716529214e-10, 
     98. , 0.04014455344698527, -0.0002032354438612008, 1.543200480361162e-6, -1.3018490403748917e-8, 1.1530419761346389e-10, 
    102. , 0.039355498176656414, -0.00019148649673842583, 1.3974111379363996e-6, -1.1329975214968702e-8, 9.644581574347333e-11, 
    106. , 0.03861120950475368, -0.00018082739272400755, 1.2701973533049693e-6, -9.912880096635268e-9, 8.122341124336407e-11, 
    110. , 0.037907608782725234, -0.00017112168039622178, 1.1586236824435991e-6, -8.715735888217835e-9, 6.883687586958787e-11, 
    114. , 0.03724111930095326, -0.00016225411239630116, 1.0603022934027595e-6, -7.698220273050227e-9, 5.86825297556326e-11, 
    118. , 0.03660858956814998, -0.00015412675304313228, 9.73276218424468e-7, -6.828462806481028e-9, 5.0300214297568924e-11, 
    122. , 0.03600723045215545, -0.00014665590301431164, 8.959304080534414e-7, -6.081065192196922e-9, 4.3335824295373526e-11, 
    126. , 0.03543456331659895, -0.00013976964940400072, 8.269233158844647e-7, -5.435631045963563e-9, 3.7514442073616205e-11, 
    130. , 0.03488837695032632, -0.00013340589888540672, 7.651338044291356e-7, -4.875662934153424e-9, 3.262084959793077e-11, 
    134. , 0.03436669158056963, -0.00012751078730467713, 7.096195970349162e-7, -4.387727221485949e-9, 2.8485280540585523e-11, 
    138. , 0.03386772863298258, -0.00012203738496188721, 6.595845104609421e-7, -3.960815767381498e-9, 2.4972785558213806e-11, 
    140. , 0.03362626090902446, -0.0001194457903333987, 6.364062592259523e-7, -3.767344020945418e-9, 2.3415581756544568e-11, 
    160. , 0.03146509145577107, -0.00009786459177090617, 4.565603884017222e-7, -2.366535054726393e-9, 1.287956807177748e-11, 
    180. , 0.02967334510498538, -0.00008208059507568491, 3.405600960734029e-7, -1.5699737230742694e-9, 7.599202734090427e-12, 
    200. , 0.028156503394832916, -0.00007012596269517085, 2.619754072598346e-7, -1.0874014579401181e-9, 4.739130312139683e-12, 
    220. , 0.026850771055172893, -0.000060815490358769736, 2.0661170996652545e-7, -7.799110428655109e-10, 3.091121759599136e-12, 
    240. , 0.025711323988236817, -0.00005339716557055743, 1.6633993295614435e-7, -5.757388879881753e-10, 2.092362524561131e-12, 
    260. , 0.024705622258356166, -0.0000473732318203475, 1.3625604637634847e-7, -4.3544197501216297e-10, 1.4611283103566075e-12, 
    280. , 0.02380939520725028, -0.000042402484283599745, 1.1327148242978868e-7, -3.3620363697889344e-10, 1.047777099458247e-12, 
    300. , 0.02300412204026895, -0.00003824403198438367, 9.53693152436208e-8, -2.6424503470762195e-10, 7.687581040499714e-13, 
    320. , 0.022275391229106198, -0.000034723467903591265, 8.119108544681064e-8, -2.1093435158270335e-10, 5.75402572811079e-13, 
    340. , 0.02161179800766431, -0.00003171174141889874, 6.979707410995073e-8, -1.7069016122403524e-10, 4.3829469133921384e-13, 
    360. , 0.021004183099425124, -0.00002911153109356676, 6.052190567977511e-8, -1.3980240619673734e-10, 3.3907935053506533e-13, 
    380. , 0.020445094497393762, -0.00002684820497220186, 5.288469303796339e-8, -1.1574428611729748e-10, 2.6598742739608605e-13, 
    400. , 0.019928398958903547, -0.000024863674360813687, 4.653150188077548e-8, -9.675733825265098e-11, 2.1125635962793154e-13, 
    420. , 0.019448996362814202, -0.000023112122692129505, 4.119756602874447e-8, -8.159407264424212e-11, 1.6968349468571287e-13, 
    440. , 0.01900260621881085, -0.00002155697906459572, 3.668190739006261e-8, -6.935398956520178e-11, 1.3768456860741018e-13, 
    460. , 0.018585605734035716, -0.00002016873592616472, 3.2829939877743186e-8, -5.93768502248071e-11, 1.127577848800132e-13, 
    480. , 0.018194905340263978, -0.000018923350427010877, 2.9521325923750896e-8, -5.117166326090716e-11, 9.313621922302229e-14, 
    500. , 0.01782785185289806, -0.00001780105645068919, 2.6661357946031194e-8, -4.436849759120731e-11, 7.752847842878996e-14, 
    520. , 0.01748215229333044, -0.000016785470228947857, 2.417474869766123e-8, -3.868534529175016e-11, 6.500019706506821e-14, 
    540. , 0.017155813358330156, -0.00001586290889551284, 2.200109486781565e-8, -3.3904860256145364e-11, 5.4862906752328744e-14, 
    560. , 0.016847092874732614, -0.000015021865551780034, 2.0091520743989455e-8, -2.985781327041748e-11, 4.6589685234743605e-14, 
    580. , 0.016554460532146736, -0.000014252600784823988, 1.840616583280441e-8, -2.641124645375026e-11, 3.979282181543198e-14, 
    600. , 0.016276565868339667, -0.000013546821814251764, 1.6912283630834968e-8, -2.3459756706700752e-11, 3.416884830631517e-14, 
    620. , 0.016012211975543245, -0.000012897428271495953, 1.558278849358996e-8, -2.091913664861722e-11, 2.948737174574001e-14, 
    640. , 0.01576033375746826, -0.000012298309139818775, 1.4395134481995364e-8, -1.8721582444036766e-11, 2.5565270386773697e-14, 
    660. , 0.015519979834588865, -0.000011744179332475249, 1.333044285893592e-8, -1.6812147594219562e-11, 2.2264850553510573e-14, 
    680. , 0.015290297395644999, -0.000011230447241637048, 1.2372817417317938e-8, -1.5145964343821205e-11, 1.9469668932625694e-14, 
    700. , 0.015070519444716848, -0.000010753106677944754, 1.1508803268028833e-8, -1.3686207549212792e-11, 1.7089845164020012e-14, 
    800. , 0.01409812540653, -8.80305742e-6, 8.24511e-9, -8.58e-12, 1.e-14, 
    900. , 0.01329253322259, -7.37857988e-6, 6.14368e-9, -5.68e-12, 6.e-15, 
   1000. , 0.01261093025693, -6.30073129e-6, 4.722e-9, -3.93e-12, 3.e-15, 
   1100. , 0.01202446052514, -5.46193389e-6, 3.7215e-9, -2.82e-12, 2.e-15, 
   1200. , 0.01151287181376, -4.79402928e-6, 2.99439e-9, -2.08e-12, 0, 
   1300. , 0.01106147556213, -4.25195732e-6, 2.45163e-9, -1.57e-12, 0, 
   1400. , 0.01065932435194, -3.80486068e-6, 2.03722e-9, -1.21e-12, 0, 
   1500. , 0.01029806968913, -3.43097241e-6, 1.71463e-9, -9.5e-13, 0, 
   1600. , 0.00997121900081, -3.1145444e-6, 1.45926e-9, -7.6e-13, 0, 
   1700. , 0.00967363677914, -2.84393132e-6, 1.25412e-9, -6.1e-13, 0, 
   1800. , 0.00940120039392, -2.61035585e-6, 1.08719e-9, -5.e-13, 0, 
   1900. , 0.00915055691824, -2.40709025e-6, 9.4979e-10, -4.2e-13, 0, 
   2000. , 0.00891894770294, -2.22890036e-6, 8.3552e-10, -3.5e-13, 0, 
   2100. , 0.00870407946653, -2.07165938e-6, 7.3961e-10, -2.9e-13, 0, 
   2200. , 0.00850402799321, -1.93207445e-6, 6.5844e-10, -2.5e-13, 0, 
   2300. , 0.00831716511901, -1.80748953e-6, 5.8921e-10, -2.1e-13, 0, 
   2400. , 0.00814210263027, -1.69574108e-6, 5.2975e-10, -1.8e-13, 0, 
   2500. , 0.00797764863153, -1.59505088e-6, 4.7837e-10, -1.6e-13, 0, 
   2600. , 0.0078227732345, -1.50394535e-6, 4.3371e-10, -1.4e-13, 0, 
   2700. , 0.0076765813026, -1.4211941e-6, 3.9467e-10, -1.2e-13, 0, 
   2800. , 0.00753829059819, -1.34576262e-6, 3.6038e-10, -1.1e-13, 0, 
   2900. , 0.00740721411067, -1.27677548e-6, 3.3012e-10, -0.9e-13, 0, 
   3000. , 0.00728274565205, -1.21348739e-6, 3.033e-10, -8.e-14, 0, 
   3100. , 0.00716434802901, -1.15526035e-6, 2.7943e-10, -8.e-14, 0, 
   3200. , 0.00705154326403, -1.1015453e-6, 2.581e-10, -7.e-14, 0, 
   3300. , 0.0069439044589, -1.0518675e-6, 2.39e-10, -6.e-14, 0, 
   3400. , 0.00684104898416, -1.0058146e-6, 2.218e-10, -5.e-14, 0, 
   3500. , 0.00674263274657, -9.630268e-7, 2.063e-10, -5.e-14, 0, 
   3600. , 0.00664834533843, -9.231889e-7, 1.923e-10, -4.e-14, 0, 
   3700. , 0.00655790591297, -8.860238e-7, 1.796e-10, -4.e-14, 0, 
   3800. , 0.00647105966061, -8.512871e-7, 1.68e-10, -4.e-14, 0, 
   3900. , 0.00638757478537, -8.187623e-7, 1.574e-10, -3.e-14, 0, 
   4000. , 0.00630723989966, -7.882571e-7, 1.478e-10, -3.e-14, 0, 
   4500. , 0.00594658476272, -6.606215e-7, 1.101e-10, -2.e-14, 0, 
   5000. , 0.00564147266684, -5.640626e-7, 8.46e-11, 1.e-14, 0, 
   5500. , 0.00537896981775, -4.889306e-7, 6.67e-11, 1.e-14, 0, 
   6000. , 0.00515000078171, -4.291131e-7, 5.36e-11, 0, 0, 
   6500. , 0.00494798449576, -3.805703e-7, 4.39e-11, 0, 0, 
   7000. , 0.00476801681553, -3.405361e-7, 3.65e-11, 0, 0, 
   7500. , 0.00460635832059, -3.070598e-7, 3.07e-11, 0, 0, 
   8000. , 0.00446010120517, -2.787302e-7, 2.61e-11, 0, 0, 
   8500. , 0.00432694566643, -2.545038e-7, 2.25e-11, 0, 0, 
   9000. , 0.00420504564641, -2.335942e-7, 1.95e-11, 0, 0, 
   9500. , 0.0040928998607, -2.153988e-7, 1.7e-11, 0, 0, 
  10000. , 0.00398927319598, -1.994487e-7, 1.5e-11, 0, 0, 
  10500. , 0.00389313895568, -1.853743e-7, 1.32e-11, 0, 0, 
  11000. , 0.00380363571881, -1.728807e-7, 1.18e-11, 0, 0, 
  11500. , 0.00372003463412, -1.617301e-7, 1.05e-11, 0, 0, 
  12000. , 0.00364171429188, -1.517286e-7, 9.5e-12, 0, 0, 
  12500. , 0.00356814118218, -1.427171e-7, 8.6e-12, 0, 0, 
  13000. , 0.0034988543288, -1.345636e-7, 7.8e-12, 0, 0, 
  13500. , 0.0034334530839, -1.271579e-7, 7.1e-12, 0, 0, 
  14000. , 0.0033715873419, -1.204074e-7, 6.5e-12, 0, 0, 
  14500. , 0.0033129496272, -1.142337e-7, 5.9e-12, 0, 0, 
  15000. , 0.0032572686439, -1.085702e-7, 5.4e-12, 0, 0, 
  15500. , 0.0032043039804, -1.033596e-7, 5.e-12, 0, 0, 
  16000. , 0.0031538417312, -9.85529e-8, 4.6e-12, 0, 0, 
  16500. , 0.0031056908539, -9.41076e-8, 4.3e-12, 0, 0, 
  17000. , 0.0030596801207, -8.99866e-8, 4.e-12, 0, 0, 
  17500. , 0.0030156555517, -8.61579e-8, 3.7e-12, 0, 0, 
  18000. , 0.0029734782438, -8.25932e-8, 3.4e-12, 0, 0, 
  18500. , 0.0029330225239, -7.92677e-8, 3.2e-12, 0, 0, 
  19000. , 0.002894174372, -7.61595e-8, 3.e-12, 0, 0, 
  19500. , 0.0028568300678, -7.32492e-8, 2.8e-12, 0, 0, 
  20000. , 0.0028208950241, -7.05197e-8, 2.6e-12, 0, 0, 
  20500. , 0.0027862827787, -6.79556e-8, 2.5e-12, 0, 0, 
  21000. , 0.0027529141179, -6.55432e-8, 2.3e-12, 0, 0, 
  21500. , 0.0027207163142, -6.32703e-8, 2.2e-12, 0, 0, 
  22000. , 0.0026896224585, -6.11257e-8, 2.1e-12, 0, 0, 
  22500. , 0.0026595708751, -5.90996e-8, 2.e-12, 0, 0, 
  23000. , 0.0026305046071, -5.7183e-8, 1.9e-12, 0, 0, 
  23500. , 0.0026023709619, -5.53678e-8, 1.8e-12, 0, 0, 
  24000. , 0.0025751211094, -5.36467e-8, 1.7e-12, 0, 0, 
  24500. , 0.0025487097254, -5.20129e-8, 1.6e-12, 0, 0, 
  25000. , 0.0025230946746, -5.04604e-8, 1.5e-12, 0, 0, 
  25500. , 0.0024982367275, -4.89836e-8, 1.4e-12, 0, 0, 
  26000. , 0.0024740993082, -4.75775e-8, 1.4e-12, 0, 0, 
  26500. , 0.0024506482676, -4.62373e-8, 1.3e-12, 0, 0, 
  27000. , 0.0024278516803, -4.4959e-8, 1.2e-12, 0, 0, 
  27500. , 0.0024056796619, -4.37384e-8, 1.2e-12, 0, 0, 
  28000. , 0.0023841042043, -4.25721e-8, 1.1e-12, 0, 0, 
  28500. , 0.0023630990267, -4.14568e-8, 1.1e-12, 0, 0, 
  29000. , 0.0023426394412, -4.03893e-8, 1.e-12, 0, 0, 
  29500. , 0.0023227022308, -3.93668e-8, 1.e-12, 0, 0, 
  30000. , 0.0023032655383, -3.83868e-8, 1.e-12, 0, 0, 
  31000. , 0.0022658124837, -3.65445e-8, 9.e-13, 0, 0, 
  32000. , 0.0022301290103, -3.48449e-8, 8.e-13, 0, 0, 
  33000. , 0.0021960800195, -3.32732e-8, 8.e-13, 0, 0, 
  34000. , 0.0021635444252, -3.18161e-8, 7.e-13, 0, 0, 
  35000. , 0.0021324133385, -3.04624e-8, 7.e-13, 0, 0, 
  36000. , 0.0021025885326, -2.9202e-8, 6.e-13, 0, 0, 
  37000. , 0.0020739811368, -2.80262e-8, 6.e-13, 0, 0, 
  38000. , 0.0020465105209, -2.69272e-8, 5.e-13, 0, 0, 
  39000. , 0.0020201033384, -2.58983e-8, 5.e-13, 0, 0, 
  40000. , 0.0019946927014, -2.49332e-8, 5.e-13, 0, 0, 
  41000. , 0.001970217468, -2.40266e-8, 4.e-13, 0, 0, 
  42000. , 0.0019466216219, -2.31737e-8, 4.e-13, 0, 0, 
  43000. , 0.0019238537338, -2.237e-8, 4.e-13, 0, 0, 
  44000. , 0.0019018664888, -2.16118e-8, 4.e-13, 0, 0, 
  45000. , 0.0018806162731, -2.08954e-8, 3.e-13, 0, 0, 
  46000. , 0.0018600628096, -2.02177e-8, 3.e-13, 0, 0, 
  47000. , 0.0018401688369, -1.9576e-8, 3.e-13, 0, 0, 
  48000. , 0.001820899825, -1.89674e-8, 3.e-13, 0, 0, 
  49000. , 0.001802223723, -1.83898e-8, 3.e-13, 0, 0, 
  50000. , 0.0017841107351, -1.78408e-8, 3.e-13, 0, 0, 
  51000. , 0.0017665331206, -1.73187e-8, 3.e-13, 0, 0, 
  52000. , 0.0017494650148, -1.68215e-8, 2.e-13, 0, 0, 
  53000. , 0.0017328822696, -1.63477e-8, 2.e-13, 0, 0, 
  54000. , 0.0017167623091, -1.58957e-8, 2.e-13, 0, 0, 
  55000. , 0.0017010840009, -1.54642e-8, 2.e-13, 0, 0, 
  56000. , 0.0016858275392, -1.50518e-8, 2.e-13, 0, 0, 
  57000. , 0.0016709743399, -1.46575e-8, 2.e-13, 0, 0, 
  58000. , 0.0016565069452, -1.428e-8, 2.e-13, 0, 0, 
  59000. , 0.0016424089373, -1.39185e-8, 2.e-13, 0, 0, 
  60000. , 0.0016286648604, -1.3572e-8, 2.e-13, 0, 0, 
  61000. , 0.0016152601489, -1.32397e-8, 2.e-13, 0, 0, 
  62000. , 0.001602181063, -1.29207e-8, 2.e-13, 0, 0, 
  63000. , 0.0015894146291, -1.26143e-8, 2.e-13, 0, 0, 
  64000. , 0.0015769485862, -1.23198e-8, 1.e-13, 0, 0, 
  65000. , 0.0015647713359, -1.20366e-8, 1.e-13, 0, 0, 
  66000. , 0.0015528718971, -1.1764e-8, 1.e-13, 0, 0, 
  67000. , 0.0015412398648, -1.15017e-8, 1.e-13, 0, 0, 
  68000. , 0.0015298653714, -1.12489e-8, 1.e-13, 0, 0, 
  69000. , 0.0015187390515, -1.10052e-8, 1.e-13, 0, 0, 
  70000. , 0.0015078520099, -1.07703e-8, 1.e-13, 0, 0, 
  71000. , 0.0014971957914, -1.05435e-8, 1.e-13, 0, 0, 
  72000. , 0.0014867623532, -1.03246e-8, 1.e-13, 0, 0, 
  73000. , 0.0014765440395, -1.01132e-8, 1.e-13, 0, 0, 
  74000. , 0.0014665335578, -9.9089e-9, 1.e-13, 0, 0, 
  75000. , 0.0014567239571, -9.7114e-9, 1.e-13, 0, 0, 
  76000. , 0.0014471086076, -9.5204e-9, 1.e-13, 0, 0, 
  77000. , 0.0014376811817, -9.3355e-9, 1.e-13, 0, 0, 
  78000. , 0.0014284356369, -9.1566e-9, 1.e-13, 0, 0, 
  79000. , 0.0014193661989, -8.9832e-9, 1.e-13, 0, 0, 
  80000. , 0.0014104673472, -8.8153e-9, 1.e-13, 0, 0, 
  81000. , 0.0014017338004, -8.6526e-9, 1.e-13, 0, 0, 
  82000. , 0.0013931605033, -8.4948e-9, 1.e-13, 0, 0, 
  83000. , 0.0013847426143, -8.3417e-9, 1.e-13, 0, 0, 
  84000. , 0.0013764754943, -8.1932e-9, 1.e-13, 0, 0, 
  85000. , 0.0013683546959, -8.0491e-9, 1.e-13, 0, 0, 
  86000. , 0.0013603759531, -7.9091e-9, 1.e-13, 0, 0, 
  87000. , 0.001352535172, -7.7731e-9, 1.e-13, 0, 0, 
  89000. , 0.0013372519276, -7.5126e-9, 1.e-13, 0, 0, 
  91000. , 0.0013224753321, -7.2663e-9, 1.e-13, 0, 0, 
  93000. , 0.0013081779974, -7.0332e-9, 1.e-13, 0, 0, 
  95000. , 0.0012943345643, -6.8122e-9, 0, 0, 0, 
  97000. , 0.0012809215131, -6.6026e-9, 0, 0, 0, 
  99000. , 0.001267916996, -6.4036e-9, 0, 0, 0, 
 101000. , 0.0012553006868, -6.2143e-9, 0, 0, 0, 
 103000. , 0.0012430536476, -6.0342e-9, 0, 0, 0, 
 105000. , 0.0012311582091, -5.8626e-9, 0, 0, 0, 
 107000. , 0.0012195978635, -5.699e-9, 0, 0, 0, 
 109000. , 0.0012083571681, -5.5429e-9, 0, 0, 0, 
 111000. , 0.0011974216584, -5.3938e-9, 0, 0, 0, 
 113000. , 0.0011867777701, -5.2512e-9, 0, 0, 0, 
 115000. , 0.0011764127682, -5.1148e-9, 0, 0, 0, 
 117000. , 0.0011663146827, -4.9842e-9, 0, 0, 0, 
 119000. , 0.001156472251, -4.8591e-9, 0, 0, 0, 
 121000. , 0.0011468748647, -4.7391e-9, 0, 0, 0, 
 123000. , 0.0011375125216, -4.624e-9, 0, 0, 0, 
 125000. , 0.0011283757819, -4.5135e-9, 0, 0, 0, 
 127000. , 0.0011194557283, -4.4073e-9, 0, 0, 0, 
 129000. , 0.001110743929, -4.3052e-9, 0, 0, 0, 
 131000. , 0.0011022324047, -4.207e-9, 0, 0, 0, 
 133000. , 0.0010939135978, -4.1124e-9, 0, 0, 0, 
 135000. , 0.0010857803437, -4.0214e-9, 0, 0, 0, 
 137000. , 0.0010778258455, -3.9336e-9, 0, 0, 0, 
 139000. , 0.0010700436498, -3.8491e-9, 0, 0, 0, 
 141000. , 0.0010624276246, -3.7675e-9, 0, 0, 0, 
 143000. , 0.0010549719393, -3.6887e-9, 0, 0, 0, 
 145000. , 0.0010476710456, -3.6126e-9, 0, 0, 0, 
 147000. , 0.0010405196605, -3.5392e-9, 0, 0, 0, 
 149000. , 0.0010335127497, -3.4681e-9, 0, 0, 0, 
 151000. , 0.0010266455136, -3.3995e-9, 0, 0, 0, 
 153000. , 0.0010199133723, -3.333e-9, 0, 0, 0, 
 155000. , 0.001013311954, -3.2687e-9, 0, 0, 0, 
 157000. , 0.0010068370819, -3.2065e-9, 0, 0, 0, 
 159000. , 0.0010004847641, -3.1462e-9, 0, 0, 0, 
 161000. , 0.0009942511826, -3.0877e-9, 0, 0, 0, 
 163000. , 0.000988132684, -3.0311e-9, 0, 0, 0, 
 165000. , 0.0009821257701, -2.9761e-9, 0, 0, 0, 
 167000. , 0.00097622709, -2.9228e-9, 0, 0, 0, 
 169000. , 0.0009704334321, -2.8711e-9, 0, 0, 0, 
 171000. , 0.0009647417163, -2.8209e-9, 0, 0, 0, 
 173000. , 0.0009591489879, -2.7721e-9, 0, 0, 0, 
 175000. , 0.0009536524105, -2.7247e-9, 0, 0, 0, 
 177000. , 0.0009482492601, -2.6787e-9, 0, 0, 0, 
 179000. , 0.0009429369199, -2.6339e-9, 0, 0, 0, 
 181000. , 0.0009377128741, -2.5904e-9, 0, 0, 0, 
 183000. , 0.0009325747038, -2.548e-9, 0, 0, 0, 
 185000. , 0.0009275200817, -2.5068e-9, 0, 0, 0, 
 187000. , 0.0009225467679, -2.4667e-9, 0, 0, 0, 
 189000. , 0.0009176526055, -2.4276e-9, 0, 0, 0, 
 191000. , 0.0009128355171, -2.3896e-9, 0, 0, 0, 
 193000. , 0.0009080935009, -2.3526e-9, 0, 0, 0, 
 195000. , 0.0009034246267, -2.3165e-9, 0, 0, 0, 
 197000. , 0.0008988270336, -2.2813e-9, 0, 0, 0, 
 199000. , 0.0008942989261, -2.247e-9, 0, 0, 0, 
  };

  realnum exp_coef[501][2] =	{
-50.0,       5.184705528587072e+21,
-49.8,       4.244877861907688e+21,
-49.6,       3.475412048603725e+21,
-49.4,       2.845426723809632e+21,
-49.2,       2.329638364412868e+21,
-49.0,       1.9073465724951e+21,
-48.8,       1.561603295679627e+21,
-48.6,       1.278532642280836e+21,
-48.4,       1.046773993049375e+21,
-48.2,       8.570260596317587e+20,
-48.0,       7.016735912097631e+20,
-47.8,       5.744817477461038e+20,
-47.6,       4.703458739617215e+20,
-47.4,       3.850866315958034e+20,
-47.2,       3.152822678866946e+20,
-47.0,       2.581312886190068e+20,
-46.8,       2.113400243240302e+20,
-46.6,       1.730305772703317e+20,
-46.4,       1.416654548340572e+20,
-46.2,       1.159858645214222e+20,
-46.0,       9.496119420602448e+19,
-45.8,       7.774765004548325e+19,
-45.6,       6.365439207178171e+19,
-45.4,       5.211580835765118e+19,
-45.2,       4.266881502392741e+19,
-45.0,       3.493427105748509e+19,
-44.8,       2.860176205112524e+19,
-44.6,       2.341714218347495e+19,
-44.4,       1.91723344548107e+19,
-44.2,       1.569697982645006e+19,
-44.0,       1.285160011435931e+19,
-43.8,       1.052200023988652e+19,
-43.6,       8.614685180288971e+18,
-43.4,       7.053107685187749e+18,
-43.2,       5.774596166633851e+18,
-43.0,       4.727839468229346e+18,
-42.8,       3.870827568255235e+18,
-42.6,       3.169165569992612e+18,
-42.4,       2.594693313748874e+18,
-42.2,       2.124355210772014e+18,
-42.0,       1.739274941520501e+18,
-41.8,       1.423997882680749e+18,
-41.6,       1.165870858868658e+18,
-41.4,       9.545343262732137e+17,
-41.2,       7.815066077884502e+17,
-41.0,       6.398434935300549e+17,
-40.8,       5.238595453099109e+17,
-40.6,       4.288999200386713e+17,
-40.4,       3.511535545283491e+17,
-40.2,       2.875002141450046e+17,
-40.0,       2.3538526683702e+17,
-39.8,       1.927171567809359e+17,
-39.6,       1.577834629023027e+17,
-39.4,       1.291821734052547e+17,
-39.2,       1.057654181163343e+17,
-39.0,       8.659340042399374e+16,
-38.8,       7.089667994071994e+16,
-38.6,       5.804529215859412e+16,
-38.4,       4.752346576163744e+16,
-38.2,       3.8908922911901e+16,
-38.0,       3.185593175711376e+16,
-37.8,       2.608143099750266e+16,
-37.6,       2.135366964193671e+16,
-37.4,       1.748290602692123e+16,
-37.2,       1.431379281741287e+16,
-37.0,       1.171914237280261e+16,
-36.8,       9594822260312810.,
-36.6,       7855576054835269.,
-36.4,       6431601699236623.,
-36.2,       5265750102713651.,
-36.0,       4311231547115195.,
-35.8,       3529737851263192.,
-35.6,       2889904929132562.,
-35.4,       2366054038952468.,
-35.2,       1937161205134763.,
-35.0,       1586013452313431.,
-34.8,       1298517988204390.,
-34.6,       1063136610367884.,
-34.4,       870422637631267.9,
-34.2,       712641781603974.2,
-34.0,       583461742527454.9,
-33.8,       477698071851696.8,
-33.6,       391106102111038.4,
-33.4,       320210593514763.7,
-33.2,       262166260371891.1,
-33.0,       214643579785916.1,
-32.8,       175735299721477.7,
-32.6,       143879894283349.9,
-32.4,       117798894199387.1,
-32.2,       96445577359618.14,
-32.0,       78962960182680.69,
-31.8,       64649403855632.91,
-31.6,       52930455104764.95,
-31.4,       43335791368684.55,
-31.2,       35480345102513.43,
-31.0,       29048849665247.43,
-30.8,       23783186562477.12,
-30.6,       19472026244891.05,
-30.4,       15942346711433.88,
-30.2,       13052489528882.56,
-30.0,       10686474581524.46,
-29.8,       8749345381880.24,
-29.6,       7163358133446.177,
-29.4,       5864861599163.679,
-29.2,       4801742553781.419,
-29.0,       3931334297144.042,
-28.8,       3218704289702.042,
-28.6,       2635252187043.086,
-28.4,       2157562007648.186,
-28.2,       1766462367334.243,
-28.0,       1446257064291.475,
-27.8,       1184095135391.711,
-27.6,       969455101915.2316,
-27.4,       793722705666.3497,
-27.2,       649845188545.3043,
-27.0,       532048240601.7986,
-26.8,       435604256701.7262,
-26.6,       356642601133.379,
-26.4,       291994265405.6219,
-26.2,       239064684809.9971,
-26.0,       195729609428.8388,
-25.8,       160249850527.3325,
-25.6,       131201480802.8771,
-25.4,       107418687182.686,
-25.2,       87946982651.72874,
-25.0,       72004899337.38588,
-24.8,       58952625459.80225,
-24.6,       48266327438.62814,
-24.4,       39517126612.13651,
-24.2,       32353886830.6325,
-24.0,       26489122129.84347,
-23.8,       21687458909.7414,
-23.6,       17756189565.52037,
-23.4,       14537538454.77391,
-23.2,       11902329806.97717,
-23.0,       9744803446.248903,
-22.8,       7978370264.144282,
-22.6,       6532137094.69783,
-22.4,       5348061522.750579,
-22.2,       4378622438.028963,
-22.0,       3584912846.131592,
-21.8,       2935078394.232251,
-21.6,       2403038944.052687,
-21.4,       1967441884.339976,
-21.2,       1610805175.602833,
-21.0,       1318815734.483215,
-20.8,       1079754999.464535,
-20.6,       884028623.8513151,
-20.4,       723781420.9482797,
-20.2,       592582107.8368373,
-20.0,       485165195.4097903,
-19.8,       397219665.8050841,
-19.6,       325215956.121981,
-19.4,       266264304.6687256,
-19.2,       217998774.6792111,
-19.0,       178482300.9631872,
-18.8,       146128948.6786814,
-18.6,       119640264.1981907,
-18.4,       97953163.60543343,
-18.2,       80197267.40504706,
-18.0,       65659969.13733051,
-17.8,       53757835.97888369,
-17.6,       44013193.53483411,
-17.4,       36034955.08814172,
-17.2,       29502925.91644544,
-17.0,       24154952.7535753,
-16.8,       19776402.65849779,
-16.6,       16191549.04176531,
-16.4,       13256519.1404636,
-16.2,       10853519.89906441,
-16.0,       8886110.520507872,
-15.8,       7275331.958389593,
-15.6,       5956538.013184625,
-15.4,       4876800.853272269,
-15.2,       3992786.835210952,
-15.0,       3269017.372472111,
-14.8,       2676445.055189099,
-14.6,       2191287.875606813,
-14.4,       1794074.772606215,
-14.2,       1468864.189654096,
-14.0,       1202604.284164777,
-13.8,       984609.1112290357,
-13.6,       806129.7591239914,
-13.4,       660003.2247661569,
-13.2,       540364.9372466925,
-13.0,       442413.3920089205,
-12.8,       362217.4496112482,
-12.6,       296558.5652982033,
-12.4,       242801.6174983236,
-12.2,       198789.1511429547,
-12.0,       162754.7914190039,
-11.8,       133252.352945531,
-11.6,       109097.7992765077,
-11.4,       89321.72336080558,
-11.2,       73130.44183341558,
-11.0,       59874.14171519782,
-10.8,       49020.80113638175,
-10.6,       40134.83743087585,
-10.4,       32859.62567444333,
-10.2,       26903.18607429759,
-10.0,       22026.46579480672,
-9.8,       18033.74492782852,
-9.6,       14764.78156557729,
-9.4,       12088.38073021699,
-9.2,       9897.129058743927,
-9.0,       8103.083927575384,
-8.8,       6634.24400627789,
-8.6,       5431.659591362978,
-8.4,       4447.066747699858,
-8.2,       3640.950307332359,
-8.0,       2980.957987041728,
-7.8,       2440.601977624501,
-7.6,       1998.195895104119,
-7.4,       1635.984429995927,
-7.2,       1339.430764394418,
-7.0,       1096.633158428459,
-6.8,       897.8472916504184,
-6.6,       735.0951892419732,
-6.4,       601.8450378720822,
-6.2,       492.7490410932563,
-6.0,       403.4287934927351,
-5.8,       330.2995599096489,
-5.6,       270.4264074261528,
-5.4,       221.4064162041872,
-5.2,       181.2722418751512,
-5.0,       148.4131591025766,
-4.8,       121.510417518735,
-4.6,       99.48431564193386,
-4.4,       81.45086866496814,
-4.2,       66.68633104092515,
-4.0,       54.59815003314424,
-3.8,       44.70118449330084,
-3.6,       36.59823444367799,
-3.4,       29.96410004739703,
-3.2,       24.53253019710935,
-3.0,       20.08553692318767,
-2.8,       16.44464677109706,
-2.6,       13.46373803500169,
-2.4,       11.0231763806416,
-2.2,       9.025013499434122,
-2.0,       7.38905609893065,
-1.8,       6.049647464412947,
-1.6,       4.953032424395115,
-1.4,       4.055199966844675,
-1.2,       3.320116922736548,
-1.0,       2.718281828459045,
-0.8,       2.225540928492468,
-0.6,       1.822118800390509,
-0.4,       1.49182469764127,
-0.2,       1.22140275816017,
 0. ,       1 ,
 0.2,       0.8187307530779818 ,
 0.4,       0.6703200460356393 ,
 0.6,       0.5488116360940264 ,
 0.8,       0.44932896411722156,
 1. ,       0.36787944117144233, 
 1.2,       0.301194211912202  ,
 1.4,       0.24659696394160643, 
 1.6,       0.20189651799465538, 
 1.8,       0.16529888822158653, 
 2. ,       0.1353352832366127 ,
 2.2,       0.11080315836233387, 
 2.4,       0.09071795328941247, 
 2.6,       0.07427357821433388, 
 2.8,       0.06081006262521795, 
 3. ,       0.049787068367863944, 
 3.2,       0.04076220397836621, 
 3.4,       0.033373269960326066, 
 3.6,       0.02732372244729256, 
 3.8,       0.02237077185616559, 
 4. ,       0.01831563888873418, 
 4.2,       0.014995576820477703, 
 4.4,       0.012277339903068436, 
 4.6,       0.010051835744633576, 
 4.8,       0.008229747049020023, 
 5. ,       0.006737946999085467, 
 5.2,       0.0055165644207607716, 
 5.4,       0.004516580942612666, 
 5.6,       0.003697863716482929, 
 5.8,       0.0030275547453758127, 
 6. ,       0.0024787521766663585, 
 6.2,       0.002029430636295734, 
 6.4,       0.001661557273173934, 
 6.6,       0.0013603680375478928, 
 6.8,       0.0011137751478448024, 
 7. ,       0.0009118819655545162, 
 7.2,       0.0007465858083766792, 
 7.4,       0.0006112527611295723, 
 7.6,       0.0005004514334406104, 
 7.8,       0.00040973497897978643, 
 8. ,       0.00033546262790251185, 
 8.2,       0.00027465356997214205, 
 8.4,       0.0002248673241788482, 
 8.6,       0.0001841057936675792, 
 8.8,       0.0001507330750954765, 
 9. ,       0.00012340980408667956, 
 9.2,       0.00010103940183709324, 
 9.4,       0.00008272406555663223, 
 9.6,       0.00006772873649085378, 
 9.8,       0.000055451599432176945, 
 10.,        0.000045399929762484854, 
 10.2,       0.000037170318684126666, 
 10.4,       0.000030432483008403625, 
 10.6,       0.00002491600973150316, 
 10.8,       0.000020399503411171922, 
 11. ,       0.00001670170079024566, 
 11.2,       0.000013674196065680938, 
 11.4,       0.00001119548484259094, 
 11.6,       9.166087736247602e-6, 
 11.8,       7.504557915076858e-6, 
 12. ,       6.14421235332821e-6, 
 12.2,       5.030455607111439e-6, 
 12.4,       4.118588707535708e-6, 
 12.6,       3.3720152341391786e-6, 
 12.8,       2.7607725720371986e-6, 
 13. ,       2.2603294069810542e-6, 
 13.2,       1.8506011975819048e-6, 
 13.4,       1.515144112143249e-6, 
 13.6,       1.2404950799567113e-6, 
 13.8,       1.0156314710024903e-6, 
 14. ,       8.315287191035679e-7, 
 14.2,       6.807981343976331e-7, 
 14.4,       5.573903692694596e-7, 
 14.6,       4.563526367903986e-7, 
 14.8,       3.73629937988526e-7, 
 15. ,       3.059023205018258e-7, 
 15.2,       2.504516372327617e-7, 
 15.4,       2.0505245756119267e-7, 
 15.6,       1.6788275299956603e-7, 
 15.8,       1.374507727921396e-7, 
 16. ,       1.1253517471925912e-7, 
 16.2,       9.213600834566135e-8, 
 16.4,       7.543458349844232e-8, 
 16.6,       6.176061335580363e-8, 
 16.8,       5.05653134833552e-8, 
 17. ,       4.139937718785167e-8, 
 17.2,       3.389494326196924e-8, 
 17.4,       2.7750832422407467e-8, 
 17.6,       2.2720459927738556e-8, 
 17.8,       1.860193926691551e-8, 
 18. ,       1.522997974471263e-8, 
 18.2,       1.2469252785750989e-8, 
 18.4,       1.0208960723597601e-8, 
 18.6,       8.358390101374608e-9, 
 18.8,       6.843271022217988e-9, 
 19. ,       5.602796437537268e-9, 
 19.2,       4.587181746647508e-9, 
 19.4,       3.755666765938289e-9, 
 19.6,       3.074879879586606e-9, 
 19.8,       2.517498719438278e-9, 
 20. ,       2.061153622438558e-9, 
 20.2,       1.6875298575085259e-9, 
 20.4,       1.381632591079536e-9, 
 20.6,       1.1311850917716326e-9, 
 20.8,       9.261360220567754e-10, 
 21. ,       7.582560427911907e-10, 
 21.2,       6.208075409403602e-10, 
 21.4,       5.082742255105915e-10, 
 21.6,       4.161397394224149e-10, 
 21.8,       3.407064022429891e-10, 
 22. ,       2.7894680928689246e-10, 
 22.2,       2.2838233123615701e-10, 
 22.4,       1.8698363804268407e-10, 
 22.6,       1.5308925478794762e-10, 
 22.8,       1.2533888086068347e-10, 
 23. ,       1.026187963170189e-10, 
 23.2,       8.401716438858868e-11, 
 23.4,       6.878743627134586e-11, 
 23.6,       5.631838950074272e-11, 
 23.8,       4.610959744808222e-11, 
 24. ,       3.775134544279098e-11, 
 24.2,       3.090818748408321e-11, 
 24.4,       2.5305483615118915e-11, 
 24.6,       2.0718377657208856e-11, 
 24.8,       1.6962772941840653e-11, 
 25. ,       1.3887943864964021e-11, 
 25.2,       1.1370486739266698e-11, 
 25.4,       9.309367170903036e-12, 
 25.6,       7.62186519451289e-12, 
 25.8,       6.2402554305624016e-12, 
 26. ,       5.109089028063325e-12, 
 26.2,       4.182968307488728e-12, 
 26.4,       3.42472479249158e-12, 
 26.6,       2.8039275084414685e-12, 
 26.8,       2.2956616805623547e-12, 
 27. ,       1.8795288165390832e-12, 
 27.2,       1.5388280433968073e-12, 
 27.4,       1.2598858428277863e-12, 
 27.6,       1.0315072848906821e-12, 
 27.8,       8.445267361639731e-13, 
 28. ,       6.914400106940203e-13, 
 28.2,       5.661032006637615e-13, 
 28.4,       4.634860997992977e-13, 
 28.6,       3.794703235298559e-13, 
 28.8,       3.1068402375434455e-13, 
 29. ,       2.543665647376923e-13, 
 29.2,       2.0825772910554945e-13, 
 29.4,       1.7050700738489696e-13, 
 29.6,       1.3959933056130979e-13, 
 29.8,       1.1429426503964337e-13, 
 30. ,       9.357622968840175e-14, 
 30.2,       7.661373700298315e-14, 
 30.4,       6.272602259257088e-14, 
 30.6,       5.1355723714802096e-14, 
 30.8,       4.204651035188472e-14, 
 31. ,       3.442477108469977e-14, 
 31.2,       2.8184618754713292e-14, 
 31.4,       2.3075614138262242e-14, 
 31.6,       1.8892714941156384e-14, 
 31.8,       1.5468046731460616e-14, 
 32. ,       1.2664165549094176e-14, 
 32.2,       1.0368541797114079e-14, 
 32.4,       8.489044033871777e-15, 
 32.6,       6.950241414763969e-15, 
 32.8,       5.6903763875834665e-15, 
 33. ,       4.658886145103398e-15, 
 33.2,       3.81437336208507e-15, 
 33.4,       3.122944775260516e-15, 
 33.6,       2.5568509276699837e-15, 
 33.8,       2.0933724855193764e-15, 
 34. ,       1.713908431542013e-15, 
 34.2,       1.403229540863091e-15, 
 34.4,       1.1488671787321141e-15, 
 34.6,       9.406128904299175e-16, 
 34.8,       7.701087001365414e-16, 
 35. ,       6.305116760146989e-16, 
 35.2,       5.162192993279735e-16, 
 35.4,       4.2264461569218167e-16, 
 35.6,       3.4603214449001314e-16, 
 35.8,       2.833071582474967e-16, 
 36. ,       2.319522830243569e-16, 
 36.2,       1.899064673586884e-16, 
 36.4,       1.554822650349588e-16, 
 36.6,       1.2729811194234181e-16, 
 36.8,       1.0422287905595846e-16, 
 37. ,       8.533047625744066e-17, 
 37.2,       6.986268508675704e-17, 
 37.4,       5.719872877313073e-17, 
 37.6,       4.683035828352842e-17, 
 37.8,       3.834145450438482e-17, 
 38. ,       3.1391327920480296e-17, 
 38.2,       2.570104554845264e-17, 
 38.4,       2.1042236376776083e-17, 
 38.6,       1.722792603520286e-17, 
 38.8,       1.4105032856773367e-17, 
 39. ,       1.1548224173015786e-17, 
 39.2,       9.454886273886542e-18, 
 39.4,       7.741006159285781e-18, 
 39.6,       6.33779980237337e-18, 
 39.8,       5.188951605054619e-18, 
 40. ,       4.248354255291589e-18, 
 40.2,       3.478258278776922e-18, 
 40.4,       2.847757019982746e-18, 
 40.6,       2.331546249553593e-18, 
 40.8,       1.908908616733152e-18, 
 41. ,       1.5628821893349888e-18, 
 41.2,       1.2795797118463968e-18, 
 41.4,       1.0476312611033044e-18, 
 41.6,       8.57727931351148e-19, 
 41.8,       7.022482351711429e-19, 
 42. ,       5.74952226429356e-19, 
 42.2,       4.7073106932836765e-19, 
 42.4,       3.85402002888417e-19, 
 42.6,       3.1554047206259754e-19, 
 42.8,       2.5834268831839164e-19, 
 43. ,       2.1151310375910805e-19, 
 43.2,       1.7317228272655536e-19, 
 43.4,       1.4178147344894544e-19, 
 43.6,       1.1608085252936149e-19, 
 43.8,       9.503896380929803e-20, 
 44. ,       7.781132241133797e-20, 
 44.2,       6.37065225958282e-20, 
 44.4,       5.215848922086174e-20, 
 44.6,       4.2703759159206115e-20, 
 44.8,       3.496288089567749e-20, 
 45. ,       2.8625185805493937e-20, 
 45.2,       2.343631993152914e-20, 
 45.4,       1.9188035866917315e-20, 
 45.6,       1.5709835055408607e-20, 
 45.8,       1.286212508564553e-20, 
 46. ,       1.0530617357553812e-20, 
 46.2,       8.621740279526076e-21, 
 46.4,       7.058883911899134e-21, 
 46.6,       5.779325341079253e-21, 
 46.8,       4.731711388784468e-21, 
 47. ,       3.873997628687187e-21, 
 47.2,       3.1717609959573677e-21, 
 47.4,       2.596818268803538e-21, 
 47.6,       2.126094976824191e-21, 
 47.8,       1.7406993414905792e-21, 
 48. ,       1.4251640827409352e-21, 
 48.2,       1.1668256627221738e-21, 
 48.4,       9.553160535512378e-22, 
 48.6,       7.821466319514939e-22, 
 48.8,       6.403675009950519e-22, 
 49. ,       5.242885663363464e-22, 
 49.2,       4.292511727467311e-22, 
 49.4,       3.5144113592253705e-22, 
 49.6,       2.8773566587644137e-22, 
 49.8,       2.3557803841041275e-22, 
 50. ,       1.9287498479639178e-22 
  };
 
  realnum sincos_coef[32][3] =	{
 0.    ,    0, 1, 
 0.2094,    0.20787304380969335, 0.978155814610992, 
 0.4188,    0.40666445300667403, 0.9135775953145866, 
 0.6282,    0.5876893547984601, 0.8090866592995893, 
 0.8376,    0.743039066155518, 0.6692480453215652, 
 1.047 ,    0.8659266112878228, 0.5001710745970701, 
 1.2564,    0.9509832335596342, 0.3092424445131395, 
 1.4658,    0.9944929477200158, 0.1048035158530187, 
 1.6752,    0.9945548852442831, -0.10421410766652844, 
 1.8846,    0.951166340182911 ,-0.30867878661004017, 
 2.094 ,    0.8662228873800595, -0.4996577922728242, 
 2.3034,    0.7434355678969442, -0.6688075630446684, 
 2.5128,    0.5881687596739823, -0.8087382210230757, 
 2.7222,    0.4072058165983376, -0.9133364237390739, 
 2.9316,    0.20845271482417993, -0.9780324461296921, 
 3.141 ,    0.0005926535550994539, -0.9999998243808664, 
 3.3504,    -0.20729329978223912, -0.9782788395265385, 
 3.5598,    -0.4061229465788928, -0.9138184460066878,
 3.7692,    -0.5872097435039474, -0.8094348133939065, 
 3.9786,    -0.7426423034303379, -0.669688292532938, 
 4.188 ,    -0.8656300310490236, -0.5006841812420945, 
 4.3974,    -0.9507997929146544, -0.30980599379845813, 
 4.6068,    -0.9944306608917687, -0.10539288722850386, 
 4.8162,    -0.9946164734428159, 0.1036246628760558, 
 5.0256,    -0.9513491127201708, 0.30811502028713855, 
 5.235 ,    -0.8665188592216699, 0.49914433444964174, 
 5.4444,    -0.7438318085153497, 0.6683668458569622, 
 5.6538,    -0.5886479579621281, 0.808389498686751, 
 5.8632,    -0.40774703716373567, 0.9130949313648582, 
 6.0726,    -0.2090323126220957, 0.9779087341259707, 
 6.282 ,    -0.0011853069020363, 0.9999992975235272 
  };


// saving to one-dimensional array by columns
  for (unsigned i        = 0; i < dimbess1; i++) 
	for (unsigned j  = 0; j < dimbess2; j++)
  	{
	  Bess0arr[i+j*dimbess1] = i0e_coef[i][j];
	  Bess1arr[i+j*dimbess1] = i1e_coef[i][j];
	}	


  for (unsigned i        = 0; i < dimexp1; i++) 
	for (unsigned j  = 0; j < dimexp2; j++)
	  Exparr[i+j*dimexp1]   = exp_coef[i][j];


  for (unsigned i        = 0; i < dimsincos1; i++) 
	for (unsigned j  = 0; j < dimsincos2; j++)
	  SinCosarr[i+j*dimsincos1] = sincos_coef[i][j];
  
}
