// Clipper brigantine
/* Copyright 2007 Kevin Cowtan & University of York all rights reserved */

#include <clipper/clipper-ccp4.h>
#include <clipper/clipper-contrib.h>
#include "brigantine-lib.h"
#include "pirate-lib.h"


int main( int argc, char** argv )
{
  CCP4Program prog( "cbrigantine", "0.1.0", "$Date: 2007/02/26" );

  std::cout << "\nCopyright 2007 Kevin Cowtan and University of York. All rights reserved.\n\n";
  std::cout << " Cowtan K. (2000) Acta Cryst. D56, 1612-1621.\n\n";

  // defaults
  clipper::String title;
  clipper::String ipmtz = "NONE";
  clipper::String ipcol_fo = "/*/*/FP";
  clipper::String ipcol_hl = "/*/*/FC";
  clipper::String ipcol_fr = "NONE";
  clipper::String opmap = "NONE";
  double res_in = 1.0;         // Resolution limit
  int nomit = 30;
  int verbose = 0;
  enum MODE { SIMPLE, SIGMAA, PIRATE };
  MODE mode = SIMPLE;

  // command input
  CCP4CommandInput args( argc, argv, true );
  int arg = 0;
  while ( ++arg < args.size() ) {
    if        ( args[arg] == "-title" ) {
      if ( ++arg < args.size() ) title = args[arg];
    } else if ( args[arg] == "-mtzin" ) {
      if ( ++arg < args.size() ) ipmtz = args[arg];
    } else if ( args[arg] == "-colin-fo" ) {
      if ( ++arg < args.size() ) ipcol_fo = args[arg];
    } else if ( args[arg] == "-colin-hl" ) {
      if ( ++arg < args.size() ) ipcol_hl = args[arg];
    } else if ( args[arg] == "-colin-wrk-free" ) {
      if ( ++arg < args.size() ) ipcol_fr = args[arg];
    } else if ( args[arg] == "-simple" ) {
      mode = SIMPLE;
    } else if ( args[arg] == "-sigmaa" ) {
      mode = SIGMAA;
    } else if ( args[arg] == "-pirate" ) {
      mode = PIRATE;
    } else if ( args[arg] == "-nomit" ) {
      if ( ++arg < args.size() ) nomit = clipper::String(args[arg]).i();
    } else if ( args[arg] == "-resolution" ) {
      if ( ++arg < args.size() ) res_in = clipper::String(args[arg]).f();
    } else if ( args[arg] == "-verbose" ) {
      if ( ++arg < args.size() ) verbose = clipper::String(args[arg]).i();
    } else {
      std::cout << "\nUnrecognized:\t" << args[arg] << std::endl;
      args.clear();
    }
  }
  if ( args.size() <= 1 ) {
    std::cout << "\nUsage: cbrigantine\n\t-mtzin <filename>\t\tCOMPULSORY\n\t-colin-fo <colpath>\n\t-colin-hl <colpath>\n\t-colin-free <colpath>\n\t-resolution <resolution/A>\n.\n";
    exit(1);
  }

  // other initialisations
  clipper::Resolution resol;
  clipper::CCP4MTZfile mtzfile;
 
  // Get resolution for calculation
  mtzfile.open_read( ipmtz );
  double res_wrk = clipper::Util::max( mtzfile.resolution().limit(), res_in );
  mtzfile.close_read();
  resol = clipper::Resolution( clipper::Util::max( res_in, res_wrk ) );

  // Get work reflection data
  clipper::HKL_info hkls;
  mtzfile.open_read( ipmtz );
  hkls.init( mtzfile.spacegroup(), mtzfile.cell(), resol, true );
  clipper::HKL_data<clipper::data32::F_sigF> fobs( hkls );
  clipper::HKL_data<clipper::data32::ABCD>   abcd( hkls );
  clipper::HKL_data<clipper::data32::Flag>   flag( hkls );
  mtzfile.import_hkl_data( fobs, ipcol_fo );
  mtzfile.import_hkl_data( abcd, ipcol_hl );
  if ( ipcol_fr != "NONE" ) mtzfile.import_hkl_data( flag, ipcol_fr );
  mtzfile.close_read();

  // apply free flag
  clipper::HKL_data<clipper::data32::F_sigF> fobs1 = fobs;
  fobs1.mask( flag != 0 );

  // calculate map
  clipper::HKL_data<clipper::data32::Phi_fom> phiw( hkls );
  clipper::HKL_data<clipper::data32::F_phi>   fphi( hkls );
  phiw.compute( abcd, clipper::data32::Compute_phifom_from_abcd() );
  fphi.compute( fobs1, phiw, clipper::data32::Compute_fphi_from_fsigf_phifom() );
  clipper::Grid_sampling grid( hkls.spacegroup(), hkls.cell(), hkls.resolution() );
  clipper::Xmap<float> xmap( hkls.spacegroup(), hkls.cell(), grid );
  xmap.fft_from( fphi );

  // prepare list of omit coordinates
  OmitCoordinates oc( hkls.spacegroup(), hkls.cell(), nomit );

  // do omit calculation
  clipper::Spacegroup spgr = hkls.spacegroup();
  clipper::Cell       cell = hkls.cell();
  typedef clipper::HKL_info::HKL_reference_index HRI;
  typedef clipper::Xmap<float>::Map_reference_index MRI;
  clipper::Xmap<float> xmsk( spgr, cell, grid );
  clipper::Xmap<float> xomit( spgr, cell, grid );
  clipper::Xmap<float> xrslt( spgr, cell, grid );
  clipper::Xmap<float> xwght( spgr, cell, grid );
  xrslt = 0.0; xwght = 0.0;
  const std::vector<clipper::Coord_orth>& omits = oc.coordinates();
  float rad = oc.radius();
  for ( int z = 0; z < omits.size(); z++ ) {
    xmsk = 1.0;
    clipper::Coord_orth x0 = omits[z];
    // make the mask parameters
    float r0 = 1.10 * oc.radius();
    float r1 = 1.25 * r0;
    // make the mask 
    {
      MapFilterFn_smooth flt( r0, r1 );
      clipper::Grid_range gm( cell, grid, r1 );
      clipper::Xmap_base::Map_reference_coord i0, iu, iv, iw ;
      clipper::Coord_grid cent = x0.coord_frac( cell ).coord_grid( grid );
      clipper::Coord_grid g0 = cent + gm.min();
      clipper::Coord_grid g1 = cent + gm.max();
      i0 = clipper::Xmap_base::Map_reference_coord( xmap, g0 );
      for ( iu = i0; iu.coord().u() <= g1.u(); iu.next_u() )
	for ( iv = iu; iv.coord().v() <= g1.v(); iv.next_v() )
	  for ( iw = iv; iw.coord().w() <= g1.w(); iw.next_w() ) {
	    double r2 = ( iw.coord_orth() - x0 ).lengthsq();
	    xmsk[iw] = xmsk[iw] * ( 1.0 - flt( sqrt( r2 ) ) );
	  }
    }
    /*
    clipper::CCP4MAPfile mapout;
    mapout.open_write( "map"+clipper::String(z+10,2)+".map" );
    mapout.export_xmap( xmsk );
    mapout.close_write();
    */
    for ( MRI ix = xmap.first(); !ix.last(); ix.next() ) {
      xomit[ix] = xmap[ix] * xmsk[ix];
    }

    // calculate structure factors from omitted map
    clipper::HKL_data<clipper::data32::F_phi>   fphic( hkls );
    xomit.fft_to( fphic );

    if        ( mode == SIMPLE ) {
      // simple phase combination
      for ( HRI ih = fphic.first(); !ih.last(); ih.next() )
	fphic[ih] = clipper::data32::F_phi( 2.0*fphi[ih].f()-fphic[ih].f(),
					    fphic[ih].phi() );
    } else if ( mode == SIGMAA ) {
      // sigmaa phase compination
      clipper::HKL_data<clipper::data32::Flag> mode( hkls );
      clipper::HKL_data<clipper::data32::F_phi> fb( hkls ), fd( hkls );
      int freeflag = 0;
      for ( HRI ih = mode.first(); !ih.last(); ih.next() )
	if ( !fobs[ih].missing() &&
	     (flag[ih].missing()||flag[ih].flag()==freeflag) )
	  mode[ih].flag() = clipper::SFweight_spline<float>::BOTH;
	else
	  mode[ih].flag() = clipper::SFweight_spline<float>::NONE;
      int n_refln = 1000;
      int n_param = 20;
      clipper::SFweight_spline<float> sfw( n_refln, n_param );
      sfw( fb, fd, phiw, fobs, fphic, mode );
      fphic = fb;
    } else if ( mode == PIRATE ) {
      // make the data lists
      clipper::HKL_data<clipper::data32::F_phi> fb( hkls );
      clipper::HKL_data<clipper::data32::ABCD> abcd1( hkls ), abcd2( hkls );
      clipper::data32::ABCD abcd0( 0.0, 0.0, 0.0, 0.0 );
      abcd1 = abcd0;
      // make the parameters
      std::pair<double,double> one( 1.0, 1.0 );
      std::vector<std::pair<double,double> > w_cyc( 10, one );
      double llkscale = 10.0;
      // make the target function
      Xmap_target_gaussian xtgt;
      xtgt.init( xomit, xmsk );
      // do the calculation
      Refine_HL_coeff refhl( w_cyc, llkscale );
      refhl( fb, abcd2, abcd1, fobs1, fobs, xtgt );
      fphic = fb;
    }

    // calc new map
    xomit.fft_from( fphic );

    // now accumulate 
    r0 = 0.90 * oc.radius();
    r1 = 1.10 * oc.radius();
    {
      MapFilterFn_smooth flt( r0, r1 );
      clipper::Grid_range gm( cell, grid, r1 );
      clipper::Xmap_base::Map_reference_coord i0, iu, iv, iw ;
      clipper::Coord_grid cent = x0.coord_frac( cell ).coord_grid( grid );
      clipper::Coord_grid g0 = cent + gm.min();
      clipper::Coord_grid g1 = cent + gm.max();
      i0 = clipper::Xmap_base::Map_reference_coord( xmap, g0 );
      for ( iu = i0; iu.coord().u() <= g1.u(); iu.next_u() )
	for ( iv = iu; iv.coord().v() <= g1.v(); iv.next_v() )
	  for ( iw = iv; iw.coord().w() <= g1.w(); iw.next_w() ) {
	    double r2 = ( iw.coord_orth() - x0 ).lengthsq();
	    float wgt = flt( sqrt( r2 ) );
	    xrslt[iw] += wgt + xomit[iw];
	    xwght[iw] += wgt;
	  }
    }
  }

  // accumulate the omit map
  for ( MRI ix = xmap.first(); !ix.last(); ix.next() )
    xrslt[ix] /= xwght[ix];

  clipper::CCP4MAPfile mapout;
  mapout.open_write( "omit.map" );
  mapout.export_xmap( xrslt );
  mapout.close_write();
  mapout.open_write( "test.map" );
  mapout.export_xmap( xwght );
  mapout.close_write();
}
